#include "abruijn.h"
#include <cstdio>
#include <cmath>

struct vscp_index { int specIdx, peakIdx, vertexIdx; };
bool operator<(const vscp_index &a, const vscp_index &b) 
	{ return a.specIdx<b.specIdx or (a.specIdx==b.specIdx and a.peakIdx<b.peakIdx); }


Vertex::Vertex(const Vertex &other) {
	reset();
	specPeaks.assign(other.specPeaks.begin(), other.specPeaks.end());    
	outMatchEdges.assign(other.outMatchEdges.begin(), other.outMatchEdges.end());
	outEdges.assign(other.outEdges.begin(), other.outEdges.end());      
	compositeVertex = other.compositeVertex;
}

void Vertex::addPeak(TwoValues<int> specPeak) {
	int lastSpecIdx=-1;
	list<TwoValues<int> >::iterator pivot = specPeaks.begin();
	while (pivot!=specPeaks.end() and (*pivot)<specPeak) { lastSpecIdx=(*pivot)[0]; pivot++; }
	if (pivot==specPeaks.end() or (*pivot)[0]!=specPeak[0] or (*pivot)[1]!=specPeak[1]) {
		specPeaks.insert(pivot, specPeak);
		if (lastSpecIdx==specPeak[0] or (pivot!=specPeaks.end() and (*pivot)[0]==specPeak[0]))
			compositeVertex = true;
	}
}

void Vertex::addMatchEdge(float edgeMass, Match from, Match to) {
	outMatchEdges.push_front(MatchEdge(edgeMass, from, to));
}

void Vertex::addEdge(float edgeMass, float edgeScore, int specIdx, int peakIdx1, int peakIdx2) {
	outEdges.push_front(SimpleEdge(edgeMass, edgeScore, specIdx, peakIdx1, peakIdx2));
}

void Vertex::merge(Vertex &withVertex) {
	specPeaks.merge(withVertex.specPeaks);
	outMatchEdges.splice(outMatchEdges.begin(),withVertex.outMatchEdges);
	outEdges.splice(outEdges.begin(),withVertex.outEdges);
	compositeVertex = compositeVertex or withVertex.compositeVertex;
	withVertex.reset();
}

void Vertex::reset() {
	specPeaks.clear();   outMatchEdges.clear();   outEdges.clear();
	compositeVertex = false;
}

bool Vertex::replaceEdge(int toSpecIdx, int toPeakIdx, int newPeakIdx, float addEdgeMass, float addEdgeScore) {
	bool found = false;
	list<SimpleEdge>::iterator iter = outEdges.begin();
	for(; iter != outEdges.end(); iter++)
		if((*iter).specIdx==toSpecIdx and (*iter).peaks[1]==toPeakIdx) 
		{ iter->peaks[1] = newPeakIdx; iter->edgeMass += addEdgeMass; iter->edgeScore += addEdgeScore; found=true; }
	return found;
}

VertexSet::VertexSet(const VertexSet &other) {
	freeVertices.resize(other.freeVertices.size());   freeVertices.assign(other.freeVertices.begin(),other.freeVertices.end()); 
	firstFreeVertex = other.firstFreeVertex;
	peakToVertex.resize(other.peakToVertex.size());  
	for(unsigned int i=0;i<peakToVertex.size();i++) { peakToVertex[i].resize(other.peakToVertex[i].size()); peakToVertex[i].assign(other.peakToVertex[i].begin(), other.peakToVertex[i].end()); }
	vertices.resize(other.vertices.size()); vertices.assign(other.vertices.begin(),other.vertices.end());
	numUsedVertices = other.numUsedVertices;   specSet = other.specSet;
}

int VertexSet::mergeVertices(Match m) {
	int v1 = peakToVertex[m.specPeak1[0]][m.specPeak1[1]],
		v2 = peakToVertex[m.specPeak2[0]][m.specPeak2[1]],
		maxV=max(v1,v2), minV=min(v1,v2), v;

	if (maxV==minV and maxV>-1) { 
		return maxV; }
	if(maxV==-1) {
		v = getFreeVertex();
		peakToVertex[m.specPeak1[0]][m.specPeak1[1]] = v;	vertices[v].addPeak(m.specPeak1);
		peakToVertex[m.specPeak2[0]][m.specPeak2[1]] = v;	vertices[v].addPeak(m.specPeak2);
		numUsedVertices++;
		return v;
	} else {
		if (minV>-1) {
			list<TwoValues<int> >::iterator pivot1=vertices[minV].specPeaks.begin(), pivot2=vertices[maxV].specPeaks.begin();
			while(pivot1!=vertices[minV].specPeaks.end() and pivot2!=vertices[maxV].specPeaks.end()) {
				if((*pivot1)[0]==(*pivot2)[0]) {
					vertices[minV].compositeVertex=true;
					break;
				}
				if((*pivot1)<(*pivot2)) pivot1++; else pivot2++;
			}
			list<TwoValues<int> >::iterator pivot = vertices[maxV].specPeaks.begin();

			while(pivot!=vertices[maxV].specPeaks.end()) 
				{ peakToVertex[(*pivot)[0]][(*pivot)[1]]=minV; pivot++; }

			vertices[minV].merge(vertices[maxV]);
			vertices[maxV].reset();   addFreeVertex(maxV);
			numUsedVertices--;
			return minV;
		} else {
			if(v1==-1) { peakToVertex[m.specPeak1[0]][m.specPeak1[1]] = maxV; vertices[maxV].addPeak(m.specPeak1); }
			else { peakToVertex[m.specPeak2[0]][m.specPeak2[1]] = maxV; vertices[maxV].addPeak(m.specPeak2); }
			return maxV;
		}
	}
}

inline int VertexSet::addVertex(int specIdx, int peakIdx) {
	int v = peakToVertex[specIdx][peakIdx];
	if(v==-1) {
		v = getFreeVertex();
		peakToVertex[specIdx][peakIdx] = v;	   vertices[v].addPeak(TwoValues<int>(specIdx,peakIdx));
		numUsedVertices++;
	}
	return v;
}

void VertexSet::addGlues(int specIdx1, int specIdx2, vector<TwoValues<int> > &matches,
                         vector<MSGraph> *spectrumGraphs, bool addMissingEdges) {
	float edgeMass1, edgeMass2;
	Match from, to;
	unsigned int i=0;
	int fromV, toV;

	while (i<matches.size() and (matches[i][0]==-1 or matches[i][1]==-1)) {
		i++; 
	}
	if (i<matches.size()) {
		from.set(specIdx1,matches[i][0],specIdx2,matches[i][1]);   fromV = mergeVertices(from);
		i++;
		while(i<matches.size()) {
			while (i<matches.size() and (matches[i][0]==-1 or matches[i][1]==-1)) {
				i++; 
			}
			if (i==matches.size()) break;
			to.set(specIdx1,matches[i][0],specIdx2,matches[i][1]);	toV = mergeVertices(to);
				
			if(spectrumGraphs==0) {
				edgeMass1 = (*specSet)[to.specPeak1[0]][to.specPeak1[1]][0]-(*specSet)[from.specPeak1[0]][from.specPeak1[1]][0];
				edgeMass2 = (*specSet)[to.specPeak2[0]][to.specPeak2[1]][0]-(*specSet)[from.specPeak2[0]][from.specPeak2[1]][0];
				vertices[fromV].addEdge(edgeMass1, 0, from.specPeak1[0], from.specPeak1[1], to.specPeak1[1]);
				vertices[fromV].addEdge(edgeMass2, 0, from.specPeak2[0], from.specPeak2[1], to.specPeak2[1]);
			}
			
			from = to;   fromV = toV;   i++;
		}
	}
}

void VertexSet::addEdges(vector<MSGraph> &spectrumGraphs, vector<bool> *usedSpectra) {
	int fromV, toV;
	for(unsigned int s=0; s<spectrumGraphs.size(); s++) {
		if(usedSpectra and !(*usedSpectra)[s]) continue;
		for(unsigned int e=0; e<spectrumGraphs[s].edges.size(); e++) {
			fromV = addVertex(s,spectrumGraphs[s].edges[e][0]);
			toV = addVertex(s,spectrumGraphs[s].edges[e][1]);
			vertices[fromV].addEdge(spectrumGraphs[s].eMasses[e], spectrumGraphs[s].eScores[e], s, spectrumGraphs[s].edges[e][0], spectrumGraphs[s].edges[e][1]);
		}
	}
}

void VertexSet::addEndpointEdges(vector<Results_ASP> &aligns, vector<vector<TwoValues<int> > > &matches, vector<float> &modPos, 
								 AAJumps &jumps, float peakTol) {
	unsigned int lowPM, highPM, fromV, toV;
	for(unsigned int pIdx=0; pIdx<aligns.size(); pIdx++) {
		if((*specSet)[aligns[pIdx].spec1].parentMass > (*specSet)[aligns[pIdx].spec2].parentMass)
			{ highPM=aligns[pIdx].spec1; lowPM=aligns[pIdx].spec2; }
		else { highPM=aligns[pIdx].spec2; lowPM=aligns[pIdx].spec1; }
		
		float pmDiff = (*specSet)[highPM].parentMass-(*specSet)[lowPM].parentMass;
		bool pmIsValid=false; for(unsigned int j=0;j<jumps.size();j++) if(fabs(jumps[j]-pmDiff)<=peakTol) pmIsValid=true;
		if(!pmIsValid) continue;
		
		if(modPos[pIdx]<=peakTol) {
			for(unsigned int p=0; p<2; p++) {
				fromV = addVertex(highPM,p);	  toV = addVertex(lowPM,p);   if(fromV==toV) continue;
				vertices[fromV].addEdge(pmDiff, (*specSet)[highPM][p][1]+(*specSet)[lowPM][p][1], lowPM, p, p);
			}
		}
		
		if(modPos[pIdx]>=(*specSet)[lowPM].parentMass-peakTol) {
			int sz1=(*specSet)[lowPM].size(), sz2=(*specSet)[highPM].size();
			for(unsigned int p=1; p<3; p++) {
				fromV = addVertex(lowPM,sz1-p);	  toV = addVertex(highPM,sz2-p);  if(fromV==toV) continue; 
				vertices[fromV].addEdge(pmDiff, (*specSet)[lowPM][sz1-p][1]+(*specSet)[highPM][sz2-p][1], highPM, sz1-p, sz2-p);
			}
		}
	}
}

void VertexSet::removeEndpoints(bool typeB, float peakTol) {
	list<TwoValues<int> >::iterator peakIter;
	list<SimpleEdge>::iterator edgeIter;

	vector<int> inDegree(vertices.size());  for(unsigned int vIdx=0; vIdx<inDegree.size(); vIdx++) inDegree[vIdx]=0;
	for(unsigned int vIdx=0; vIdx<vertices.size(); vIdx++)
		for(edgeIter = vertices[vIdx].outEdges.begin(); edgeIter != vertices[vIdx].outEdges.end(); edgeIter++)
			inDegree[peakToVertex[edgeIter->specIdx][edgeIter->peaks[1]]]++;
	
	unsigned int otherV, specIdx, peakIdx;
	for(unsigned int vIdx=0; vIdx<vertices.size(); vIdx++) {
		edgeIter = vertices[vIdx].outEdges.begin();
		while(edgeIter != vertices[vIdx].outEdges.end()) {
			specIdx = edgeIter->specIdx;   peakIdx = edgeIter->peaks[1];
			if(typeB) {
				if((peakIdx<=1 and (*specSet)[specIdx][peakIdx][0]<=peakTol) or 
					(peakIdx>=(*specSet)[edgeIter->specIdx].size()-2 and abs((*specSet)[specIdx][peakIdx][0]+19-(*specSet)[specIdx].parentMass)<=peakTol) )
					{ otherV=peakToVertex[specIdx][peakIdx]; if(--inDegree[otherV]==0) { peakToVertex[specIdx][peakIdx]=-1; releaseVertex(otherV); }
					  edgeIter=vertices[vIdx].outEdges.erase(edgeIter); } else edgeIter++;
			} else {
				if((peakIdx<=1 and abs((*specSet)[specIdx][peakIdx][0]-18)<=peakTol) or 
					(peakIdx>=(*specSet)[specIdx].size()-2 and abs((*specSet)[specIdx][peakIdx][0]+1-(*specSet)[specIdx].parentMass)<=peakTol) )
					{ otherV=peakToVertex[specIdx][peakIdx]; if(--inDegree[otherV]==0) { peakToVertex[specIdx][peakIdx]=-1; releaseVertex(otherV); }
					  edgeIter=vertices[vIdx].outEdges.erase(edgeIter); } else edgeIter++;
			}
		}
	}

	for(unsigned int vIdx=0; vIdx<vertices.size(); vIdx++) {
		peakIter = vertices[vIdx].specPeaks.begin();
		while(peakIter != vertices[vIdx].specPeaks.end()) {
			specIdx = (*peakIter)[0];   peakIdx = (*peakIter)[1];
			if(typeB) { 
				if((peakIdx<=1 and (*specSet)[specIdx][peakIdx][0]<=peakTol) or 
					(peakIdx>=(*specSet)[specIdx].size()-2 and abs((*specSet)[specIdx][peakIdx][0]+19-(*specSet)[specIdx].parentMass)<=peakTol) )
					{ peakToVertex[specIdx][peakIdx]=-1; peakIter=vertices[vIdx].specPeaks.erase(peakIter); } else peakIter++;
			} else {
				if((peakIdx<=1 and abs((*specSet)[specIdx][peakIdx][0]-18)<=peakTol) or 
					(peakIdx>=(*specSet)[specIdx].size()-2 and abs((*specSet)[specIdx][peakIdx][0]+1-(*specSet)[specIdx].parentMass)<=peakTol) )
					{ peakToVertex[specIdx][peakIdx]=-1; peakIter=vertices[vIdx].specPeaks.erase(peakIter); } else peakIter++;
			}
		}
		if(vertices[vIdx].specPeaks.size()==0) releaseVertex(vIdx);
	}

}

inline void VertexSet::scv_addEdges(list<SimpleEdge> &edges, vector<TwoValues<bool> > &vertexAdjChanged) {
	unsigned int fromV, toV;
	list<SimpleEdge>::iterator iter=edges.begin();
	for(;iter!=edges.end();iter++) {
		fromV = addVertex(iter->specIdx,iter->peaks[0]);
		toV = addVertex(iter->specIdx,iter->peaks[1]);
		if(toV>=scv_predEdges.size() or fromV>=scv_predEdges.size()) {
			int prevNumVerts=scv_predEdges.size();
			scv_predEdges.resize(prevNumVerts+2048);   scv_succEdges.resize(prevNumVerts+2048);
			scv_intnEdges.resize(prevNumVerts+2048);   scv_compEdges.resize(prevNumVerts+2048);
			vertexAdjChanged.resize(prevNumVerts+2048); for(unsigned int i=prevNumVerts-1;i<vertexAdjChanged.size();i++) vertexAdjChanged[i].set(false,false);
		}
		
		if(!(vertices[fromV].compositeVertex or vertices[toV].compositeVertex)) continue;

		if(fromV==toV) { scv_intnEdges[fromV].push_back(*iter); continue; }
		else if(vertices[fromV].compositeVertex and vertices[toV].compositeVertex) {
			scv_compEdges[fromV].push_back(*iter);    scv_compEdges[toV].push_back(*iter);
			continue;
		}
		
		if(vertices[fromV].compositeVertex) { scv_succEdges[fromV].push_back(*iter); vertexAdjChanged[fromV][1]=true; }
		else { scv_predEdges[toV].push_back(*iter); vertexAdjChanged[toV][0]=true; }
	}
}

inline void VertexSet::scv_findSplitEdge(int cvIdx, int direction, float peakTol){
	int vIdx = scv_compVerts[cvIdx];
	list<SimpleEdge> &edgesList = (direction==0?scv_predEdges[vIdx]:scv_succEdges[vIdx]);  if(edgesList.size()==0) { scv_splitEdges[cvIdx][direction].clear(); return; }
	vector<TwoValues<float> > edges(edgesList.size());   list<SimpleEdge>::iterator iter=edgesList.begin();
	for(unsigned int i=0;i<edges.size();i++) { edges[i].set(peakToVertex[iter->specIdx][iter->peaks[direction]],iter->edgeMass); iter++; }
	sort(edges.begin(),edges.end());

	int hmVert=-1, hmMult=-1, curVert, curMult;
	unsigned int eIdx=0;
	float hmMass=0.0, curMass,
		  hmMinMass, hmMaxMass, curMinMass, curMaxMass;
	while(eIdx<edges.size()) {
		curVert=(int)edges[eIdx][0];   curMass=edges[eIdx][1];   curMult=1;  eIdx++; curMinMass=curMass; curMaxMass=curMass; 
		while(eIdx<edges.size() and curVert==(int)edges[eIdx][0] and edges[eIdx][1]-curMass<=peakTol)
			{ curMass=(curMult*curMass+edges[eIdx][1])/(curMult+1); curMult++; curMaxMass=edges[eIdx][1]; eIdx++; }
		if(curMult>hmMult or (curMult==hmMult and curMass<hmMass))
			{ hmVert=curVert; hmMult=curMult; hmMass=curMass; hmMinMass=curMinMass; hmMaxMass=curMaxMass; }
	}
	
	scv_splitEdges[cvIdx][direction].clear();
	for(iter=edgesList.begin();iter!=edgesList.end();iter++)
		if(peakToVertex[iter->specIdx][iter->peaks[direction]]==hmVert and iter->edgeMass>=hmMinMass and iter->edgeMass<=hmMaxMass)
			scv_splitEdges[cvIdx][direction].push_back(*iter);
	scv_splitEdges[cvIdx][direction].sort();
}

void VertexSet::scv_useSplitEdge(int cvIdx, int direction, vector<TwoValues<bool> > &vertexAdjChanged) {
	int vertIdx=scv_compVerts[cvIdx];
	list<SimpleEdge> &edges = scv_splitEdges[cvIdx][direction];
	
	list<TwoValues<int> >::iterator peaksIter = vertices[vertIdx].specPeaks.begin();
	list<SimpleEdge>::iterator edgesIter = edges.begin();
	list<TwoValues<int> > newSpecPeaks;
	while(peaksIter != vertices[vertIdx].specPeaks.end() and edgesIter != edges.end())
		if((*peaksIter)[0]==edgesIter->specIdx and (*peaksIter)[1]==edgesIter->peaks[1-direction]) {
			newSpecPeaks.push_back(*peaksIter);    peaksIter = vertices[vertIdx].specPeaks.erase(peaksIter);   edgesIter++;
		} else if((*peaksIter)[0]<edgesIter->specIdx or ((*peaksIter)[0]==edgesIter->specIdx and (*peaksIter)[1]<edgesIter->peaks[1-direction]))
			peaksIter++; else edgesIter++;

	list<int> otherVerts;   list<int>::iterator vIter;
	for(edgesIter=scv_compEdges[vertIdx].begin();edgesIter!=scv_compEdges[vertIdx].end();edgesIter++)
		if(peakToVertex[edgesIter->specIdx][edgesIter->peaks[0]]==vertIdx) otherVerts.push_back(peakToVertex[edgesIter->specIdx][edgesIter->peaks[1]]);
		else otherVerts.push_back(peakToVertex[edgesIter->specIdx][edgesIter->peaks[0]]);
	otherVerts.sort();   vIter=otherVerts.begin();
	while(vIter!=otherVerts.end()) {
		edgesIter=scv_compEdges[*vIter].begin();
		while(edgesIter!=scv_compEdges[*vIter].end())
			if(peakToVertex[edgesIter->specIdx][edgesIter->peaks[0]]==vertIdx or peakToVertex[edgesIter->specIdx][edgesIter->peaks[1]]==vertIdx)
				edgesIter = scv_compEdges[*vIter].erase(edgesIter); else edgesIter++;
		int lastVert = *vIter; vIter++; while(vIter!=otherVerts.end() and lastVert==*vIter) vIter++;
	}

	int newVertIdx = getFreeVertex(); numUsedVertices++; 
	for(peaksIter = newSpecPeaks.begin(); peaksIter!=newSpecPeaks.end(); peaksIter++)
		{ peakToVertex[(*peaksIter)[0]][(*peaksIter)[1]] = newVertIdx;  vertices[newVertIdx].addPeak(*peaksIter); }
	
	int stillComposite=false;
	if(vertices[vertIdx].specPeaks.size()>=2) {
		peaksIter = vertices[vertIdx].specPeaks.begin();
		int prevSpecIdx=(*peaksIter)[0];   peaksIter++;
		while(peaksIter != vertices[vertIdx].specPeaks.end() and not stillComposite)
			{ stillComposite = stillComposite or (prevSpecIdx==(*peaksIter)[0]);  prevSpecIdx=(*peaksIter)[0]; peaksIter++; }
	}
	vertices[vertIdx].compositeVertex = stillComposite;

	list<SimpleEdge> leftoverEdges;
	if(not stillComposite) {
		int lastVert = scv_compVerts.size()-1;
		if(cvIdx<lastVert) {
			scv_compVerts[cvIdx]=scv_compVerts[lastVert];
			scv_splitEdges[cvIdx][0].clear();   scv_splitEdges[cvIdx][0].splice(scv_splitEdges[cvIdx][0].begin(), scv_splitEdges[lastVert][0]);
			scv_splitEdges[cvIdx][1].clear();   scv_splitEdges[cvIdx][1].splice(scv_splitEdges[cvIdx][1].begin(), scv_splitEdges[lastVert][1]);
		}
		scv_compVerts.resize(lastVert);   scv_splitEdges.resize(lastVert);
		scv_predEdges[vertIdx].clear();   scv_succEdges[vertIdx].clear();   scv_intnEdges[vertIdx].clear();

		leftoverEdges.splice(leftoverEdges.begin(),scv_compEdges[vertIdx]);
	} else {
		leftoverEdges.splice(leftoverEdges.begin(),scv_predEdges[vertIdx]);
		leftoverEdges.splice(leftoverEdges.begin(),scv_succEdges[vertIdx]);
		leftoverEdges.splice(leftoverEdges.begin(),scv_intnEdges[vertIdx]);
		leftoverEdges.splice(leftoverEdges.begin(),scv_compEdges[vertIdx]);
		vertexAdjChanged[vertIdx][0]=true;   vertexAdjChanged[vertIdx][1]=true;
	}
	scv_addEdges(leftoverEdges,vertexAdjChanged);
}

void VertexSet::splitComposite(vector<MSGraph> &spectrumGraphs, float peakTol, vector<bool> *usedSpectra) {
	vector<TwoValues<bool> > vertexAdjChanged(vertices.size()); for(unsigned int i=0;i<vertexAdjChanged.size();i++) vertexAdjChanged[i].set(false,false);
	scv_predEdges.resize(vertices.size());   scv_succEdges.resize(vertices.size());
	scv_intnEdges.resize(vertices.size());   scv_compEdges.resize(vertices.size());
	
	list<int> compVerts; for(unsigned int i=0;i<vertices.size();i++) if(vertices[i].compositeVertex) compVerts.push_back(i);
	list<int>::iterator iter=compVerts.begin();   scv_compVerts.resize(compVerts.size());   scv_splitEdges.resize(compVerts.size());
	for(unsigned int i=0;i<scv_compVerts.size();i++) { scv_compVerts[i]=(*iter); iter++; }
	compVerts.clear();

	list<SimpleEdge> allEdges;
	for(unsigned int s=0; s<spectrumGraphs.size(); s++) {
		if(usedSpectra and !(*usedSpectra)[s]) continue;
		for(unsigned int e=0; e<spectrumGraphs[s].edges.size(); e++) 
			allEdges.push_back(SimpleEdge(spectrumGraphs[s].eMasses[e], spectrumGraphs[s].eScores[e], s, spectrumGraphs[s].edges[e][0], spectrumGraphs[s].edges[e][1]));
	}
	scv_addEdges(allEdges,vertexAdjChanged);	

	int vIdx;
	while(scv_compVerts.size()>0) {
		int hmVert=-1, hmDir=-1, hmMult=0;
		for(unsigned int cvIdx=0;cvIdx<scv_compVerts.size();cvIdx++) {
			vIdx = scv_compVerts[cvIdx];
			for(unsigned int dir=0; dir<2; dir++) {
				if(vertexAdjChanged[vIdx][dir]) { scv_findSplitEdge(cvIdx,dir,peakTol); vertexAdjChanged[vIdx][dir]=false; }
				if((int)scv_splitEdges[cvIdx][dir].size()>hmMult) { hmVert=cvIdx; hmDir=dir; hmMult=scv_splitEdges[cvIdx][dir].size(); }
			}
		}
		if(hmVert<0) return;
		scv_useSplitEdge(hmVert,hmDir,vertexAdjChanged);
	}
	
	scv_predEdges.resize(0); scv_succEdges.resize(0); scv_intnEdges.resize(0); scv_compEdges.resize(0); 
	scv_compVerts.resize(0); scv_splitEdges.resize(0);
}

void VertexSet::consolidatePaths() {
	vector<bool> keepVertex(vertices.size()); 
	vector<int> predVertex(vertices.size());
	list<SimpleEdge>::iterator iter;
	list<vscp_index> verticesToRemove;
	list<vscp_index>::reverse_iterator removeIter;
	vscp_index tmp;
	
	for(unsigned int i=0;i<vertices.size();i++) 
		{ keepVertex[i]=false;  predVertex[i]=-1; }
		
	int succV;
	for(unsigned int v=0;v<vertices.size();v++)  {
		if(vertices[v].specPeaks.size()!=1 or vertices[v].outMatchEdges.size()!=0 or vertices[v].outEdges.size()!=1)
			keepVertex[v]=true;
		for(iter = vertices[v].outEdges.begin(); iter != vertices[v].outEdges.end(); iter++) {
			succV = peakToVertex[iter->specIdx][iter->peaks[1]];
			if(predVertex[succV]>=0) keepVertex[succV]=true;
			else predVertex[succV]=v;
		}
	}

	for(unsigned int v=0;v<vertices.size();v++)  {
		if(!keepVertex[v] and predVertex[v]>=0) {
			iter = vertices[v].outEdges.begin();
			tmp.specIdx = iter->specIdx;   tmp.peakIdx = iter->peaks[0];   tmp.vertexIdx = v;
			verticesToRemove.push_front(tmp);
		}
	}
	verticesToRemove.sort();
		
	for(removeIter=verticesToRemove.rbegin(); removeIter!=verticesToRemove.rend(); removeIter++) {
		int v = removeIter->vertexIdx;
		TwoValues<int> thisPeak = *vertices[v].specPeaks.begin();
		SimpleEdge thisEdge = *vertices[v].outEdges.begin();
		vertices[predVertex[v]].replaceEdge(thisPeak[0],thisPeak[1],thisEdge.peaks[1],thisEdge.edgeMass,thisEdge.edgeScore);
		peakToVertex[thisPeak[0]][thisPeak[1]] = -1;
		vertices[v].reset();  addFreeVertex(v);
	}
}

void VertexSet::buildGraph(MSGraph &g, AAJumps &jumps, float peakTol, vector<int> &vSet_index,
						   short edgeScoreType, vector<SpectrumPeakLabels> *labels) {
	vector<int> newVertNums(vertices.size());   int curVert=0, numEdges=0;
	for(int i=0; i<(int)vertices.size(); i++) {
		if(vertices[i].size()==0) newVertNums[i]=-1;
		else { newVertNums[i]=curVert++; numEdges += vertices[i].outEdges.size(); }
	}

	g.vNext.resize(curVert);   g.vScores.resize(curVert);   g.vPeakLabels.resize(curVert);
	g.edges.resize(numEdges);  g.eScores.resize(numEdges);	g.eMasses.resize(numEdges);
	vSet_index.resize(curVert);
	vector<list<SimpleEdge> >  curDest(curVert);
	vector<TwoValues<float> > curEdges(jumps.size());
	vector<float> curEdgeScores(jumps.size());
	list<float> addtnlJumps;
	list<TwoValues<float> > addtnlEdges;
	list<TwoValues<float> > addtnlEdgeScores;

	int gVert, gEdge = 0;
	int vPivot, ePivot;
	curVert = 0;
	list<float>::iterator addtnlIter;
	list<TwoValues<float> >::iterator addtnlIter2, addtnlIter2sc;
	list<SimpleEdge>::iterator edge;
	list<int> gSuccList;
	bool matchedJumps;
	while(curVert<(int)vertices.size()) {
		while (curVert<(int)vertices.size() and newVertNums[curVert]==-1) curVert++;
		if(curVert==(int)vertices.size()) break;
		gVert = newVertNums[curVert];   gSuccList.clear();
		vSet_index[gVert] = curVert;
		
		g.vScores[gVert] = 0;
		list<TwoValues<int> >::iterator iter = vertices[curVert].specPeaks.begin();
		for(; iter!=vertices[curVert].specPeaks.end(); iter++) {
			g.vScores[gVert] += (*specSet)[(*iter)[0]][(*iter)[1]][1];
			if(labels!=0) g.vPeakLabels[gVert].merge((*labels)[(*iter)[0]][(*iter)[1]]);
		}

		for(vPivot=0; vPivot<(int)curDest.size(); vPivot++) curDest[vPivot].clear();
		for(edge = vertices[curVert].outEdges.begin(); edge!=vertices[curVert].outEdges.end(); edge++) {
			vPivot = newVertNums[ peakToVertex[edge->specIdx][edge->peaks[1]] ];
			curDest[vPivot].push_front(*edge);
		}
		
		for(vPivot=0; vPivot<(int)curDest.size(); vPivot++) {
			if(curDest[vPivot].size()==0) continue;
			
			addtnlEdges.clear();   addtnlJumps.clear();   addtnlEdgeScores.clear();
			for(ePivot=0; ePivot<(int)curEdges.size(); ePivot++) { curEdges[ePivot].set(0,0); curEdgeScores[ePivot]=0; }
			for(edge = curDest[vPivot].begin(); edge!=curDest[vPivot].end(); edge++) {
				matchedJumps = false;
				for(ePivot=0; ePivot<(int)curEdges.size(); ePivot++)
					if(abs(edge->edgeMass-jumps[ePivot])<=peakTol) { 
						curEdges[ePivot][0] = (edge->edgeMass + curEdges[ePivot][0]*curEdges[ePivot][1])/(curEdges[ePivot][1]+1);
						curEdges[ePivot][1]++;
						curEdgeScores[ePivot]+=((edgeScoreType>0)?edge->edgeScore:1);
						matchedJumps = true;
					}
				
				addtnlIter = addtnlJumps.begin();   addtnlIter2 = addtnlEdges.begin();   addtnlIter2sc=addtnlEdgeScores.begin();
				TwoValues<float> addtnlEntry(edge->edgeMass,1.0);   float addtnlEntryScore=((edgeScoreType>0)?edge->edgeScore:1);
				for(; addtnlIter!=addtnlJumps.end(); addtnlIter++, addtnlIter2++, addtnlIter2sc++ )
					if(abs(edge->edgeMass-(*addtnlIter))<=peakTol) { 
						(*addtnlIter2)[0] = (edge->edgeMass + (*addtnlIter2)[0]*(*addtnlIter2)[1])/((*addtnlIter2)[1]+1);
						(*addtnlIter2)[1]++;
						(*addtnlIter2sc)[1]+=((edgeScoreType>0)?edge->edgeScore:1);
						addtnlEntry[0] = ((*addtnlIter) + addtnlEntry[0]*addtnlEntry[1])/(addtnlEntry[1]+1);
						addtnlEntry[1]++;
						addtnlEntryScore+=((edgeScoreType>0)?(*addtnlIter2sc)[0]:1);
					}

				if(!matchedJumps) {
					addtnlJumps.push_front(edge->edgeMass);
					addtnlEdges.push_front(addtnlEntry);
					addtnlEdgeScores.push_front(TwoValues<float>(edge->edgeScore,addtnlEntryScore));
				}
			}

			addtnlEdges.sort();		addtnlIter2 = addtnlEdges.begin();   addtnlIter2sc=addtnlEdgeScores.begin();
			list<TwoValues<float> >::iterator nextAddtnl = addtnlIter2, nextAddtnlSc=addtnlIter2sc; nextAddtnl++;  nextAddtnlSc++;
			while(addtnlIter2!=addtnlEdges.end() and nextAddtnl!=addtnlEdges.end()) {
				while(nextAddtnl!=addtnlEdges.end() and abs((*addtnlIter2)[0]-(*nextAddtnl)[0])<=0.0001) {
					(*addtnlIter2)[1] = max((*addtnlIter2)[1],(*nextAddtnl)[1]);
					(*addtnlIter2sc)[1] = max((*addtnlIter2sc)[1],(*nextAddtnlSc)[1]);
					nextAddtnl = addtnlEdges.erase(nextAddtnl);   nextAddtnlSc = addtnlEdgeScores.erase(nextAddtnlSc);
				}
				addtnlIter2 = nextAddtnl;      nextAddtnl++;
				addtnlIter2sc = nextAddtnlSc;  nextAddtnlSc++;
			}
			
			
			int numEdges=addtnlEdges.size(); for(ePivot=0; ePivot<(int)curEdges.size(); ePivot++) if(curEdges[ePivot][1]>0) numEdges++;
			for(ePivot=0; ePivot<(int)curEdges.size(); ePivot++)
				if(curEdges[ePivot][1]>0) {
					if(gEdge==g.edges.size()) { g.edges.resize(2*g.edges.size()); g.eMasses.resize(g.edges.size()); g.eScores.resize(g.edges.size()); }
					g.edges[gEdge].set(gVert,vPivot);
					g.eMasses[gEdge] = curEdges[ePivot][0];
					switch(edgeScoreType) {
						case EST_ABVERTEX_SCORES: g.eScores[gEdge]=g.vScores[vPivot]+curEdges[ePivot][1]; break;
						default: g.eScores[gEdge] = curEdgeScores[ePivot];
					}
					gSuccList.push_front(gEdge++);
				}
			
			addtnlIter2 = addtnlEdges.begin();  addtnlIter2sc=addtnlEdgeScores.begin();
			for(; addtnlIter2 != addtnlEdges.end(); addtnlIter2++, addtnlIter2sc++) {
				if(gEdge==g.edges.size()) { g.edges.resize(2*g.edges.size()); g.eMasses.resize(g.edges.size()); g.eScores.resize(g.edges.size()); }
				g.edges[gEdge].set(gVert,vPivot);
				g.eMasses[gEdge] = (*addtnlIter2)[0];
				switch(edgeScoreType) {
					case EST_ABVERTEX_SCORES: g.eScores[gEdge]=g.vScores[vPivot]+(*addtnlIter2)[1]; break;
					default: g.eScores[gEdge] = (*addtnlIter2sc)[1]; // (*addtnlIter2)[1];
				}
				gSuccList.push_front(gEdge++);
			}
		}
		g.vNext[gVert].resize(gSuccList.size()); 
		g.vNext[gVert].assign(gSuccList.begin(),gSuccList.end());
		curVert++;
	} 
	g.edges.resize(gEdge);   g.eMasses.resize(gEdge);   g.eScores.resize(gEdge);
}

void VertexSet::enumerateHeaviestPath(MSGraph &g, vector<int> &vSet_index, SpecSet &enumSpecs, vector<int> &enumSpecsIdx) {
	
	vector<bool> matchedVerts(g.vNext.size()); 
	for(unsigned int i=0; i<matchedVerts.size();i++) matchedVerts[i]=false;
	for(unsigned int i=0; i<g.edges.size();i++) 
		if(g.edgeInPath[i]) { matchedVerts[g.edges[i][0]]=true;  matchedVerts[g.edges[i][1]]=true; }
	
	list<int> vSet_verts;
	for(unsigned int i=0;i<matchedVerts.size();i++) 
		if(matchedVerts[i]) vSet_verts.push_front(vSet_index[i]);

	list<int> matchedSpecsIdx;          
	vector<bool> matchedSpecs(specSet->size());     for(unsigned int i=0;i<matchedSpecs.size();i++) matchedSpecs[i]=false;
	vector<vector<bool> > matchedPeaks(specSet->size());  
	list<TwoValues<int> >::iterator iter;  list<int>::iterator vertexIter;
	for(vertexIter = vSet_verts.begin(); vertexIter != vSet_verts.end(); vertexIter++)
		for(iter = vertices[*vertexIter].specPeaks.begin(); iter!=vertices[*vertexIter].specPeaks.end(); iter++) {
			if(!matchedSpecs[(*iter)[0]]) {
				matchedSpecs[(*iter)[0]]=true; 
				matchedSpecsIdx.push_front((*iter)[0]);
				matchedPeaks[(*iter)[0]].resize((*specSet)[(*iter)[0]].size());
			}
			matchedPeaks[(*iter)[0]][(*iter)[1]]=true;
		}

	enumSpecs.resize(matchedSpecsIdx.size());   enumSpecsIdx.resize(matchedSpecsIdx.size());
	list<int>::iterator specIter = matchedSpecsIdx.begin();
	int esIdx = 0;
	for(;specIter != matchedSpecsIdx.end(); specIter++)  {
		int numMatchedPeaks = 0;
		for(unsigned int i=0;i<matchedPeaks[*specIter].size();i++) if(matchedPeaks[*specIter][i]) numMatchedPeaks++;
		enumSpecs[esIdx].copyNP((*specSet)[*specIter]);   
		enumSpecs[esIdx].peakList.resize(numMatchedPeaks);   numMatchedPeaks=0;
		for(unsigned int i=0;i<matchedPeaks[*specIter].size();i++) 
			if(matchedPeaks[*specIter][i]) 
				{ enumSpecs[esIdx][numMatchedPeaks]=(*specSet)[*specIter][i]; numMatchedPeaks++; }
		enumSpecsIdx[esIdx++] = *specIter;
	}
}

void VertexSet::getMatchedPeaks(vector<int> &matchedVertices, vector<list<TwoValues<int> > > &putHere) {
	putHere.resize(matchedVertices.size());
	for(unsigned int vIdx=0; vIdx<matchedVertices.size(); vIdx++)
		{ putHere[vIdx].clear();    putHere[vIdx].assign(vertices[matchedVertices[vIdx]].specPeaks.begin(), vertices[matchedVertices[vIdx]].specPeaks.end()); }
}

void VertexSet::outputGraph(vector<SpectrumPeakLabels> *labels) {
	PeakLabel vLabel;   string vLabelText("<empty>");
	
	for(unsigned int v=0; v<vertices.size(); v++) {
		if(vertices[v].size()==0) continue;
		if(labels) {
			vLabel.set(0,0,0,0,0); 
			list<TwoValues<int> >::iterator iterP = vertices[v].specPeaks.begin();
			for(; iterP!=vertices[v].specPeaks.end(); iterP++) vLabel.merge((*labels)[(*iterP)[0]][(*iterP)[1]]);
			if(vLabel.isEmpty()) vLabelText.assign("<empty>"); else vLabelText = vLabel.asString();
		}
		
		list<TwoValues<int> >::iterator iterP = vertices[v].specPeaks.begin();
		list<MatchEdge>::iterator iterE = vertices[v].outMatchEdges.begin();
		list<SimpleEdge>::iterator iterE2 = vertices[v].outEdges.begin();
	}
}

void VertexSet::output_graphviz_ma(char *filename, vector<int> &matchedVertices) {
	ofstream out(filename);
	if(!out) { cerr<<"Error opening "<<filename<<"!\n"; return; }
	
	list<int> specIndices;
	for(unsigned int vIdx=0; vIdx<matchedVertices.size(); vIdx++)
		for(list<TwoValues<int> >::iterator iter=vertices[matchedVertices[vIdx]].specPeaks.begin(); iter!=vertices[matchedVertices[vIdx]].specPeaks.end(); iter++)
			specIndices.push_back((*iter)[0]+1);
	specIndices.sort();   specIndices.unique();

	out << "digraph G {\n";
	out << "  ranksep=.75; size = \"7.5,7.5\";\n";
	out << "  { node [shape=plaintext, fontsize=16];\n";
	list<int>::iterator indicesIter=specIndices.begin();  out << "  \"" << *indicesIter;   indicesIter++;
	for(; indicesIter!=specIndices.end(); indicesIter++) out << "\" -> \"" << *indicesIter; out<<"\";\n";
	out << "  }\n";

	list<TwoValues<int> >::iterator iterPrev;   bool firstVertex;
	vector<int> lastUsedPeak(specSet->size());  for(unsigned int specIdx=0; specIdx<lastUsedPeak.size(); specIdx++) lastUsedPeak[specIdx]=-1; 
	for(unsigned int vIdx=0; vIdx<matchedVertices.size(); vIdx++) {
		firstVertex=true;
		for(list<TwoValues<int> >::iterator iter=vertices[matchedVertices[vIdx]].specPeaks.begin(); iter!=vertices[matchedVertices[vIdx]].specPeaks.end(); iter++) {
			out <<"  \""<<(*iter)[0]+1<<"-"<<(*iter)[1]<<"\" [shape=ellipse, label=\""<<(*specSet)[(*iter)[0]][(*iter)[1]][0]<<"\"];\n";
			out <<"  { rank=same; \""<<(*iter)[0]+1<<"\"; \""<<(*iter)[0]+1<<"-"<<(*iter)[1]<<"\" }\n";

			if(lastUsedPeak[(*iter)[0]]>=0)
				out<<"  \""<<(*iter)[0]+1<<"-"<<lastUsedPeak[(*iter)[0]]<<"-"<<(*iter)[1]<<"\" [shape=diamond, label=\""<<(*specSet)[(*iter)[0]][(*iter)[1]][0]-(*specSet)[(*iter)[0]][lastUsedPeak[(*iter)[0]]][0]<<"\"];\n"
			       <<"  { rank=same; \""<<(*iter)[0]+1<<"\"; \""<<(*iter)[0]+1<<"-"<<lastUsedPeak[(*iter)[0]]<<"-"<<(*iter)[1]<<"\" }\n"
				   <<"  \""<<(*iter)[0]+1<<"-"<<lastUsedPeak[(*iter)[0]]<<"\" -> \""<<(*iter)[0]+1<<"-"<<lastUsedPeak[(*iter)[0]]<<"-"<<(*iter)[1]<<"\";\n"
				   <<"  \""<<(*iter)[0]+1<<"-"<<lastUsedPeak[(*iter)[0]]<<"-"<<(*iter)[1]<<"\" -> \""<<(*iter)[0]+1<<"-"<<(*iter)[1]<<"\";\n";
			lastUsedPeak[(*iter)[0]] = (*iter)[1];
			
			if(firstVertex) firstVertex=false; else out << "  \""<<(*iterPrev)[0]+1<<"-"<<(*iterPrev)[1]<< "\" -> \"" << (*iter)[0]+1<<"-"<<(*iter)[1]<<"\" [arrowhead=none, minlen=0, constraint=false];\n\n";
			iterPrev=iter;
		}
	}
	out << "}\n";
	out.close();
}

void Save_abinfo(char *filename, SpecSet &specSet, vector<list<int> > &cSpectra, 
                   vector<bool> &specFlipped, vector<vector<list<TwoValues<int> > > > &abVertices) {
	FILE *fp = fopen(filename,"w");   if(fp==0) { cerr<<"Error opening "<<filename<<"!!\n"; return; }
	unsigned int cIdx, vIdx;
	list<TwoValues<int> >::iterator ltiIter;

	unsigned int specCount = 0; for(cIdx=0; cIdx<cSpectra.size(); cIdx++) specCount+=cSpectra[cIdx].size();
	unsigned int *specIndices = (unsigned int *) malloc(specCount*sizeof(unsigned int));
	unsigned short *specInfo = (unsigned short *) malloc(2*specCount*sizeof(unsigned short));
	unsigned int indIdx=0, infoIdx = 0;
	for(cIdx=0; cIdx<cSpectra.size(); cIdx++) {
		for(list<int>::iterator liIter=cSpectra[cIdx].begin(); liIter!=cSpectra[cIdx].end(); liIter++) {
			specIndices[indIdx++] = (unsigned int)*liIter;
			specInfo[infoIdx++] = (unsigned short)cIdx;
			specInfo[infoIdx++] = (unsigned short)specFlipped[*liIter];
		}
	}
	fwrite(&specCount, sizeof(unsigned int), 1, fp);
	fwrite(specIndices, sizeof(unsigned int), specCount, fp);   free(specIndices);
	fwrite(specInfo, sizeof(unsigned short), 2*specCount, fp);  free(specInfo);

	unsigned int numComponents=cSpectra.size(), numVertices=0, numPeaks=0;
	for(cIdx=0; cIdx<numComponents; cIdx++) { numVertices+=abVertices[cIdx].size(); for(vIdx=0; vIdx<abVertices[cIdx].size(); vIdx++) numPeaks+=abVertices[cIdx][vIdx].size(); }
	unsigned short *vertsPerComp = (unsigned short *) malloc(numComponents*sizeof(unsigned short));
	unsigned short *peaksPerVert = (unsigned short *) malloc(numVertices*sizeof(unsigned short));
	unsigned int   *vertsSpecIdx = (unsigned int *) malloc(numPeaks*sizeof(unsigned int));
	float *vertsPeakMass = (float *) malloc(numPeaks*sizeof(float));
	unsigned int gVidx=0, gPidx=0;
	for(cIdx=0; cIdx<numComponents; cIdx++) { 
		vertsPerComp[cIdx] = abVertices[cIdx].size();
		for(vIdx=0; vIdx<abVertices[cIdx].size(); vIdx++) {
			peaksPerVert[gVidx++] = abVertices[cIdx][vIdx].size(); 
			for(ltiIter=abVertices[cIdx][vIdx].begin(); ltiIter!=abVertices[cIdx][vIdx].end(); ltiIter++) 
				{ vertsSpecIdx[gPidx]=(unsigned int)(*ltiIter)[0];    vertsPeakMass[gPidx]=specSet[(*ltiIter)[0]][(*ltiIter)[1]][0];    gPidx++; }
		}
	}
	fwrite(&numComponents, sizeof(unsigned int), 1, fp);
	fwrite(vertsPerComp, sizeof(unsigned short), numComponents, fp);  free(vertsPerComp);
	fwrite(peaksPerVert, sizeof(unsigned short), numVertices, fp);    free(peaksPerVert);
	fwrite(vertsSpecIdx, sizeof(unsigned int), numPeaks, fp);         free(vertsSpecIdx);
	fwrite(vertsPeakMass, sizeof(float), numPeaks, fp);               free(vertsPeakMass);

	fclose(fp);
}
