#include "alignment_scoring.h"
#include "inputParams.h"

#include <iostream>
#include <fstream>
#include <list>
#include <algorithm>
#include <cmath>


int FindMatchPeaksAll(Spectrum &spec1, Spectrum &spec2, float shift, float tolerance, vector<int> &idx1, vector<int> &idx2){
    int i,j;            // Iterators over the peaks indices
    int low=0,high=0;   // Index bounds of the peaks in spec2 the lie within tolerance of current peak in spec1

    idx1.resize(0);   idx2.resize(0);
    for(i=0; i<(int)spec1.size(); i++) {
    	while ((low<(int)spec2.size()) && (spec1[i][0]-tolerance-0.000001) > (spec2[low][0]+shift)) low++;
    	while ((high<(int)spec2.size()) && (spec1[i][0]+tolerance+0.000001) >= (spec2[high][0]+shift)) high++;  // high is index of first unreachable peak
    	for (j=low; j<high; j++) { idx1.push_back(i); idx2.push_back(j); }
    }
    return idx1.size();
}

int FindMatchPeaksAll2(Spectrum &spec1, Spectrum &spec2, float shift, float tolerance, vector<int> &idx1, vector<int> &idx2){
    int i,j;            // Iterators over the peaks indices
    int low=0,high=0;   // Index bounds of the peaks in spec2 the lie within tolerance of current peak in spec1

    idx1.resize(0);   idx2.resize(0);
    for(i=0; i<(int)spec1.size(); i++) {
        while (low>0 && (spec1[i][0]-tolerance-0.000001) < (spec2[low][0]+shift)) low--;
        while ((low<(int)spec2.size()) && (spec1[i][0]-tolerance-0.000001) > (spec2[low][0]+shift)) low++;
        while ((high<(int)spec2.size()) && (spec1[i][0]+tolerance+0.000001) >= (spec2[high][0]+shift)) high++;  // high is index of first unreachable peak
        for (j=low; j<high; j++) { idx1.push_back(i); idx2.push_back(j); }
    }
    return idx1.size();
}

void FindMatchPeaks(Spectrum &spec1, Spectrum &spec2, float shift, float tolerance, vector<int> &idx1, vector<int> &idx2){
    unsigned int i,j;            // Iterators over the peaks indices
    unsigned int low=0,high=0;   // Index bounds of the peaks in spec2 the lie within tolerance of current peak in spec1
	vector<bool> match1(spec1.size()), match2(spec2.size());
	for(i=0;i<spec1.size();i++) match1[i]=false;   for(i=0;i<spec2.size();i++) match2[i]=false;

    for(i=0; i<spec1.size(); i++) {
        while ((low<spec2.size()) && (spec1[i][0]-tolerance-0.000001) > (spec2[low][0]+shift)) low++;
        while ((high<spec2.size()) && (spec1[i][0]+tolerance+0.000001) >= (spec2[high][0]+shift)) high++;  // high is index of first unreachable peak
        for (j=low; j<high; j++) { match1[i]=true; match2[j]=true; }
    }

    idx1.resize(0);   idx2.resize(0);
	for(i=0;i<spec1.size();i++) if(match1[i]) idx1.push_back(i);
	for(i=0;i<spec2.size();i++) if(match2[i]) idx2.push_back(i);
}

static unsigned int csMaxVecSize=0; 
static TwoValues<float> *shiftScoresV=0;
static TwoValues<int> *shiftMPcount=0; 
static unsigned int *shiftPairsV=0;
static char* validShift=0; 

void computeShifts_cleanup() { 
	if(csMaxVecSize>0) 
		{ delete shiftScoresV; delete shiftMPcount; delete shiftPairsV; delete validShift; csMaxVecSize=0; }
}

TwoValues<int> computeShifts(Spectrum &spec1, Spectrum &spec2, float peakTol, float pmTol,
                     float minRatio, float minPeakAreaOvlp, int minNumMatchedPeaks, 
                     AAJumps &validJumps, list<float> &shiftScores, list<TwoValues<unsigned int> > &shiftPairs, 
                     vector<list<TwoValues<int> > > &shiftMatchedPeaks, 
                     TwoValues<float> &bestCandidateScores, TwoValues<int> &bestCandidateMP,
                     float minAbsShift) {
	unsigned int szSpec1=spec1.size(), szSpec2=spec2.size(), 
	             szVecs=0;  // Size of the vectors used in computeShifts. Determined by minimum/maximum shifts to consider
	vector<list<TwoValues<int> > >::iterator shiftMPiter;
	
	if(csMaxVecSize==0) {
		szVecs = 1+(unsigned int)ceil(2*(InputParams::MaxShift+InputParams::PeakTol+InputParams::PMTol)/InputParams::Resolution);
	    shiftScoresV = new TwoValues<float>[szVecs];   shiftPairsV = new unsigned int[szVecs];
		shiftMPcount = new TwoValues<int>[szVecs];     validShift = new char[szVecs];
		csMaxVecSize = szVecs;
	}
	if(minPeakAreaOvlp>0) minPeakAreaOvlp = max(minPeakAreaOvlp*spec1.parentMass,minPeakAreaOvlp*spec2.parentMass);
	if(spec2.parentMass<minPeakAreaOvlp or spec1.parentMass<minPeakAreaOvlp) return(TwoValues<int>(0,0)); // Test if required overlap areas are feasible

	int shiftsOffset = (int)ceil((min(spec2.parentMass-minPeakAreaOvlp,InputParams::MaxShift)+InputParams::PeakTol)/InputParams::Resolution);
	szVecs = 1+shiftsOffset+(int)ceil((min(spec1.parentMass-minPeakAreaOvlp,InputParams::MaxShift)+InputParams::PeakTol)/InputParams::Resolution);
	if(szVecs>csMaxVecSize) { computeShifts_cleanup();
	    shiftScoresV = new TwoValues<float>[szVecs];   shiftPairsV = new unsigned int[szVecs];
		shiftMPcount = new TwoValues<int>[szVecs];     validShift = new char[szVecs];
		csMaxVecSize = szVecs;
	}
	shiftMatchedPeaks.resize(szVecs);
	
	for(unsigned int i=0; i<szVecs; i++) { shiftScoresV[i].set(0,0);   shiftMPcount[i].set(0,0); }
	float totalScore1=0; for(unsigned int i=0;i<szSpec1;i++) totalScore1+=spec1[i][1];
	float totalScore2=0; for(unsigned int i=0;i<szSpec2;i++) totalScore2+=spec2[i][1];
	float minScore1 = minRatio*totalScore1, minScore2 = minRatio*totalScore2;
	shiftScores.clear();   shiftPairs.clear(); for(unsigned int i=0; i<szVecs; i++) shiftMatchedPeaks[i].clear(); //.resize(0);
	int shiftIndex, intPeakTol=(int)round(peakTol/InputParams::Resolution), intPMTol=(int)round(pmTol/InputParams::Resolution);
	
	TwoValues<int> peakPair;
	for(unsigned int idxSpec1=0; idxSpec1<szSpec1; idxSpec1++)
		for(unsigned int idxSpec2=0; idxSpec2<szSpec2; idxSpec2++) {
			shiftIndex = shiftsOffset + (int)round((spec1[idxSpec1][0]-spec2[idxSpec2][0])/InputParams::Resolution);
			peakPair.set((int)idxSpec1,(int)idxSpec2);
						
			for(int tolIdx=-intPeakTol; tolIdx<=intPeakTol; tolIdx++) {
				int shiftIndexTol = shiftIndex+tolIdx; // float curPenalty=abs(tolIdx/10.0);
				if(shiftIndexTol>=0 and shiftIndexTol<szVecs) {
					shiftMatchedPeaks[shiftIndexTol].push_back(peakPair);
				}
			}
		}

	int lastPeak;                      // Index of last peak in spectrum 1 (per shift)
	char *match2 = new char[szSpec2];   // Matched peaks in spectrum 2 (per shift)
	shiftMPiter=shiftMatchedPeaks.begin();
	for(unsigned int shiftIndex=0; shiftIndex<szVecs; shiftIndex++, shiftMPiter++) {  // Add the scores of all matched peaks without double-counting errors
		if(shiftMPiter->size()>0) {
			lastPeak = -1;
			for(unsigned int j=0; j<szSpec2; j++) match2[j]=0;
			list<TwoValues<int> >::iterator matchStart = shiftMPiter->begin(),
										    matchEnd = shiftMPiter->end(); 
			for(list<TwoValues<int> >::iterator matchIter = matchStart; matchIter != matchEnd; matchIter++) {
				if((*matchIter)[0]>lastPeak) { shiftScoresV[shiftIndex][0]+=spec1[(*matchIter)[0]][1];   shiftMPcount[shiftIndex][0]++;   lastPeak = (*matchIter)[0]; }
				if(!match2[(*matchIter)[1]]) { shiftScoresV[shiftIndex][1]+=spec2[(*matchIter)[1]][1];   shiftMPcount[shiftIndex][1]++;   match2[(*matchIter)[1]]=1; }
			}
		}
	}
	delete[] match2;
	
	float shiftMass=0, maxMass=validJumps.masses[validJumps.size()-1];
  	for(shiftIndex=0, shiftMPiter=shiftMatchedPeaks.begin(); shiftIndex<(int)szVecs; shiftIndex++,shiftMPiter++) { validShift[shiftIndex]=0;
		if(not shiftMPiter->empty()) {
			shiftMass = fabs((shiftIndex-shiftsOffset)*InputParams::Resolution);
			if(shiftMass>minAbsShift-peakTol and (shiftMass<=peakTol+0.000001 or shiftMass>maxMass+peakTol-0.000001 or validJumps.isValid(shiftMass,peakTol))) validShift[shiftIndex]=1;
		}
	}

    int middleShift = (int)round((spec1.parentMass-spec2.parentMass)/(2*InputParams::Resolution));
    int middleTimesTwo = (int)round((spec1.parentMass-spec2.parentMass)/InputParams::Resolution);
    int upperShiftLimit = shiftsOffset+middleShift;
    int shiftSym;
    list<TwoValues<float> > shiftsTmp; 
    TwoValues<float> curShift; 

	bestCandidateScores.set(0,0);     bestCandidateMP.set(0,0);
	for(shiftIndex=0; shiftIndex<upperShiftLimit; shiftIndex++) {
		if(!validShift[shiftIndex]) continue;
		shiftSym = shiftsOffset+(middleTimesTwo-(shiftIndex-shiftsOffset));

		float maxScore=0, score1, score2, curPenalty; int maxScoreIdx=-1;
		int numPeaks1, numPeaks2;
		for(int symIdx=max(0,shiftSym-intPMTol); symIdx<=shiftSym+intPMTol and symIdx<(int)szVecs ; symIdx++) {
			if(!validShift[symIdx]) continue;
			score1 = shiftScoresV[shiftIndex][0]+shiftScoresV[symIdx][0];  // +shiftScoresV[shiftSym][0];  Bug fixed 2006/06/19
			score2 = shiftScoresV[shiftIndex][1]+shiftScoresV[symIdx][1];  // +shiftScoresV[shiftSym][1];  
			numPeaks1 = shiftMPcount[shiftIndex][0]+shiftMPcount[symIdx][0];
			numPeaks2 = shiftMPcount[shiftIndex][1]+shiftMPcount[symIdx][1];
			if(score1>=minScore1 and score2>=minScore2 and numPeaks1>=minNumMatchedPeaks and numPeaks2>=minNumMatchedPeaks) { 
				curPenalty=round(abs(shiftSym-symIdx)*InputParams::Resolution); 
				if(score1+score2-curPenalty>maxScore) { maxScore=score1+score2-curPenalty; maxScoreIdx=symIdx; }
			} 
			if((score1+score2)>(bestCandidateScores[0]+bestCandidateScores[1])) bestCandidateScores.set(score1,score2);
			if((numPeaks1+numPeaks2)>(bestCandidateMP[0]+bestCandidateMP[1])) bestCandidateMP.set(numPeaks1,numPeaks2);
		}
		if(maxScoreIdx==-1) continue;

		curShift.set(maxScore,shiftIndex);   shiftsTmp.push_back(curShift);
		shiftPairsV[shiftIndex] = shiftSym;
	}

	shiftsTmp.sort();
	int maxUsedShiftIndex=0;  for(unsigned int i=0;i<szVecs;i++) validShift[i]=0;
	list<TwoValues<float> >::reverse_iterator iter = shiftsTmp.rbegin();
	for(; iter!=shiftsTmp.rend(); iter++) {
		shiftIndex = (int)round((*iter)[1]);   shiftSym = shiftPairsV[shiftIndex];   if(maxUsedShiftIndex<shiftSym) maxUsedShiftIndex=shiftSym;
		shiftScores.push_back((*iter)[0]);
		shiftPairs.push_back(TwoValues<unsigned int>(shiftIndex,shiftSym));
		validShift[shiftIndex]=1;   for(int shiftSymTol=max(0,shiftSym-intPMTol); shiftSymTol<=shiftSym+intPMTol and shiftSymTol<(int)szVecs; shiftSymTol++) validShift[shiftSymTol]=1;
	}
	
	shiftMPiter=shiftMatchedPeaks.begin();
	for(unsigned int i=0; i<szVecs; i++,shiftMPiter++)
		if(!validShift[i]) shiftMPiter->clear(); //.resize(0);

	return TwoValues<int>(shiftsOffset,maxUsedShiftIndex);
}

float ScoreOverlap6(Spectrum &spec1, vector<int> idx1all, Spectrum &spec2, vector<int> idx2all, float shift, 
					 float tolerance, vector<int> &idxMatched1, vector<int> &idxMatched2, 
					 float minIPdist, float *offsetPenalty){
    
    vector< TwoValues<float> > values(idx1all.size()+1);  // Keeps the values for the dynamic programming recursion: predecessor (col 0) and predecessor score (col 1)
                                                          // +1 because first line is (0,0) for DP initialization
    int forbiddenIdx;  // Index of the first forbidden PRM (because it is too close to the current PRM
    int maxIdx;        // Index of the best PRM that the current PRM can connect to
    float bestMatch;     // Best path score so far  
    int bestMatchIdx;  // Index of the last PRM in the best path so far
    int   i,j;           // Iterator vars
    
    idxMatched1.resize(0);   idxMatched2.resize(0);
    
    values[0][0]=0;   values[0][1]=0;
    maxIdx = 0;       forbiddenIdx = 1;
    bestMatch = 0;    bestMatchIdx = 0;
    for (i=1; i<(int)values.size(); i++) {
        while (forbiddenIdx<i and 
               spec1[idx1all[forbiddenIdx-1]][0] <= (spec1[idx1all[i-1]][0]-minIPdist+2*tolerance) and
               spec2[idx2all[forbiddenIdx-1]][0] <= (spec2[idx2all[i-1]][0]-minIPdist+2*tolerance)) {
            // This is executed only when forbidden is advanced
            if (values[forbiddenIdx][1] > values[maxIdx][1]) { maxIdx = forbiddenIdx; }
            forbiddenIdx++;
        }
        values[i][0] = maxIdx;
        values[i][1] = values[maxIdx][1] + spec1[idx1all[i-1]][1] + spec2[idx2all[i-1]][1];
        if(offsetPenalty) values[i][1]-= abs(spec1[idx1all[i-1]][0] - (spec2[idx2all[i-1]][0]+shift));        	
        if (values[i][1]>bestMatch) { bestMatch=values[i][1];  bestMatchIdx=i; }  // Keep track of where the best path ends
    }

    list<int> bestPath;
    while (bestMatchIdx>0) { 
    	if(offsetPenalty) (*offsetPenalty)+= abs(spec1[idx1all[bestMatchIdx-1]][0] - (spec2[idx2all[bestMatchIdx-1]][0]+shift));
    	bestPath.push_back(bestMatchIdx-1);  bestMatchIdx=(int)values[bestMatchIdx][0]; 
    }
	
    unsigned int bestPathLength = bestPath.size();
    idxMatched1.resize(bestPathLength);    
    idxMatched2.resize(bestPathLength);

    for(unsigned int idxPath=0; idxPath<bestPathLength; idxPath++) {
        idxMatched1[idxPath] = idx1all[bestPath.back()];
        idxMatched2[idxPath] = idx2all[bestPath.back()];
        bestPath.pop_back();
    }

    return bestMatch;
}

float ScoreOverlap7(Spectrum &spec1, vector<int> idx1all, Spectrum &spec2, vector<int> idx2all, float shift, float tolerance, 
                    vector<int> &idxMatched1, vector<int> &idxMatched2, float symmetryOffset){
	Spectrum tmpSpec;   vector<int> idxMatched;   float score;
	
	tmpSpec.parentMass = (spec1.parentMass+spec2.parentMass)/2;
	tmpSpec.peakList.resize(idx1all.size());
	tmpSpec.idDist = spec1.idDist;

	for(unsigned int i=0; i<tmpSpec.size(); i++) {
		tmpSpec[i].set((spec1[idx1all[i]][0]+spec2[idx2all[i]][0]+shift)/2,spec1[idx1all[i]][1]+spec2[idx2all[i]][1]);
	}
	score = getMaxSparseSet(tmpSpec, tolerance, symmetryOffset, idxMatched, true);
	idxMatched1.resize(idxMatched.size());   idxMatched2.resize(idxMatched.size());
	for(unsigned int i=0; i<idxMatched.size(); i++)
		{ idxMatched1[i]=idx1all[idxMatched[i]]; idxMatched2[i]=idx2all[idxMatched[i]]; }

	return score;
}

float getMaxSparseSet(Spectrum &spec, float tolerance, float pmOffset, vector<int> &idxMatched, bool includeSymmetric){
	vector<vector<TwoValues<int> > > prevPair,     // Previous pair for every possible pair
	                                   bestPair;     // Best pair so far
	vector<vector<float> > scores,         // Scores for every possible path
	                       bestScores;     // Best path score so far
	vector<float> peakScores;    // Adjusted peak scores (including the score of the symmetric peaks)
	vector<int> peakPairs;     // Indices of the other paired peak (if any) used in peakScores
	vector<TwoValues<int> > prefixPRMs;  // PRMs with mass<=aaMass/2 and sorted by increasing mass
	                                       //  Col 1 is PRM index, col 2 is index of closest PRM >=57-2*tolerance Da away
	vector<TwoValues<int> > suffixPRMs;  // PRMs with mass>aaMass/2 and sorted by increasing distance to aaMass
	TwoValues<int> globalBestPair(0,0);
	float aaMass = spec.parentMass+(pmOffset-spec.idDist)*AAJumps::massHion,
		  minInterPeakDist = 57*spec.idDist-2*tolerance,
		  globalBestScore=0;
	int i,j,k,p,idxPref,idxSuff;

	if(spec.size()==0) { idxMatched.resize(0); return 0; }

	short addZero=0, addPM=0;  
	Spectrum oldSpec;           // oldSpec is used to keep a copy of the input spectrum
	                            //   whenever addZero or addPM are >0
	if(spec[0][0]>tolerance) addZero++;
	if(spec[spec.size()-1][0]<aaMass-AAJumps::massH2O*spec.idDist-tolerance) addPM++;
	if(addZero+addPM>0) {
		oldSpec = spec;
		spec.peakList.resize(spec.size()+addZero+addPM);
		if (addZero) { for(i=spec.size()-1-addPM; i>=1; i--) spec[i]=spec[i-1]; spec[0].set(0,0); }
		if (addPM) spec[spec.size()-1].set(aaMass-AAJumps::massH2O*spec.idDist,0);
	}

	prefixPRMs.resize(spec.size());
	for(i=0; i<(int)spec.size() && spec[i][0]<=aaMass/2; i++) { 
		for(p=i-1; p>=0 && spec[i][0]-spec[p][0]<minInterPeakDist; p--);
		prefixPRMs[i].set(i,max((int)p,0)); 
	}
	prefixPRMs.resize(i);   suffixPRMs.resize(spec.size()-prefixPRMs.size());
	suffixPRMs[suffixPRMs.size()-1].set(spec.size()-1,0);
	for (j=spec.size()-1, k=0; j>=0 && spec[j][0]>aaMass/2; j--) {
		suffixPRMs[k][0]=j;
		for(p=0; suffixPRMs[p][0]>j && spec[suffixPRMs[p][0]][0]-spec[j][0]>minInterPeakDist; p++);
		suffixPRMs[k++].set(j,max(p-1,0));
	}

	prevPair.resize(prefixPRMs.size());   bestPair.resize(prefixPRMs.size());
	scores.resize(prefixPRMs.size());     bestScores.resize(prefixPRMs.size());
	for(int i=0; i<(int)prefixPRMs.size(); i++) { 
		prevPair[i].resize(suffixPRMs.size());   bestPair[i].resize(suffixPRMs.size());
		scores[i].resize(suffixPRMs.size());     bestScores[i].resize(suffixPRMs.size());
		for(j=0; j<suffixPRMs.size(); j++) { scores[i][j]=0; bestScores[i][j]=0; }
	}

	peakScores.resize(spec.size());   peakPairs.resize(spec.size());
	for(k=0;k<spec.size();k++) { peakScores[k]=spec[k][1]; peakPairs[k]=-1; }
	vector<vector<float> > pairs; vector<vector<int> > pairsIdx;
	spec.getPairs(pmOffset, tolerance, pairs, pairsIdx);
	for(k=0;k<pairsIdx.size();k++) {
		if(pairsIdx[k][0]<0 || pairsIdx[k][1]<0) continue;
		if(peakScores[pairsIdx[k][0]]<pairs[k][2]) { peakScores[pairsIdx[k][0]]=pairs[k][2]; peakPairs[pairsIdx[k][0]]=pairsIdx[k][1]; }
		if(peakScores[pairsIdx[k][1]]<pairs[k][2]) { peakScores[pairsIdx[k][1]]=pairs[k][2]; peakPairs[pairsIdx[k][1]]=pairsIdx[k][0]; }
	}


	scores[0][0]=peakScores[prefixPRMs[0][0]]+peakScores[suffixPRMs[0][0]];
	bestScores[0][0] = scores[0][0];  bestPair[0][0].set(0,0);
	globalBestScore = scores[0][0];   globalBestPair.set(0,0);
	for(i=0; i<prefixPRMs.size(); i++)
		for(j=0; j<suffixPRMs.size(); j++) {  
			if ((i==0 && j==0) || spec[suffixPRMs[j][0]][0]-spec[prefixPRMs[i][0]][0]<minInterPeakDist) continue;
			
			// Set default values of best scores/pairs for position [i][j]
			if(i>0) { bestScores[i][j]=bestScores[i-1][j];  bestPair[i][j]=bestPair[i-1][j]; }
			if(j>0 && bestScores[i][j-1]>bestScores[i][j]) { bestScores[i][j]=bestScores[i][j-1];  bestPair[i][j]=bestPair[i][j-1]; }

			idxPref = prefixPRMs[i][0];   idxSuff = suffixPRMs[j][0];
			// j ranges over suffixes whose masses differ from spec[i][0]
			if (fabs(spec[idxPref][0]+spec[idxSuff][0]-aaMass)>2*tolerance) {
				if (spec[idxPref][0]>aaMass-spec[idxSuff][0]) {  // last jump was on the prefix side
					scores[i][j] = peakScores[idxPref] + bestScores[prefixPRMs[i][1]][j];
					prevPair[i][j] = bestPair[prefixPRMs[i][1]][j];
				} else {    // last jump was on the suffix side
					scores[i][j] = peakScores[idxSuff] + bestScores[i][suffixPRMs[j][1]];
					prevPair[i][j] = bestPair[i][suffixPRMs[j][1]];
				}
			} else {  // still consider these pairs but don't increase the score
				scores[i][j] = bestScores[prefixPRMs[i][1]][j];
				prevPair[i][j] = bestPair[prefixPRMs[i][1]][j];
				if(scores[i][j] < bestScores[i][suffixPRMs[j][1]]) {
					scores[i][j] = bestScores[i][suffixPRMs[j][1]];
					prevPair[i][j] = bestPair[i][suffixPRMs[j][1]];
				}
			}
			
			if(scores[i][j]>bestScores[i][j]) { bestScores[i][j] = scores[i][j];  bestPair[i][j].set(i,j); }
			if(bestScores[i][j]>globalBestScore) { globalBestScore=bestScores[i][j];  globalBestPair.set(i,j); }
		}

	TwoValues<int> tmpPair;     int curPeakIdx;
	vector<bool> idxMatchedBool;   // Boolean vector used to mark matched peaks
	idxMatchedBool.resize(spec.size());   for(int i=0; i<spec.size(); i++) idxMatchedBool[i]=false;
	if(globalBestPair[0]>0) { 
		idxMatchedBool[prefixPRMs[globalBestPair[0]][0]-addZero]=true;
		if(includeSymmetric and peakPairs[prefixPRMs[globalBestPair[0]][0]]>0) idxMatchedBool[peakPairs[prefixPRMs[globalBestPair[0]][0]]-addZero]=true;
	}
	if(globalBestPair[1]>0) {
		idxMatchedBool[suffixPRMs[globalBestPair[1]][0]-addZero]=true;
		if(includeSymmetric and peakPairs[suffixPRMs[globalBestPair[1]][0]]>0) idxMatchedBool[peakPairs[suffixPRMs[globalBestPair[1]][0]]-addZero]=true;
	}
	while (globalBestPair[0]>0 || globalBestPair[1]>0) {
		tmpPair = prevPair[globalBestPair[0]][globalBestPair[1]];
		curPeakIdx=-1;
		if(tmpPair[0]!=globalBestPair[0]) 
			{ if(tmpPair[0]>0) curPeakIdx=prefixPRMs[tmpPair[0]][0]; }
		else if(tmpPair[1]>0) curPeakIdx=suffixPRMs[tmpPair[1]][0];
		if (curPeakIdx>0) {
			idxMatchedBool[curPeakIdx-addZero]=true;
			if (includeSymmetric and peakPairs[curPeakIdx]>0) idxMatchedBool[peakPairs[curPeakIdx]-addZero]=true;
		}
		globalBestPair = tmpPair;
	}
	if(addZero==0) idxMatchedBool[prefixPRMs[0][0]]=true;      
	if(addPM==0) idxMatchedBool[suffixPRMs[0][0]-addZero]=true; 

	idxMatched.resize(spec.size()); k=0;
	for(int i=0; i<spec.size(); i++) if(idxMatchedBool[i]) idxMatched[k++]=i;
	idxMatched.resize(k);

	if (addZero+addPM>0) spec=oldSpec; 
	return globalBestScore;
}
