#include "aminoacid.h"
#include "utils.h"

#include <iostream>
#include <cmath>

const unsigned int AAcount = 20;
// Blocked cysteines
const float AAmasses[]  = {71.0371137870,156.1011110260,115.0269430310,114.0429274460,160.0306482000,129.0425930950,128.0585775100,57.0214637230,137.0589118610,113.0840639790,113.0840639790,128.0949630160,131.0404846050,147.0684139150,97.0527638510,87.0320284090,101.0476784730,186.0793129520,163.0633285370,99.0684139150};
// Non-blocked cysteines
//const float AAmasses[]  = {71.0371137870,156.1011110260,115.0269430310,114.0429274460,103.009184477,129.0425930950,128.0585775100,57.0214637230,137.0589118610,113.0840639790,113.0840639790,128.0949630160,131.0404846050,147.0684139150,97.0527638510,87.0320284090,101.0476784730,186.0793129520,163.0633285370,99.0684139150};
const char  AAletters[] = {'A','R','D','N','C','E','Q','G','H','I','L','K','M','F','P','S','T','W','Y','V'};
const char* AAnames[]   = {"Alanine","Arginine","Aspartic acid","Asparagine","Cysteine","Glutamic acid","Glutamine","Glycine","Histidine","Isoleucine","Leucine","Lysine","Methionine","Phenylalanine","Proline","Serine","Threonine","Tryptophan","Tyrosine","Valine"};

void getMasses(char *sequence, vector<float> &masses) {
    masses.resize(strlen(sequence));
    int m;
    for(int i=0; i<masses.size(); i++) {
        m=0;
        while (m<20 and AAletters[m]!=sequence[i]) m++;
        if(m>=20) masses[i]=0; else masses[i]=AAmasses[m];
    }
}

//
// ************************************************************************************************
//  AAJumps::AAJumps(short maxJumpSize, float resolution, short useMods)
// ************************************************************************************************
//
AAJumps::AAJumps(short maxJumpSize, float resolution, short useMods) {
	aaLetters.resize(AAcount);
    for(unsigned int aaIdx=0; aaIdx<AAcount; aaIdx++) aaLetters[aaIdx]=AAletters[aaIdx];
    getjumps(maxJumpSize, resolution, useMods);
}

//
// ************************************************************************************************
//  void AAJumps::getjumps(short maxJumpSize, float resolution, short useMods)
// ************************************************************************************************
//
void AAJumps::getjumps(short maxJumpSize, float resolution, short useMods) {
    short step=0, aaIndex = 0;

	if(maxJumpSize==0) { masses.resize(1); masses[0]=0; return; }
	if(maxJumpSize<0) { masses.resize(0); return; }

    if (useMods==USE_MODS) { cerr << "(ERROR) AAjumps::getjumps: USE_MODS not implemented yet\n"; masses.resize(0); exit(-1); }
    else modsUsed = useMods;

    int numMassesPerStep=AAcount, totalNumMasses=numMassesPerStep;
    for (step=2; step<=maxJumpSize; step++) { numMassesPerStep*=AAcount;   totalNumMasses+=numMassesPerStep; }
    masses.resize(totalNumMasses);

    // This takes care of step 1
    for (aaIndex=0; aaIndex<AAcount ; aaIndex++) masses[aaIndex]=AAmasses[aaIndex];

    int prevStepStart = 0,          // Start of previous step's jumps
        prevStepEnd   = AAcount-1,  // End of previous step's jumps
        prevIter,                   // Iterator for previous step's jumps
        curStepStart  = AAcount,    // Start of current step's jumps
        curStepIter = curStepStart; // Iterates through current step's jumps

    for (step=2; step<=maxJumpSize ; step++) {
        for (prevIter=prevStepStart ; prevIter<=prevStepEnd ; prevIter++)
            for (aaIndex=0; aaIndex<AAcount ; aaIndex++) masses[curStepIter++]=masses[prevIter]+AAmasses[aaIndex];
        prevStepStart = curStepStart;
        prevStepEnd   = curStepIter-1;
        curStepStart  = curStepIter;
    }

	if(maxJumpSize==1) {
		vector<unsigned int> idx;      masses = Utils::unique(masses,resolution,&idx);
		aaLetters.resize(idx.size());  
		for(unsigned int aaIdx=0;aaIdx<idx.size();aaIdx++) aaLetters[aaIdx]=AAletters[idx[aaIdx]];
	} else { aaLetters.resize(0); masses = Utils::unique(masses,resolution); }
}

//
// ************************************************************************************************
//  void AAJumps::alljumps(short maxJumpMass, float resolution, short useMods)
// ************************************************************************************************
//
void AAJumps::alljumps(float maxJumpMass, float resolution, short useMods) {
	unsigned int vecSize = (unsigned int)ceil(10*maxJumpMass/resolution)+1, jumpIdx, destIdx;  // Additional order of magnitute to minimize rounding errors
	short aaIdx;
	vector<bool> *jumpOk = new vector<bool>(vecSize);

	for(jumpIdx=0; jumpIdx<vecSize; jumpIdx++) (*jumpOk)[jumpIdx]=false;
	
	getjumps(4,resolution,useMods); // Compute exact jump masses up to 4 amino acid jumps
	for(jumpIdx=0; jumpIdx<masses.size(); jumpIdx++)  // and initialize valid jumps
		(*jumpOk)[(int)round(10*masses[jumpIdx]/resolution)]=true;
	
	// Add new valid jumps from every valid jump
	unsigned int countOk=0;
	for(jumpIdx=0; jumpIdx<vecSize; jumpIdx++)
		if((*jumpOk)[jumpIdx]) {   countOk++;
			for(aaIdx=0; aaIdx<AAcount; aaIdx++) {
				destIdx = jumpIdx + (unsigned int)round(10*AAmasses[aaIdx]/resolution);
				if(destIdx<vecSize) (*jumpOk)[destIdx]=true;
			}
		}

	masses.resize(countOk+1);   destIdx=0;
	for(jumpIdx=0; jumpIdx<vecSize; jumpIdx++)
		if((*jumpOk)[jumpIdx]) masses[destIdx++]=jumpIdx*resolution/10;
	
    masses = Utils::unique(masses,resolution);  // Round all final jumps down to the required resolution
	
	delete jumpOk;
}

//
// ************************************************************************************************
//  void AAJumps::forceDoubleSided()
//
//       Makes masses = [-masses; masses];
// ************************************************************************************************
//
void AAJumps::forceDoubleSided(){
    vector<float> tmp(masses);

    masses.resize(2*masses.size());
    for(int i=0; i<tmp.size(); i++) { masses[i] = -tmp[i];   masses[i+tmp.size()] = tmp[i]; }
}

//
// ************************************************************************************************
//  void AAJumps::forceJump(float mass)
//
//      Adds a given mass to the set of valid jumps
// ************************************************************************************************
//
void AAJumps::forceJumps(vector<float> newMasses){
    masses.reserve(masses.size()+newMasses.size());
    for(int i=0; i<newMasses.size(); i++) masses.push_back(newMasses[i]);
    sort(masses.begin(),masses.end());
}

//
// ************************************************************************************************
//  void AAJumps::forceTolerance(float tolerance, float resolution)
//
//      Every mass m in masses is replaced by a set of masses m+[-tolerance:resolution:tolerance]
// ************************************************************************************************
//
void AAJumps::forceTolerance(float tolerance, float resolution){
    float iter;

    int szRange = 0;
    for (iter=-tolerance ; iter<=tolerance+0.00001 ; iter+=resolution) szRange++;  if (szRange==0) return;

    vector<float> tmp(masses);
    masses.resize(masses.size()*szRange);
    for(int i=0, curMass=0; i<tmp.size(); i++)
        for(iter=-tolerance; iter<=tolerance+0.00001 ; iter+=resolution)
            masses[curMass++] = tmp[i]+iter;
}


//
// ************************************************************************************************
//  void AAJumps::removeHigherJumps(float highestMass)
//
//      Removes all jumps of mass higher than largestJump
// ************************************************************************************************
//
void AAJumps::removeHigherJumps(float largestJump) {
	unsigned int idx;
	for(idx=0; idx<masses.size() and masses[idx]<=largestJump; idx++);
	masses.resize(idx);
}


//
// ************************************************************************************************
//  bool AAJumps::isValid(float mass, float tolerance)
//
//      Test whether a given mass is a valid jump in the current set
// ************************************************************************************************
//
bool AAJumps::isValid(float mass, float tolerance) {
	if(masses.size()==0) return false;
	unsigned int curIdx = (masses.size())/2, high = masses.size(), low=0;

	while(low<high) {
		if(fabs(masses[curIdx]-mass)<=tolerance+0.000001) return true;
		if(masses[curIdx]<mass) low=curIdx+1; else high=curIdx;  // High is first non-checked position, low is lowest eligible position
		curIdx = (int)(high+low)/2;
	}
	if(curIdx<masses.size() and fabs(masses[curIdx]-mass)<=tolerance+0.0001) return true;
	return false;
}

//
// ************************************************************************************************
//  unsigned int AAJumps::find(float mass, float tolerance, TwoValues<unsigned int> &idxBounds)
//
//      Find the indices of all jumps within tolerance of mass. On exit, idxBounds[0] is the
//    index of the first matching mass and idxBounds[1] is the index of the last match plus one.
// ************************************************************************************************
//
unsigned int AAJumps::find(float mass, float tolerance, TwoValues<unsigned int> &idxBounds) {
	idxBounds.set(0,0);
	if(masses.size()==0) return 0;

	unsigned int curIdx = (masses.size())/2, high = masses.size(), low=0;
	bool found=false;
	while(low<high and not found) {
		if(fabs(masses[curIdx]-mass)<=tolerance+0.000001) { found=true; break; }
		if(masses[curIdx]<mass) low=curIdx+1; else high=curIdx;  // High is first non-checked position, low is lowest eligible position
		curIdx = (int)(high+low)/2;
	}
	if(not found) return 0;

	for(; curIdx>0 and fabs(masses[curIdx]-mass)<=tolerance+0.000001; curIdx--);  // Find first match
	if(fabs(masses[curIdx]-mass)>tolerance+0.000001) curIdx++;
	idxBounds[0]=curIdx;
	for(; curIdx<masses.size() and masses[curIdx]-mass<=tolerance+0.000001; curIdx++);  // Find first index after last match
	idxBounds[1]=curIdx;

	return idxBounds[1]-idxBounds[0];
}
