#include "batch.h"
#include "aminoacid.h"
#include "alignment_scoring.h"
#include "inputParams.h"

#include <iostream>
#include <fstream>
#include <list>
#include <algorithm>
#include <cmath>
#include <ctime>
#include <cstdio>

inline void mergeVectors(vector<int> &v1, vector<int> &v2, vector<int> &vOut) {
	unsigned int i1=0, i2=0, iOut=0;
	vOut.resize(v1.size()+v2.size());
	while(i1<v1.size() or i2<v2.size()) {
		if(i1==v1.size()) { vOut[iOut++]=v2[i2++]; continue; }
		if(i2==v2.size()) { vOut[iOut++]=v1[i1++]; continue; }
		if(v1[i1]==v2[i2]) { vOut[iOut++]=v1[i1++]; i2++; continue; }
		if(v1[i1]<v2[i2]) vOut[iOut++]=v1[i1++]; else vOut[iOut++]=v2[i2++]; 
	}
	vOut.resize(iOut);
}

inline void aux_updateMeanVariance(float newValue, float sampleSize, float &mean, float &variance) {
	float updateRatio = sampleSize/(sampleSize+1);   sampleSize++;
	mean = mean*updateRatio + newValue/sampleSize;
	variance = updateRatio*variance+(newValue*newValue)/sampleSize;
}

void getPairAlignsPA2(SpecSet &specSet, unsigned int startIdx, unsigned int endIdx, 
                        float peakTol, float pmTol, float minRatio, float minPeakAreaOvlp, 
                        short minNumMatchedPeaks, AAJumps &allowedJumps, float minAbsShift,
						list<Results_PA> &results, list<TwoValues<float> > &ratios, list<TwoValues<int> > &numMatchedPeaks,
						vector<TwoValues<float> > &means, vector<float> &varTerms,
						list<vector<float> > &alignStats, vector<vector<float> > &specStats){
	unsigned int spec1, spec2;

	TwoValues<int> res;
	list<float> shiftScores;
	list<TwoValues<unsigned int> > shiftPairs;
	vector<list<TwoValues<int> > > shiftMatchedPeaks;
	vector<TwoValues<float> > shiftDPscores;
	vector<float> shiftDPpenalties;
	vector<TwoValues<vector<int> > > shiftDPmatchedPeaks;
	vector<TwoValues<float> > minMaxMatchScores(specSet.size());
	Results_PA curResult;
	vector<float> curAlignStats(4);

	means.resize(specSet.size());       varTerms.resize(specSet.size());     specStats.resize(specSet.size());
 	for(unsigned int i=0; i<means.size(); i++) { 
		means[i].set(0,0); varTerms[i]=0; 
		specStats[i].resize(9); for(unsigned int j=0; j<9; j++) specStats[i][j]=0;
	}
	
	float maxParentMass = 0;  unsigned int maxNumPeaks=0;   
	int intPeakTol = (int)round(peakTol/InputParams::Resolution), intPMTol = (int)round(pmTol/InputParams::Resolution);
	for(unsigned int i=0; i<specSet.size(); i++) {
		if(specSet[i].parentMass>maxParentMass) maxParentMass=specSet[i].parentMass;
		if(specSet[i].size()>maxNumPeaks) maxNumPeaks=specSet[i].size();
		minMaxMatchScores[i].set(0,0); for(unsigned int j=0; j<specSet[i].size();j++) minMaxMatchScores[i][1]+=specSet[i][j][1];
		minMaxMatchScores[i][0]=minRatio*minMaxMatchScores[i][1];
	}

	shiftMatchedPeaks.resize(1+(int)ceil(2*(min(maxParentMass,InputParams::MaxShift)+peakTol+pmTol)/InputParams::Resolution));
	shiftDPscores.reserve(shiftMatchedPeaks.capacity());   shiftDPpenalties.reserve(shiftMatchedPeaks.capacity());
	shiftDPmatchedPeaks.reserve(shiftMatchedPeaks.capacity());
	vector<int> idx1, idx2; idx1.reserve(maxNumPeaks*(2*intPeakTol+1)+1); idx2.reserve(idx1.capacity());
	vector<int> idxMerged1, idxMerged2; idxMerged1.reserve(idx1.capacity()); idxMerged2.reserve(idx1.capacity());
	
	time_t curTime=time(0); double totTime=0.0;  int numResults=0;
	float bestShiftScore;   
	TwoValues<int> bestShiftPair; 
	TwoValues<int> bestNumMatchedPeaks;   
	TwoValues<float> bestShiftScores, bestCandidateScores;   
	list<int> shiftsToCompute;
    for(spec1=startIdx; spec1<=endIdx; spec1++) {
	    for(spec2=spec1+1; spec2<specSet.size(); spec2++) {
	    	for(unsigned int i=0; i<curAlignStats.size(); i++) curAlignStats[i]=0;

			res = computeShifts(specSet[spec1], specSet[spec2], peakTol, pmTol, minRatio, minPeakAreaOvlp, minNumMatchedPeaks, allowedJumps, shiftScores, shiftPairs, shiftMatchedPeaks, bestCandidateScores, bestNumMatchedPeaks, minAbsShift);
			if(res[1]==0) continue;
			
			aux_updateMeanVariance(bestCandidateScores[0],specStats[spec1][0],specStats[spec1][5],specStats[spec1][6]);
			aux_updateMeanVariance(bestCandidateScores[1],specStats[spec2][0],specStats[spec2][5],specStats[spec2][6]);
			aux_updateMeanVariance(bestNumMatchedPeaks[0],specStats[spec1][0],specStats[spec1][1],specStats[spec1][2]);
			aux_updateMeanVariance(bestNumMatchedPeaks[1],specStats[spec2][0],specStats[spec2][1],specStats[spec2][2]);
			curAlignStats[0] = min(bestNumMatchedPeaks[0],bestNumMatchedPeaks[1]);
			curAlignStats[1] = min(bestCandidateScores[0]/minMaxMatchScores[spec1][1],bestCandidateScores[1]/minMaxMatchScores[spec2][1]);

 			shiftDPscores.resize(shiftMatchedPeaks.size()); for(unsigned int i=0; i<shiftDPscores.size(); i++) shiftDPscores[i].set(-1.0,-1.0);
			shiftDPpenalties.resize(shiftMatchedPeaks.size()); for(unsigned int i=0; i<shiftDPpenalties.size(); i++) shiftDPpenalties[i]=0.0;
			shiftDPmatchedPeaks.resize(shiftMatchedPeaks.size()); 
			for(unsigned int i=0; i<shiftDPmatchedPeaks.size(); i++) { shiftDPmatchedPeaks[i][0].resize(0);  shiftDPmatchedPeaks[i][1].resize(0); }

			bestShiftScore = 0;   bestShiftPair.set(-1,-1);   bestShiftScores.set(0,0);   
			bestCandidateScores.set(0,0);   bestNumMatchedPeaks.set(0,0);
			list<float>::iterator nextShiftScore = shiftScores.begin();
			list<TwoValues<unsigned int> >::iterator nextShiftPair = shiftPairs.begin();
			while(nextShiftScore!=shiftScores.end() and *nextShiftScore > bestShiftScore) {
				int shiftIndex=(*nextShiftPair)[0], shiftSym=(*nextShiftPair)[1];
				if(shiftDPscores[shiftIndex][0]+shiftDPscores[shiftIndex][1]<0.0001 and shiftMatchedPeaks[shiftIndex].size()>0) shiftsToCompute.push_back(shiftIndex);
				for(int tolIdx=max(0,shiftSym-intPMTol); tolIdx<=shiftSym+intPMTol and tolIdx<(int)shiftDPscores.size(); tolIdx++)
					if(shiftDPscores[tolIdx][0]+shiftDPscores[tolIdx][1]<0.0001 and shiftMatchedPeaks[tolIdx].size()>0) shiftsToCompute.push_back(tolIdx);
					
				for(list<int>::iterator curShift = shiftsToCompute.begin(); curShift != shiftsToCompute.end(); curShift++) {
					float shiftMass = ((*curShift)-res[0])*InputParams::Resolution;

					idx1.resize(shiftMatchedPeaks[*curShift].size());   idx2.resize(shiftMatchedPeaks[*curShift].size());  int pivot=0;
					for(list<TwoValues<int> >::iterator iterPeaks=shiftMatchedPeaks[*curShift].begin(); iterPeaks!=shiftMatchedPeaks[*curShift].end(); iterPeaks++)
						{ idx1[pivot]=(*iterPeaks)[0]; idx2[pivot]=(*iterPeaks)[1]; pivot++; }

					ScoreOverlap6(specSet[spec1], idx1, specSet[spec2], idx2, shiftMass, peakTol, shiftDPmatchedPeaks[*curShift][0], shiftDPmatchedPeaks[*curShift][1], AAJumps::minAAmass, &shiftDPpenalties[*curShift]);
		            shiftDPscores[*curShift][0]=0; for(unsigned int i=0;i<shiftDPmatchedPeaks[*curShift][0].size();i++) shiftDPscores[*curShift][0]+=specSet[spec1][shiftDPmatchedPeaks[*curShift][0][i]][1];
		            shiftDPscores[*curShift][1]=0; for(unsigned int i=0;i<shiftDPmatchedPeaks[*curShift][1].size();i++) shiftDPscores[*curShift][1]+=specSet[spec2][shiftDPmatchedPeaks[*curShift][1][i]][1];
				}
				shiftsToCompute.clear();
				
				for(int tolIdx=max(0,shiftSym-intPMTol); tolIdx<=shiftSym+intPMTol and tolIdx<(int)shiftDPmatchedPeaks.size(); tolIdx++) {
					if(shiftMatchedPeaks[tolIdx].size()==0) continue;

					mergeVectors(shiftDPmatchedPeaks[shiftIndex][0], shiftDPmatchedPeaks[tolIdx][0], idxMerged1);
					mergeVectors(shiftDPmatchedPeaks[shiftIndex][1], shiftDPmatchedPeaks[tolIdx][1], idxMerged2);

					float score1=0;   for(unsigned int i=0;i<idxMerged1.size();i++) score1+=specSet[spec1][idxMerged1[i]][1];
					float score2=0;   for(unsigned int i=0;i<idxMerged2.size();i++) score2+=specSet[spec2][idxMerged2[i]][1];
					if(bestShiftScore<0.0001) {
						if(score1+score2>bestCandidateScores[0]+bestCandidateScores[1])	bestCandidateScores.set(score1,score2);
						if(idxMerged1.size()+idxMerged2.size()>bestNumMatchedPeaks[0]+bestNumMatchedPeaks[1]) bestNumMatchedPeaks.set(idxMerged1.size(),idxMerged2.size()); 
					}

					if(score1<minMaxMatchScores[spec1][0] or score2<minMaxMatchScores[spec2][0]) continue;
					if(score1+score2-shiftDPpenalties[shiftIndex]-shiftDPpenalties[tolIdx]>bestShiftScore) { 
						bestShiftScore=score1+score2-shiftDPpenalties[shiftIndex]-shiftDPpenalties[tolIdx]; 
						bestShiftPair.set(shiftIndex,tolIdx); 
						bestShiftScores.set(score1,score2);
						bestNumMatchedPeaks.set(idxMerged1.size(),idxMerged2.size());
					}
				}
				nextShiftScore++;   nextShiftPair++;
			}
			
			TwoValues<float> curRatios;
			if(bestShiftScore>0) {
				curResult.spec1 = spec1;                  curResult.spec2 = spec2;
				curResult.score1 = bestShiftScores[0];    curResult.score2 = bestShiftScores[1];
				curResult.shift1 = (bestShiftPair[0]-res[0])*InputParams::Resolution;
				curResult.shift2 = (bestShiftPair[1]-res[0])*InputParams::Resolution;				
				results.push_back(curResult);
				curRatios.set(curResult.score1/minMaxMatchScores[spec1][1],curResult.score2/minMaxMatchScores[spec2][1]);
				ratios.push_back(curRatios);
				numMatchedPeaks.push_back(bestNumMatchedPeaks);
				bestCandidateScores = bestShiftScores;

				curAlignStats[2] = min(bestNumMatchedPeaks[0],bestNumMatchedPeaks[1]);
				curAlignStats[3] = min(bestCandidateScores[0]/minMaxMatchScores[spec1][1],bestCandidateScores[1]/minMaxMatchScores[spec2][1]);
				alignStats.push_back(curAlignStats);
			}
			
			aux_updateMeanVariance(bestCandidateScores[0],means[spec1][1],means[spec1][0],varTerms[spec1]);
			aux_updateMeanVariance(bestCandidateScores[1],means[spec2][1],means[spec2][0],varTerms[spec2]);
			aux_updateMeanVariance(bestCandidateScores[0],specStats[spec1][0],specStats[spec1][7],specStats[spec1][8]);
			aux_updateMeanVariance(bestCandidateScores[1],specStats[spec2][0],specStats[spec2][7],specStats[spec2][8]);
			aux_updateMeanVariance(bestNumMatchedPeaks[0],specStats[spec1][0],specStats[spec1][3],specStats[spec1][4]);
			aux_updateMeanVariance(bestNumMatchedPeaks[1],specStats[spec2][0],specStats[spec2][3],specStats[spec2][4]);
			specStats[spec1][0]++;   means[spec1][1]++;
			specStats[spec2][0]++;   means[spec2][1]++;
	    }
		time_t newTime = time(0); double ellapsed=difftime(newTime,curTime); totTime+=ellapsed; curTime=newTime;  int N=endIdx,CUR=spec1-startIdx+1;
		numResults = results.size();
    }
    computeShifts_cleanup();
}

bool Load_pabatch(char *filename, vector<PairAlign> &aligns) {
    ifstream input(filename);
    if (!input) { cerr << "ERROR: cannot open " << filename << "\n";   aligns.resize(0);  return false; }
    int numPairs, numShifts, i, j;

    input >> numPairs;    aligns.resize(numPairs);
    for(i=0; i<numPairs ; i++) {
        input >> aligns[i].spec1 >> aligns[i].spec2 >> numShifts;
        aligns[i].shifts.resize(numShifts);   aligns[i].results.resize(numShifts);
        for(j=0 ; j<numShifts ; j++)  input >> aligns[i].shifts[j];
    }

    input.close();
    return true;
}

bool Load_resultsASP(char *filename, vector<Results_ASP> &results) {
    ifstream input(filename);
    if (!input) { cerr << "ERROR: cannot open " << filename << "\n";   results.resize(0);  return false; }
    int numResults, i, j;
    char separator;

    input >> numResults;    results.resize(numResults);
    for(i=0; i<numResults ; i++) {
        input >> results[i].spec1 >> separator >> results[i].spec2 >> separator >> results[i].shift1 >> separator >> results[i].score1 >> separator >> results[i].score2;
        results[i].spec1--; results[i].spec2--;
    }

    input.close();
    return true;
}

bool Load_resultsASPbin(char *filename, vector<Results_ASP> &results) {
    FILE *fp;   unsigned int numEntries;
    fp = fopen(filename,"r");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return false; }

    fread(&numEntries,sizeof(unsigned int),1,fp);
	results.resize(numEntries);

    float data[5];   int read_result;
	for(unsigned int resIdx=0; resIdx<numEntries; resIdx++) {
		read_result = fread(data,sizeof(float),5,fp);   if(read_result!=5) { cerr<<"ERROR reading "<<filename<<"!\n"; results.resize(0); return false; }
		results[resIdx].spec1=(int)data[0]-1;   results[resIdx].spec2=(int)data[1]-1;
		results[resIdx].shift1=data[2];    results[resIdx].score1=data[3];   results[resIdx].score2=data[4];
	}

    fclose(fp);
    return true;
}

bool Load_resultsPA(char *filename, vector<Results_PA> &results) {
    ifstream input(filename);
    if (!input) { cerr << "ERROR: cannot open " << filename << "\n";   results.resize(0);  return false; }
    int numResults, i, j;
    char separator;

    input >> numResults;    results.resize(numResults);
    for(i=0; i<numResults ; i++) {
        input >> results[i].spec1 >> separator >> results[i].spec2 >> separator >> results[i].shift1 >> separator >> results[i].shift2 >> separator >> results[i].score1 >> separator >> results[i].score2;
        results[i].spec1--; results[i].spec2--;
    }

    input.close();
    return true;
}

bool Save_resultsPAbin(char *filename, list<Results_PA> &results) {
    FILE *fp;   unsigned int numEntries = results.size();
    float data[6];
    unsigned int i,p;
    
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return false; }
    
    fwrite(&numEntries,sizeof(int),1,fp);

	for(list<Results_PA>::iterator iter= results.begin();iter!=results.end();iter++) {
		data[0]=(*iter).spec1+1;  data[1]=(*iter).spec2+1;
		data[2]=(*iter).shift1;   data[3]=(*iter).shift2;
		data[4]=(*iter).score1;   data[5]=(*iter).score2;
		fwrite(data,sizeof(float),6,fp);
	}

    fclose(fp);
    return true;
}

bool Save_resultsASPbin(char *filename, list<Results_ASP> &results) {
    FILE *fp;   unsigned int numEntries = results.size();
    float data[5];
    unsigned int i,p;
    
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return false; }
    
    fwrite(&numEntries,sizeof(unsigned int),1,fp);

	for(list<Results_ASP>::iterator iter= results.begin();iter!=results.end();iter++) {
		data[0]=(*iter).spec1+1;   data[1]=(*iter).spec2+1;
		data[2]=(*iter).shift1;    data[3]=(*iter).score1;   data[4]=(*iter).score2;
		fwrite(data,sizeof(float),5,fp);
	}

    fclose(fp);
    return true;
}

bool Save_resultsASPbin(char *filename, vector<Results_ASP> &results) { 
    FILE *fp;   unsigned int numEntries = results.size();
    float data[5];
    unsigned int i,p;
    
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return false; }
    
    fwrite(&numEntries,sizeof(unsigned int),1,fp);  // Number of entries in the file

	for(i=0;i<numEntries;i++) {
		data[0]=results[i].spec1+1;  data[1]=results[i].spec2+1;
		data[2]=results[i].shift1;   data[3]=results[i].score1;    data[4]=results[i].score2;
		fwrite(data,sizeof(float),5,fp);
	}

    fclose(fp);
    return true;
}

bool Save_results(char *filename, vector<PairAlign> &aligns) {
    ofstream output(filename);
    if (!output) { cerr << "ERROR: cannot open " << filename << "\n";   aligns.resize(0);  return false; }
    unsigned int i,j,k;
    
    output << aligns.size() << endl;
    for(i=0; i<aligns.size() ; i++) {
        output << aligns[i].spec1 << " " << aligns[i].spec2 << " " << aligns[i].shifts.size() << endl;
        for(j=0; j<aligns[i].shifts.size() ; j++){
            output << aligns[i].shifts[j] << " " << aligns[i].results[j].score << ";";
            for (k=0; k<aligns[i].results[j].idxMatched1.size() ; k++) output << " " << aligns[i].results[j].idxMatched1[k]+1;
            output << ";";
            for (k=0; k<aligns[i].results[j].idxMatched2.size() ; k++) output << " " << aligns[i].results[j].idxMatched2[k]+1;
            output << endl;
        }
    }
    
    output.close();    return true;
}

bool Save_resultsCS(char *filename, list<Results_CS> &aligns, char sep) {
    ofstream output(filename);
    if (!output) { cerr << "ERROR: cannot open " << filename << "\n"; return false; }
    unsigned int i,j,k;
    list<Results_CS>::iterator iter = aligns.begin();
    
    output << aligns.size() << endl;
    for(iter = aligns.begin(); iter != aligns.end() ; iter++)
        output << iter->spec1 << sep << iter->spec2 << sep << iter->specC << sep << iter->shift1 << sep << iter->shift2 << endl;
    
    output.close();    return true;
}

/*int Save_binArray(char *filename, vector<vector<float> > data) {
    FILE *fp;   unsigned int numLines = data.size(), numCols;
    if(numLines==0) return -1; else numCols = data[0].size();
    float *streamData = (float *)malloc(numLines*numCols);
    unsigned int i,j;
    
    fp = fopen(filename,"w");
    if ((int)fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }
    
    fwrite(&numLines,sizeof(int),1,fp);
    fwrite(&numCols,sizeof(int),1,fp);

	for(i=0;i<numLines;i++)
		for(j=0;j<numCols;j++) streamData[i*numLines+j]=data[i][j];

	fwrite(streamData,sizeof(float),numLines*numCols,fp);
	free(streamData);

    fclose(fp);
    return 1;
}
*/

// Hack to force the instantiation of a few variants of the function template
void binArray_instantiator() {
	vector<vector<float> > dataF;   Save_binArray("", dataF);  Load_binArray("", dataF);
	vector<vector<int> > dataI;     Save_binArray("", dataI);  Load_binArray("", dataI);
	vector<vector<double> > dataD;  Save_binArray("", dataD);  Load_binArray("", dataD);
	vector<vector<short> > dataS;   Save_binArray("", dataS);  Load_binArray("", dataS);

	vector<list<float> > dataFL;   Save_binListArray<float,list<float>,list<float>::iterator>("", dataFL);
								   Save_binListArray<float,vector<float>,vector<float>::iterator >("", dataF);
								   Load_binListArray<float,list<float>,list<float>::iterator>("", dataFL);
	vector<list<int> > dataIL;     Save_binListArray<int,list<int>,list<int>::iterator>("", dataIL);
								   Save_binListArray<int,vector<int>,vector<int>::iterator >("", dataI);
								   Load_binListArray<int,list<int>,list<int>::iterator>("", dataIL);
	vector<list<double> > dataDL;  Save_binListArray<double,list<double>,list<double>::iterator>("", dataDL);
								   Save_binListArray<double,vector<double>,vector<double>::iterator >("", dataD);
								   Load_binListArray<double,list<double>,list<double>::iterator>("", dataDL);
	vector<list<short> > dataSL;   Save_binListArray<short,list<short>,list<short>::iterator>("", dataSL);
								   Save_binListArray<short,vector<short>,vector<short>::iterator >("", dataS);
								   Load_binListArray<short,list<short>,list<short>::iterator>("", dataSL);

	vector<float> dataF1d;  Save_binArray<float>("", dataF1d);
	vector<int> dataI1d;    Save_binArray<int>("", dataI1d);
	vector<unsigned int> dataUI1d;    Save_binArray<unsigned int>("", dataUI1d);
	vector<double> dataD1d; Save_binArray<double>("", dataD1d);
	vector<short> dataS1d;  Save_binArray<short>("", dataS1d);

	vector<TwoValues<short> > dataStv;    Save_binArray("", dataStv);
	vector<TwoValues<int> > dataItv;      Save_binArray("", dataItv);
	vector<TwoValues<unsigned int> > dataUItv; Save_binArray("", dataUItv);
	vector<TwoValues<float> > dataFtv;    Save_binArray("", dataFtv);
	vector<TwoValues<double> > dataDtv;   Save_binArray("", dataDtv);
	list<TwoValues<short> > dataSLtv;     Save_binArray("", dataSLtv);
	list<TwoValues<float> > dataFLtv;     Save_binArray("", dataFLtv);
	list<TwoValues<double> > dataDLtv;    Save_binArray("", dataDLtv);

	list<vector<unsigned int> > dataUIV;   Save_binListArray<unsigned int,vector<unsigned int>,vector<unsigned int>::iterator>("", dataUIV);
	                               		   Save_binArray<unsigned int>("", dataUIV);
    list<vector<float> > dataFV;   Save_binListArray<float,vector<float>,vector<float>::iterator>("", dataFV);
	                               Save_binArray<float>("", dataFV);
	vector<vector<unsigned int> > dataUIVV; Save_binListArray<unsigned int,vector<unsigned int>,vector<unsigned int>::iterator>("", dataUIVV);
	                                        Save_binArray<unsigned int>("", dataUIVV);
											Load_binListArray<unsigned int,vector<unsigned int>,vector<unsigned int>::iterator>("", dataUIVV);
	vector<list<unsigned int> > dataUIVL;   Save_binListArray<unsigned int,list<unsigned int>,list<unsigned int>::iterator>("", dataUIVL);
											Load_binListArray<unsigned int,list<unsigned int>,list<unsigned int>::iterator>("", dataUIVL);
}

template<class T> int Save_binArray(char *filename, vector<T> &data) {
    FILE *fp;   unsigned int numLines = data.size(), numCols=1;
    T *streamData = (T *)malloc(sizeof(T)*numLines);
    unsigned int i;
    
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    fwrite(&numLines,sizeof(int),1,fp);
    fwrite(&numCols,sizeof(int),1,fp);

	for(i=0;i<numLines;i++) streamData[i]=data[i];

	fwrite(streamData,sizeof(T),numLines,fp);
	free(streamData);

    fclose(fp);
    return 1;
}

template<class T> int Save_binArray(char *filename, vector<TwoValues<T> > &data) {
    FILE *fp;   unsigned int numLines = data.size(), numCols=2;
    T *streamData = (T *)malloc(sizeof(T)*numCols*numLines);
    unsigned int i;
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    fwrite(&numLines,sizeof(int),1,fp);
    fwrite(&numCols,sizeof(int),1,fp);

	for(i=0;i<numLines;i++) { streamData[i<<1]=data[i][0]; streamData[(i<<1)+1]=data[i][1]; }

	fwrite(streamData,sizeof(T),numLines*numCols,fp);
	free(streamData);

    fclose(fp);
    return 1;
}

template<class T> int Save_binArray(char *filename, list<TwoValues<T> > &data) {
    FILE *fp;   unsigned int numLines = data.size(), numCols=2;
    T *streamData = (T *)malloc(sizeof(T)*numCols*numLines);
    
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    fwrite(&numLines,sizeof(int),1,fp);
    fwrite(&numCols,sizeof(int),1,fp);

	typename list<TwoValues<T> >::iterator iter;   unsigned int i=0;
	for(iter = data.begin(); iter != data.end(); iter++) 
		{ streamData[i++]=(*iter)[0]; streamData[i++]=(*iter)[1]; }

	fwrite(streamData,sizeof(T),numLines*numCols,fp);
	free(streamData);

    fclose(fp);
    return 1;
}

template<class T> int Save_binArray(char *filename, vector<vector<T> > &data) {
    FILE *fp;   unsigned int numLines = data.size(), numCols;
    if(numLines==0) return -1; else numCols = data[0].size();
    T *streamData = (T *)malloc(sizeof(T)*numLines*numCols);
    unsigned int i,j;
    
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    fwrite(&numLines,sizeof(int),1,fp);
    fwrite(&numCols,sizeof(int),1,fp);

	for(i=0;i<numLines;i++) {
		if(data[i].size()!=numCols) { cerr<<"ERROR in Save_binArray: Not enough data at position "<<i<<"!\n"; exit(-1); }
		for(j=0;j<numCols;j++) streamData[i*numCols+j]=data[i][j];
	}

	fwrite(streamData,sizeof(T),numLines*numCols,fp);
	free(streamData);

    fclose(fp);
    return 1;
}

template<class T> int Save_binArray(char *filename, list<vector<T> > &data) {
    FILE *fp;   unsigned int numLines = data.size(), numCols;
    typename list<vector<T> >::iterator iter = data.begin();
    if(numLines==0) return -1; else numCols = iter->size();
    T *streamData = (T *)malloc(sizeof(T)*numLines*numCols);
    unsigned int i,j;
    
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    fwrite(&numLines,sizeof(int),1,fp);
    fwrite(&numCols,sizeof(int),1,fp);

	for(i=0; iter!=data.end(); iter++,i++) 
		for(j=0;j<numCols;j++) streamData[i*numCols+j]=(*iter)[j];

	fwrite(streamData,sizeof(T),numLines*numCols,fp);
	free(streamData);

    fclose(fp);
    return 1;
}

template<class T> int Load_binArray(char *filename, vector<vector<T> > &data) {
    FILE *fp;   unsigned int numLines=0, numCols=0;
    unsigned int i,j; int n;
    
    fp = fopen(filename,"r");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

	n=fread(&numLines,sizeof(unsigned int),1,fp); if(numLines<=0 | n!=1) { cerr << "Invalid number of lines ("<<numLines<<")\n" ;return -1; }
	n=fread(&numCols,sizeof(unsigned int),1,fp);  if(numCols<=0 | n!=1)  { cerr << "Invalid number of columns ("<<numCols<<")\n" ;return -1; }
    
    T *streamData = (T *)malloc(sizeof(T)*numLines*numCols);
	n=fread(streamData,sizeof(T),numLines*numCols,fp); 
    fclose(fp);
	if(n!=(int)(numLines*numCols))  { cerr << "Error reading binary array: not enough elements in the file.\n"; free(streamData); return -1; }

	data.resize(numLines);   unsigned int streamIdx=0;
	for(i=0;i<numLines;i++) { data[i].resize(numCols); 
		for(j=0;j<numCols;j++) data[i][j] = streamData[streamIdx++];
	}

	free(streamData);
    return 1;
}

template<class T,class T2> int getIndexData(vector<vector<T> > &data, unsigned int **index, T **dataArray) {
	unsigned int numEntries1 = data.size(), numEntries2=0;
	(*index) = (unsigned int*)malloc(sizeof(unsigned int)*numEntries1);
	for(unsigned int i=0;i<numEntries1;i++) { (*index)[i]=data[i].size(); numEntries2+=(*index)[i]; }
	(*dataArray) = (T*)malloc(sizeof(T)*numEntries2);

	int dataArrayIdx=0;
	for(unsigned int i=0;i<numEntries1;i++) 
		for(unsigned int j=0;j<data[i].size();j++) 
			(*dataArray)[dataArrayIdx++] = data[i][j];
	return numEntries2;
}

template<class T,class T2> int getIndexData(vector<list<T> > &data, unsigned int **index, T **dataArray) {
	unsigned int numEntries1 = data.size(), numEntries2=0;
	(*index) = (unsigned int*)malloc(sizeof(unsigned int)*numEntries1);
	for(unsigned int i=0;i<numEntries1;i++) { (*index)[i]=data[i].size(); numEntries2+=(*index)[i]; }
	(*dataArray) = (T*)malloc(sizeof(T)*numEntries2);

	int dataArrayIdx=0;   T2 iter;
	for(unsigned int i=0;i<numEntries1;i++) 
		for(iter=data[i].begin(); iter!=data[i].end(); iter++)
			(*dataArray)[dataArrayIdx++] = *iter;
	return numEntries2;
}

template<class T,class T2,class T3> int getIndexData2(vector<T2> &data, unsigned int **index, T **dataArray) {
	unsigned int numEntries1 = data.size(), numEntries2=0;
	(*index) = (unsigned int*)malloc(sizeof(unsigned int)*numEntries1);
	for(unsigned int i=0;i<numEntries1;i++) { (*index)[i]=data[i].size(); numEntries2+=(*index)[i]; }
	(*dataArray) = (T*)malloc(sizeof(T)*numEntries2);

	int dataArrayIdx=0;   T3 iter;
	for(unsigned int i=0;i<numEntries1;i++) 
		for(iter=data[i].begin(); iter!=data[i].end(); iter++)
			(*dataArray)[dataArrayIdx++] = *iter;
	return numEntries2;
}

template<class T1,class T2,class T3> int Load_binListArray(char *filename, vector<T2> &data) {
    FILE *fp;   unsigned int numLists=0, numElems=0;
	unsigned int *index;
    unsigned int i,streamIdx; int n;
    
    fp = fopen(filename,"r");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    n=fread(&numLists,sizeof(unsigned int),1,fp); if(numLists<=0 | n!=1) { cerr << "Invalid number of lists ("<<numLists<<")\n" ;return -1; }
	index = (unsigned int*)malloc(sizeof(unsigned int)*numLists);
	n=fread(index,sizeof(unsigned int),numLists,fp);  if(n!=numLists)  { cerr << "Invalid index!\n" ; free(index); return -1; }
	numElems=0; for(i=0;i<numLists;i++) numElems+=index[i];
	
    T1 *streamData = (T1 *)malloc(sizeof(T1)*numElems);
	n=fread(streamData,sizeof(T1),numElems,fp);    fclose(fp);
	if(n!=(int)numElems)  { cerr << "Error reading binary list array: not enough elements in the file.\n"; free(streamData); free(index); return -1; }
	
	data.resize(numLists);   T3 iter;    streamIdx=0;
	for(i=0; i<numLists; i++){ data[i].resize(index[i]);
		for(iter=data[i].begin(); iter!=data[i].end(); iter++) (*iter) = streamData[streamIdx++];
	}
	
	free(index);    free(streamData);
    return numLists;
}

template<class T1,class T2,class T3> int Save_binListArray(char *filename, vector<T2> &data) {
	unsigned int numEntries1 = data.size(), numEntries2=0;
	unsigned int *index = (unsigned int*)malloc(sizeof(unsigned int)*numEntries1);
	for(unsigned int i=0;i<numEntries1;i++) { index[i]=data[i].size(); numEntries2+=index[i]; }
	T1 *streamData = (T1*)malloc(sizeof(T1)*numEntries2);

	int dataArrayIdx=0;   T3 iter;
	for(unsigned int i=0;i<numEntries1;i++) 
		for(iter=data[i].begin(); iter!=data[i].end(); iter++)
			streamData[dataArrayIdx++] = *iter;

	FILE *fp = fopen(filename,"w"); 
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    fwrite(&numEntries1,sizeof(int),1,fp); 
    fwrite(index,sizeof(int),numEntries1,fp);
    fwrite(streamData,sizeof(T1),numEntries2,fp);

	fclose(fp);	
	free(index);   free(streamData);
	return 1;
}

template<class T1,class T2,class T3> int Save_binListArray(char *filename, list<T2> &data) {
	unsigned int numEntries1 = data.size(), numEntries2=0;
	unsigned int *index = (unsigned int*)malloc(sizeof(unsigned int)*numEntries1);
	typename list<T2>::iterator dataIter;
	unsigned int indexIdx=0;
	for(dataIter = data.begin(); dataIter != data.end(); dataIter++, indexIdx++) { index[indexIdx]=dataIter->size(); numEntries2+=index[indexIdx]; }
	T1 *streamData = (T1*)malloc(sizeof(T1)*numEntries2);

	int dataArrayIdx=0;   T3 iter;
	for(dataIter = data.begin(); dataIter != data.end(); dataIter++) 
		for(iter=dataIter->begin(); iter!=dataIter->end(); iter++)
			streamData[dataArrayIdx++] = *iter;

	FILE *fp = fopen(filename,"w"); 
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    fwrite(&numEntries1,sizeof(int),1,fp);
    fwrite(index,sizeof(int),numEntries1,fp);
    fwrite(streamData,sizeof(T1),numEntries2,fp);

	fclose(fp);	
	free(index);   free(streamData);
	return 1;
}

short BufferedLineReader::Load(char *filename) {
	reset();
    FILE *input = fopen(filename,"r");
    if(!input or ferror(input)) { cerr << "ERROR opening " << filename << "!\n"; return -1; }
    if(fseek(input,0,SEEK_END)) { cerr << "ERROR finding the end of the file (fseek)!\n"; return -1; }
	int numBytes = ftell(input); if(numBytes<=0) { cerr << "ERROR file size = "<<numBytes<<"?\n"; return -1; }
    if(fseek(input,0,SEEK_SET)) { cerr << "ERROR finding the start of the file (fseek)!\n"; return -1; }
	numBytes++;

	lines = (char *)malloc(numBytes); lines[numBytes-1]=0;
	int readBytes=fread(lines,1,numBytes-1,input);
	if(readBytes!=numBytes-1) { cerr << "ERROR reading "<<filename<<" - only got "<<readBytes<<" bytes instead of "<<numBytes-1<<"?\n"; return -1; }
	
	unsigned int numLines=0, pivot;
	for(pivot=0; pivot<(unsigned int)numBytes; pivot++) 
		if(lines[pivot]=='\n') numLines++;
		else if(lines[pivot]=='\r') lines[pivot]=0;

	linesIdx.resize(numLines+1);
	linesIdx[0]=0;   numLines=1;
	for(pivot=0; pivot<(unsigned int)numBytes; pivot++) 
		if(lines[pivot]=='\n') { linesIdx[numLines++]=pivot+1; lines[pivot]=0; }
    
    fclose(input);
    return 1;
}
