#include "mzxml.h"
#include <iostream>
#include <cstring>
#include <cmath>
#include "base64.h"
#include <netinet/in.h>

#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>
#include <xercesc/sax2/DefaultHandler.hpp>
#include <xercesc/util/XMLString.hpp>

short mzxmlSAX2Handler::STATE_SKIPPING=0, 
      mzxmlSAX2Handler::STATE_SCAN=1, 
      mzxmlSAX2Handler::STATE_PRECURSORMZ=2, 
      mzxmlSAX2Handler::STATE_PEAKS=3;

unsigned int LoadMzxml(char *filename, SpecSet &specs, vector<short> *msLevels, short minMsLevel) {
	specs.resize(0);

       try {
            XMLPlatformUtils::Initialize();
        }
        catch (const XMLException& toCatch) {
            char* message = XMLString::transcode(toCatch.getMessage());
            cout << "Error during initialization! :\n";
            cout << "Exception message is: \n"
                 << message << "\n";
            XMLString::release(&message);
            return 1;
        }

        char* xmlFile = filename;
        SAX2XMLReader* parser = XMLReaderFactory::createXMLReader();
        parser->setFeature(XMLUni::fgSAX2CoreValidation, false);   // optional
        parser->setFeature(XMLUni::fgSAX2CoreNameSpaces, true);   // optional
        parser->setFeature(XMLUni::fgXercesSchema, false);   // optional

        mzxmlSAX2Handler* defaultHandler = new mzxmlSAX2Handler(specs,msLevels,minMsLevel);
        parser->setContentHandler(defaultHandler);
        parser->setErrorHandler(defaultHandler);

        try {
            parser->parse(xmlFile);
        }
        catch (const XMLException& toCatch) {
            char* message = XMLString::transcode(toCatch.getMessage());
            cout << "Exception message is: \n"
                 << message << "\n";
            XMLString::release(&message);
            return 0;
        }
        catch (const SAXParseException& toCatch) {
            char* message = XMLString::transcode(toCatch.getMessage());
            cout << "Exception message is: \n"
                 << message << "\n";
            XMLString::release(&message);
            return 0;
        }
        catch (...) {
            cout << "Unexpected Exception \n" ;
            return 0;
        }

	if(defaultHandler->numSpecs<specs.size()) {
		specs.resize(defaultHandler->numSpecs);
		if(msLevels) msLevels->resize(defaultHandler->numSpecs);
	}

	delete parser;
        delete defaultHandler;

	return specs.size();
}

void mzxmlSAX2Handler::startElement(const   XMLCh* const    uri,
                            const   XMLCh* const    localname,
                            const   XMLCh* const    qname,
                            const   Attributes&     attrs) {
    char* name = XMLString::transcode(localname);

	char *str;   XMLCh *xmlStr;   int intValue;
    if(!strcmp(name,"msRun")) {
    	xmlStr = XMLString::transcode("scanCount");    str = XMLString::transcode(attrs.getValue(xmlStr));
		intValue = atoi(str);
		if(intValue>=0) {
			specs->resize(intValue);   
			if(msLevels) msLevels->resize(intValue);
		}
    	XMLString::release(&str);    XMLString::release(&xmlStr);
    }

    if(!strcmp(name,"scan")) {
    	if(specs->size()==0) { cerr<<"ERROR: Can't read scans before knowing how many scans in the file (from msRun.scanCount)!\n"; exit(-1); }
    	if(numSpecs>=specs->size()) { cerr<<"ERROR: Can't read more scans than indicated in msRun.scanCount!\n"; exit(-1); }

    	xmlStr = XMLString::transcode("msLevel");    str = XMLString::transcode(attrs.getValue(xmlStr));
		intValue = atoi(str);
		if(intValue>=minMsLevel) {
			curSpec.push_front(numSpecs++);
	    	if(msLevels) (*msLevels).at(curSpec.front()) = max((unsigned int)intValue,(unsigned int)curSpec.size());
	    	state = STATE_SCAN;
		} else state = STATE_SKIPPING;
    	XMLString::release(&str);    XMLString::release(&xmlStr);

		if(state == STATE_SCAN) {
	    	xmlStr = XMLString::transcode("num");   str = XMLString::transcode(attrs.getValue(xmlStr));
			intValue = atoi(str);
			(*specs)[curSpec.front()].scan = intValue;
	    	XMLString::release(&str);    XMLString::release(&xmlStr);

	    	xmlStr = XMLString::transcode("peaksCount");   str = XMLString::transcode(attrs.getValue(xmlStr));
			intValue = atoi(str);
			(*specs)[curSpec.front()].resize(intValue);
	    	XMLString::release(&str);    XMLString::release(&xmlStr);
		}
    }
    if(state==STATE_SKIPPING) { XMLString::release(&name); return; }
    if(!strcmp(name,"precursorMz")) state = STATE_PRECURSORMZ;
    if(!strcmp(name,"peaks")) state = STATE_PEAKS;
	XMLString::release(&name);
}

void mzxmlSAX2Handler::characters(const XMLCh* const chars, const unsigned int length) {
	char *str = XMLString::transcode(chars);
	if(str!=(char *)0 and str[0]==(char)0 or str[0]==(char)13 or str[0]==(char)10) { XMLString::release(&str); return; }
	
	Spectrum *spec = &(*specs)[curSpec.front()];
    if(state == STATE_PRECURSORMZ) spec->parentMass = (float)atof(str);

    float fValue;
    if(state == STATE_PEAKS) {
		char *decoded = (char *)malloc( spec->size() * 8 + 8 );  // +8 to avoid buffer overruns
		if(!decoded) {
		  cerr<<"ERROR: Not enough memory to decode peak list for scan "<<spec->scan<<" - needed "<<spec->size() * 8 + 8<<" bytes.\n";
		  exit(-1);
		}
		b64_decode_mio( decoded , str );
		
		unsigned int valIdx=0;
		for(unsigned int peakIdx=0; peakIdx<spec->peakList.size(); peakIdx++) {
			for(unsigned int colIdx=0; colIdx<2; colIdx++, valIdx++) {
			  ((u_int32_t *)&fValue)[0] = (u_int32_t) ntohl( (u_int32_t) ((u_int32_t *)decoded)[valIdx]);
			  spec->peakList.at(peakIdx)[colIdx] = fValue;
			}
		}
		free( decoded );
    }

	XMLString::release(&str);
}

void mzxmlSAX2Handler::endElement(const XMLCh* const uri, const XMLCh* const localname,
    const XMLCh* const qname) {

    char* name = XMLString::transcode(localname);
    if(!strcmp(name,"scan") and state!=STATE_SKIPPING) {
    	curSpec.pop_front();
    	if(curSpec.empty()) state=STATE_SKIPPING; else state=STATE_SCAN;
    }
    XMLString::release(&name);
};

void mzxmlSAX2Handler::endDocument() {}

void mzxmlSAX2Handler::fatalError(const SAXParseException& exception) {
    char* message = XMLString::transcode(exception.getMessage());
    cout << "Fatal Error: " << message
         << " at line: " << exception.getLineNumber()
         << endl;
}
