#ifndef __BASICDATASTRUCTS_H__
#define __BASICDATASTRUCTS_H__

#include "Config.h"
#include "includes.h"



class Peptide {
public:
	Peptide() : mass(0), n_term(N_TERM), c_term(C_TERM) {};

	void clear() { mass=0; n_term=N_TERM; c_term=C_TERM; amino_acids.clear(); gaps.clear(); }

	bool operator== (const Peptide& other) const
	{
		if (amino_acids.size() != other.amino_acids.size())
			return false;
		int i;
		for (i=0; i<amino_acids.size(); i++)
			if (amino_acids[i] != other.amino_acids[i])
				return false;
		return true;
	}

	mass_t get_mass() const { return mass; } // mass of aas + terminals, doesn't include 19
	int get_n_term() const { return n_term; }
	int get_c_term() const { return c_term; }
	void set_n_term(int aa) { n_term = aa; }
	void set_c_term(int aa) { c_term = aa; }


	int get_length() const { return amino_acids.size(); } // gaps count as amino acids
	int get_num_aas() const { return amino_acids.size() - gaps.size(); }
	int get_num_gaps() const { return gaps.size(); }

	const vector<int>& get_amino_acids() const { return amino_acids; }

	void generate_random_peptide(const Config *config, int peptide_length);

	/*********************************************************************
	Returns the global edit distance between two peptides.
	**********************************************************************/
	float peptide_edit_distance(Config *config, Peptide& other) const;

	// changes the amino acids I->L
	// an Q->K if not at terminal and tolerance > 0.1
	void convert_ILQK(const Config *config);

	// changes all the amino acids to their original form (without PTMs)
	void convert_to_org(const Config *config);

	void convert_IL();

	void reverse();

	string as_string(const Config* config) const;

	void parse_from_string(const Config* config, const string& str);

	void set_peptide(vector<int>& aas, vector<mass_t>& gaps, mass_t mass, 
					 int n_term_aa = N_TERM, int c_term_aa = C_TERM);

	void set_peptide_aas(const vector<int>& aas) { amino_acids = aas; }

	void calc_mass(const Config *config);

	void calc_expected_breakage_masses(Config *config, vector<mass_t>& break_masses) const;

private:

	mass_t mass; // mass of aas + terminals, doesn't include 19
	int n_term; // N_TERM or a modification (not the aa N-terminal to the peptide)
	int c_term; // C_TERM or a modification (not the aa C-terminal to the peptide)

	vector<int>    amino_acids;
	vector<mass_t> gaps;
};



struct PeakRange {
	PeakRange() : num_peaks(0), low_idx(-1), high_idx(-1) {};
	int num_peaks;
	int low_idx;
	int high_idx;
};
typedef struct PeakRange PeakRange;




struct BreakageFragment {
	BreakageFragment() : frag_type_idx(-1), peak_idx(-1), peak_level(-1), 
						 is_strong_fragment(0), mass(-1), expected_mass(-1), intensity(0) {};

	int frag_type_idx;    // idx in theFragmenTypeSet
	int peak_idx;

	int peak_level;          // used in models that have discrete rank/intensity levels
	int is_strong_fragment;  // 1 if this is a strong fragment according to the region
							 // in which the breakage occurs, 0 otherwise

	mass_t mass;             // the mass of the peak in the spectrum, -1 if no peak
	mass_t expected_mass; 
	intensity_t intensity;
};


struct Breakage {
	Breakage() : region_idx(-1), mass(-1), 
			     total_intensity(0),  num_frags_detected(0),
				 parent_charge(0),    parent_size_idx(0), score(NEG_INF) {};


	// returns the position in the fragments vector of the given frag_type_idx
	// returns -1 if this frag_type_idx is not found
	int get_position_of_frag_idx(int frag_type_idx) const
	{
		int i;
		for (i=0; i<fragments.size(); i++)
			if (fragments[i].frag_type_idx == frag_type_idx)
				return i;
		return -1;
	}


	bool are_all_frag_types_visible() const
	{
		return (frag_type_idxs_not_visible.size() == 0);
	}

	bool is_frag_type_visible(int f_idx) const
	{
		int i;
		if (frag_type_idxs_not_visible.size() == 0)
			return true;

		for (i=0; i<frag_type_idxs_not_visible.size(); i++)
			if (frag_type_idxs_not_visible[i] == f_idx)
				return false;
		return true;
	}


	void add_fragment(BreakageFragment& brf)
	{
		fragments.push_back(brf);
		total_intensity += brf.intensity;
		num_frags_detected++;
	}


	void remove_fragment(int frag_type_idx)
	{
		const int pos = get_position_of_frag_idx(frag_type_idx);
		if (pos<0)
		{
			cout << "Error: removing fragment type that isn't there:" << frag_type_idx << endl;
			exit(1);
		}

		if (fragments[pos].mass>0)
		{
			total_intensity -= fragments[pos].intensity;
			num_frags_detected--;

			vector<BreakageFragment> new_fragments;
			int j;
			for (j=0; j<fragments.size(); j++)
			{
				if (j==pos)
					continue;
				new_fragments.push_back(fragments[j]);
			}
			fragments = new_fragments;
		}
	}

	void clear()
	{
		region_idx=-1;
		mass=-1;
		total_intensity=0;
		num_frags_detected=0;
		score=0;
		frag_type_idxs_not_visible.clear();
		fragments.clear();
	}

	
	void print() const;

	void print(Config *config, ostream& os = cout) const;
	void print_fragments(Config *config, ostream& os = cout) const;


	int region_idx;  // model region
	mass_t mass;
	intensity_t total_intensity;
	int num_frags_detected;

	int parent_charge;   // for model_size
	int parent_size_idx; // for model_size

	score_t score;

	vector<int> frag_type_idxs_not_visible; // idxs of all frag types of the breakages region
											// that are not visible
	vector<BreakageFragment> fragments;
};


typedef enum { NODE_REG, NODE_N_TERM, NODE_C_TERM, NODE_DIGEST} NODE_TYPES;

typedef enum { EDGE_REG, EDGE_FROM_N_TERM, EDGE_TO_C_TERM, EDGE_DIGEST } EDGE_TYPES;

#define MAX_EDGE_SIZE 3

struct Node {
	Node() : mass(-1.0), score(0), prefix_penalty(0), 
		     suffix_penalty(0), log_rank(999999), idx_max_in_score_node(-1), 
			 idx_max_out_score_node(-1), type(-99), active(1),source_frag_type_idx(-1) {};

	bool operator< (const Node& other) const
	{
		return mass<other.mass;
	}

	void remove_in_edge_idx(int idx)
	{
		int i;
		for (i=0; i<in_edge_idxs.size(); i++)
			if (in_edge_idxs[i] == idx)
				break;
		if (i== in_edge_idxs.size())
			return;

		in_edge_idxs[i]=in_edge_idxs[in_edge_idxs.size()-1];
		in_edge_idxs.pop_back();
	}

	void remove_out_edge_idx(int idx)
	{
		int i;
		for (i=0; i<out_edge_idxs.size(); i++)
			if (out_edge_idxs[i] == idx)
				break;
		if (i== out_edge_idxs.size())
			return;

		out_edge_idxs[i]=out_edge_idxs[out_edge_idxs.size()-1];
		out_edge_idxs.pop_back();
	}


	void print(Config *config, ostream& os = cout) const;

	mass_t mass;
	score_t score;
	score_t prefix_penalty, suffix_penalty; // penalties that should be given if the peptide
											// starts or ends with this node
	float log_rank;
	int   idx_max_in_score_node;  // the idx of the highest scoring previous node that connects to this node
	int   idx_max_out_score_node; // the idx of the highest scoring next node theat connects to this node
	int type;
	int active;                             // should this node be used
	int source_frag_type_idx;  // the fragment according to which the node was created
	Breakage breakage;
	vector<int> in_edge_idxs, out_edge_idxs;
};



struct Edge {
	Edge() : n_idx(-1), c_idx(-1), n_break(NULL), c_break(NULL), type(0), gap_size(0), 
			 score(NEG_INF), exp_edge_mass(0), abs_edge_diff(0), num_aa(0) {};

	void print(Config *config, ostream& os = cout) const;

	void generate_all_edge_permutations(vector<Edge>& perms) const;

	bool operator== (const Edge& other) const
	{
		if (n_idx != other.n_idx && c_idx != other.c_idx)
			return false;

		if (num_aa != other.num_aa)
			return false;

		int i;
		for (i=0; i<num_aa; i++)
			if (amino_acids[i] != other.amino_acids[i])
				return false;
		return true;
	}

	bool operator< (const Edge& other) const
	{
		return score>other.score;
	}

	int n_idx, c_idx;
	Breakage *n_break, *c_break;
	int type;
	mass_t gap_size; // only used if edge represents a gap
	score_t score;
	
	mass_t exp_edge_mass; // the sum of the masses of the amino acids
	mass_t abs_edge_diff; // the difference between the expected edge mass and the actual
						  // edge mass (taking the minimum between peak diffs and the node mass diff)
	int num_aa;
	int amino_acids[MAX_EDGE_SIZE];	
};


// Contains all the amino acid combos of the same length between two nodes
struct MultiEdge {
	MultiEdge() :  num_aa(0), type(0), max_variant_score(NEG_INF),  n_idx(-1), c_idx(-1), 
				  n_break(NULL), c_break(NULL), ind_edge_overlaps(false) {};

	int get_num_variants() const { return variant_ptrs.size(); }

	bool has_variant(int num_aa, const int *aas) const
	{
		int v;
		for (v=0; v<variant_ptrs.size(); v++)
		{
			if (*variant_ptrs[v] == num_aa )
			{
				int *var_aas = variant_ptrs[v]+1;
				int j;
				for (j=0; j<num_aa; j++)
					if (aas[j] !=  var_aas[j])
						break;

				if (j==num_aa)
					return true;
			}
		}
		return false;
	}

	int get_variant_idx(int num_aa, int *aas) const
	{
		int v;
		for (v=0; v<variant_ptrs.size(); v++)
		{
			if (*variant_ptrs[v] == num_aa )
			{
				int *var_aas = variant_ptrs[v]+1;
				int j;
				for (j=0; j<num_aa; j++)
					if (aas[j] !=  var_aas[j])
						break;

				if (j==num_aa)
					return v;
			}
		}
		return -1;
	}

	int num_aa;
	int type;  // 
	
	score_t max_variant_score;

	int n_idx, c_idx;
	Breakage *n_break, *c_break;

	bool ind_edge_overlaps;  // if true there is a subpath with shorter edges that 
							 // has the same amino acids as this edge

	vector<int *>     variant_ptrs;
	vector<score_t>   variant_scores; // these are the deltas to the edge scores 
	vector<float>     variant_probs;
};


struct PathPos {
	PathPos() : mass(-1), breakage(NULL), node_idx(-1), edge_idx(-1), variant_ptr(NULL),
			aa(-1), node_score(0), edge_varaint_score(0), edge_prob(-1) {};

	mass_t mass;

	Breakage *breakage; 

	int node_idx; // the index of the node N-terminal to the aa (-1 means no node is used
				  // as in the case where we are in the middle of a multiple aa edge

	int edge_idx; // the idx of the edge leaving the node (-1 if there is no edge)

	int* variant_ptr;

	int aa; // -1 means the is no aa (should only happen for the last PathPos in the path

	score_t node_score; // this is derived from the edge score + adjustments due to the 
						// specific amino combos in this edge

	score_t edge_varaint_score;

	float	edge_prob;
};






struct AA_combo {
	AA_combo() : total_mass(0), num_aa(0) {};

	bool operator< (const AA_combo& other) const
	{
		return (total_mass<other.total_mass);
	}

	void print(ostream& os = cout) const
	{
		int i;
		os << setprecision(4) << total_mass << " ";
		for (i=0; i<num_aa; i++)
			os << amino_acids[i] << " ";
		os << endl;
	}

	mass_t total_mass;
	int num_aa;
	int num_variants;
	int variant_start_idx; // the index in the variant_vector where this edges
						   // permutations are listed, the permutations are all concatanted
						   // in the same vector of int, first comes the number of amino acids
						   // then the amino acids themselves are listed
	int amino_acids[MAX_EDGE_SIZE];
};



	


#endif




