#include "DeNovoSolutions.h"





bool comp_SeqPath_sort_key(const SeqPath& a, const SeqPath& b)
{
	return (a.sort_key>b.sort_key);
}


/**************************************************************************
	Adds a new seqpath to the existing vector.
	If vector is at maximal size, can replace the lowest confidence path.
	If a path exists with the same seq and n,c masses but lower confidence
	score, replaces it.
***************************************************************************/
void SeqPathHeap::add_path(SeqPath& new_path)
{
	if (new_path.sort_key<min_value &&
		paths.size() >= max_size)
		return;

	int i;
	for (i=0; i<paths.size(); i++)
	{
		if ( fabs(paths[i].n_term_mass-new_path.n_term_mass) < tolerance &&
			! strcmp(paths[i].seq_str.c_str(),new_path.seq_str.c_str()) )
		{
			if (new_path.sort_key>paths[i].sort_key)
				paths[i]=new_path;

			return;
		}
	}

	// add path
	if (paths.size()<max_size)
	{
		paths.push_back(new_path);
		if (new_path.sort_key<min_value)
		{
			min_value = new_path.sort_key;
			min_idx = paths.size()-1;
		}
	}
	else
	{
		paths[min_idx]=new_path;
		int i;
		min_value = 999999;
		for (i=0; i<paths.size(); i++)
		{
			if (paths[i].sort_key<min_value)
			{
				min_value = paths[i].sort_key;
				min_idx =i;
			}
		}
	}
}



void print_prm_graph_scores(Model *model, Spectrum *spec, 
					 mass_t pm_with_19, int charge)
{
	PrmGraph prm;
	Config *config= model->get_config();

	if (charge> model->get_max_score_model_charge() || charge<1)
	{
		cout << "Charge " << charge << " not supported..." << endl;
		return;
	}


	

	spec->set_charge(charge);

	model->init_model_for_scoring_spectrum(spec);

	mass_t opt_pm_with_19 = prm.find_optimal_pm_with_19_for_graph(model,spec,pm_with_19,charge);

	spec->set_corrected_pm_with_19(opt_pm_with_19);

	spec->print_expected_by();

	prm.create_graph_from_spectrum(model,spec,opt_pm_with_19);

	model->score_graph_edges(prm);

	prm.add_edge_scores_to_nodes();

//	prm.print(cout,true);

	prm.print_only_scores();

	cout << endl;
}


/**************************************************************************
	Wrapper funciton that generates the desired solutions.
	Combines both local and global solutions (similar to the PepNovoTag
	and LocalTag solutions).
***************************************************************************/
void generate_denovo_solutions(Model *model, 
							   Spectrum *spec,
							   bool denovo_mode,
							   mass_t pm_with_19,
							   int charge,
							   int num_sols, 
							   int min_length, 
							   int max_length,
							   vector<SeqPath>& solutions)
{
	PrmGraph prm;
	DeNovoDp dndp;
	Config *config= model->get_config();
	
	if (charge>3 || charge<1)
	{
		solutions.clear();
		return;
	}

	// a hack to give de novo sequences more motivation to go to the end
	float old_term_score = config->get_terminal_score();
	if (denovo_mode)
		config->set_terminal_score(old_term_score*2.0);

	spec->set_charge(charge);
	model->init_model_for_scoring_spectrum(spec);
	prm.create_graph_from_spectrum(model,spec,pm_with_19);
	model->score_graph_edges(prm);

	dndp.fill_dp_table(&prm,25);

	config->set_terminal_score(old_term_score);


	// if we are in tag mode (length less than 10)
	// generate seqeunces and sort according to probabilities
	// otherwise generate longer sequences and sort according to score

	if (! denovo_mode)
	{
		SeqPathHeap local_path_heap, denovo_path_heap, merged_prob_heap;
		vector<MultiPath> multi_paths;

		merged_prob_heap.init(num_sols,config->get_tolerance());

		//	generate global tags from parsing de novo paths
		if (min_length<=6)
		{
			denovo_path_heap.init(num_sols*2+20,config->get_tolerance());

			const int max_parsed_length = (max_length<6 ? max_length : 6);;


			multi_paths.clear();
			dndp.get_top_scoring_antisymetric_paths_with_length_limits(multi_paths, 3,
				min_length , 15, 25);

			vector<SeqPath> seq_paths;
			prm.expand_all_multi_paths(multi_paths,seq_paths);

			// parse seq paths

			int i;
			for (i=0; i<seq_paths.size(); i++)
			{
				vector<SeqPath> parsed_paths;
				seq_paths[i].parse_path_to_smaller_ones(config,min_length,max_parsed_length,parsed_paths);

				int j;
				for (j=0; j<parsed_paths.size(); j++)
				{
					if (parsed_paths[j].path_score<-5)
						continue;

					parsed_paths[j].make_seq_str(config);
					parsed_paths[j].sort_key = parsed_paths[j].path_score;
					denovo_path_heap.add_path(parsed_paths[j]);
				}
			}

			sort(denovo_path_heap.paths.begin(),denovo_path_heap.paths.end());

			// compute probabilities
			if (denovo_path_heap.paths.size()>0)
			{
				score_t top_score = denovo_path_heap.paths[0].path_score;

				for (i=0; i<denovo_path_heap.paths.size(); i++)
				{
					SeqPath& path = denovo_path_heap.paths[i];

					if (path.path_score<-5)
						continue;

					path.pm_with_19 = pm_with_19;

					path.charge     = charge;

					path.seq_prob = model->calc_seq_prob(path,charge,prm,path.multi_path_rank,i,top_score,true);
					path.sort_key = path.seq_prob;

					merged_prob_heap.add_path(path);
				}
			}
		}

		// add the local tags
		local_path_heap.init(num_sols*2+20,config->get_tolerance());
		multi_paths.clear();
		dndp.get_top_scoring_antisymetric_paths_with_length_limits(multi_paths, num_sols,
				min_length , max_length, 25);
		
		vector<SeqPath> seq_paths;
		prm.expand_all_multi_paths(multi_paths,seq_paths);
		int i;
		for (i=0; i<seq_paths.size(); i++)
		{
			if (seq_paths[i].get_num_aa()<min_length || seq_paths[i].get_num_aa()>max_length)
				continue;

			seq_paths[i].sort_key = seq_paths[i].path_score;
			local_path_heap.add_path(seq_paths[i]);
		}
		
		sort(local_path_heap.paths.begin(),local_path_heap.paths.end());

		// compute local probabilities
		if (local_path_heap.paths.size()>0)
		{
			score_t top_score = local_path_heap.paths[0].path_score;

			for (i=0; i<local_path_heap.paths.size(); i++)
			{
				SeqPath& path = local_path_heap.paths[i];

				path.seq_prob = model->calc_seq_prob(path,charge,prm,path.multi_path_rank,i,
									top_score,false);

				path.sort_key = path.seq_prob;

				path.pm_with_19 = pm_with_19;

				path.charge     = charge;

				merged_prob_heap.add_path(path);
			}
		}

		sort(merged_prob_heap.paths.begin(),merged_prob_heap.paths.end(),comp_SeqPath_sort_key);

		solutions = merged_prob_heap.paths;

	}

	else  // DE NOVO MODE

	{
		SeqPathHeap denovo_path_heap;
		vector<MultiPath> multi_paths;

		//	generate global tags from parsing de novo paths
		denovo_path_heap.init(num_sols,config->get_tolerance());

		dndp.get_top_scoring_antisymetric_paths_with_length_limits(multi_paths, num_sols,
				min_length , max_length, 25);

		vector<SeqPath> seq_paths;
		prm.expand_all_multi_paths(multi_paths,seq_paths);

		// parse seq paths

		int i;
		for (i=0; i<seq_paths.size(); i++)
		{
			SeqPath& path = seq_paths[i];

			path.seq_prob = seq_paths[i].path_score;

			path.sort_key = path.seq_prob;

			path.pm_with_19 = pm_with_19;

			path.charge     = charge;

			denovo_path_heap.add_path(seq_paths[i]);
		}

		sort(denovo_path_heap.paths.begin(),denovo_path_heap.paths.end());

		solutions = denovo_path_heap.paths;
	}
}


/***************************************************************************
	Wrapper function that generates several solutions according to different 
	precursor masses.
****************************************************************************/
void generate_denovo_solutions_from_several_pms(Model *model, 
												Spectrum *spec,
												bool denovo_mode,
												int num_sols, 
												int min_length, 
												int max_length,
												vector<mass_t>&  different_pms_with_19,
												vector<int>& charges,
												vector<SeqPath>& best_solutions)
{
	SeqPathHeap seq_heap;
	best_solutions.clear();

	seq_heap.init(num_sols, model->get_config()->get_tolerance());

	int i;
	for (i=0; i<different_pms_with_19.size(); i++)
	{
		vector<SeqPath> sols;
		generate_denovo_solutions(model,spec,denovo_mode,different_pms_with_19[i],charges[i],
								  num_sols,min_length,max_length,sols);

		int j;
		for (j=0; j<sols.size(); j++)
		{
			seq_heap.add_path(sols[j]);
		}
	}

	best_solutions = seq_heap.paths;

	sort(best_solutions.begin(),best_solutions.end(),comp_SeqPath_sort_key);
}



void output_denovo_solutions(SingleSpectrumFile *ssf, Config *config, ostream& out_stream,
							 const vector<SeqPath>& solutions)
{
	ssf->print_ssf_stats(config,out_stream);

	if (solutions.size() == 0)
	{
		out_stream << "No solutions found." << endl;
	}
	else 
	{
		out_stream << "#Index\tScore\tN-Gap\tC-Gap\t[M+H]\tCharge\tSequence" << endl;
		int i; 	
		for (i=0; i<solutions.size(); i++) 
		{
			mass_t c_gap=solutions[i].pm_with_19 - solutions[i].c_term_mass;
			if (c_gap<24.0)
				c_gap = 0;

			out_stream << setprecision(3) << fixed << i << "\t";
			out_stream << solutions[i].path_score << "\t";
			out_stream << solutions[i].n_term_mass << "\t";
			out_stream << c_gap << "\t";
			out_stream << solutions[i].pm_with_19 << "\t";
			out_stream << solutions[i].charge << "\t";
			out_stream << solutions[i].seq_str;
			out_stream << endl;
		}
	}
	out_stream << endl;
}



void output_tag_solutions(SingleSpectrumFile *ssf, Config *config, ostream& out_stream,
							 const vector<SeqPath>& solutions)
{
	ssf->print_ssf_stats(config,out_stream);

	if (solutions.size() == 0)
	{
		out_stream << "No solutions found." << endl;
	}
	else 
	{
		out_stream << "#Index\tProb\tScore\tN-Gap\tC-Gap\t[M+H]\tCharge\tSequence" << endl;
		int i; 	
		for (i=0; i<solutions.size(); i++) 
		{
			mass_t c_gap=solutions[i].pm_with_19 - solutions[i].c_term_mass;
			if (c_gap<24.0)
				c_gap = 0;

			out_stream << setprecision(3) << fixed << i << "\t";
			out_stream << solutions[i].seq_prob << "\t";
			out_stream << solutions[i].path_score << "\t";
			out_stream << solutions[i].n_term_mass << "\t";
			out_stream << c_gap << "\t";
			out_stream << solutions[i].pm_with_19 << "\t";
			out_stream << solutions[i].charge << "\t";
			out_stream << solutions[i].seq_str;
			out_stream << endl;
		}
	}
	out_stream << endl;
}


