#ifndef __DISCRETEPEAKMDODEL_H__
#define __DISCRETEPEAKMDODEL_H__

#include "Model.h"
#include "AnnotatedSpectrum.h"
#include "FileManagement.h"
#include "RegionalPepNovoModel.h"
#include "EdgeModel.h"
#include "TagModel.h"
#include "includes.h"

#define NUM_PREVIOUS_FRAG_VALS 4

struct Edge; // forward declr
// struct SeqPath;
class  PrmGraph;

class DiscretePeakModel : public Model {
public:

	virtual void write_model_specifics(ostream& os) const =0;

	virtual void read_model_specifics(istream& is) =0;

	void write_tables(ostream& os) const;

	void read_tables(ostream& os) const;

	void clone_charge_model(int source_charge, int target_charge);

	void init_score_model() { init_default_thresholds(); }

	virtual void init_default_thresholds() =0;

	// this function should always be run once before each new spectrum is 
	void init_model_for_scoring_spectrum(Spectrum *spec) =0;
	

	// prints joint scores of two top  fragments
	void print_joint_scores(ostream& os = cout) const;

	void print_report(int charge, int size, int region, ostream& os = cout) const;
	
	void print_frag_probs(const vector<int>& frag_type_idxs, 
		int charge, int size, int region, ostream& os) const;

	void print_all_table_names(ostream& os = cout) const;
	virtual void print_level_legend( ostream& os) const =0;

	// This is the interface of the model with the rest of the program
	// all peaks get their breakages set by the model
	virtual void set_breakage_peak_levels(Breakage *breakage) const =0;

	void score_breakage(Spectrum *spec, Breakage *breakage, bool verbose) const
	{
		set_breakage_peak_levels(breakage);

		score_t b_score = regional_models[breakage->parent_charge]
			[breakage->parent_size_idx][breakage->region_idx].calc_breakage_score(breakage,
			verbose, spec->get_config());

		breakage->score = b_score;

		add_breakage_iso_score(spec, breakage);
	}

	void score_edge_and_breakages(Spectrum *spec, Edge *edge) const
	{
		score_breakage(spec,edge->n_break,false);
		score_breakage(spec,edge->c_break,false);
		edge->score += edge->n_break->score + edge->c_break->score;
	}


	score_t get_prefix_score_penalty(Spectrum *spec) const
	{
		const int charge = spec->get_charge();
		const int size_idx = config.calc_peptide_size_idx(charge,spec->get_org_pm_with_19());
		const int region_idx = config.calc_region_idx(0,spec->get_org_pm_with_19());
		
		return  regional_models[charge][size_idx][region_idx].get_missed_cleavage_score();
	}

	score_t get_suffix_score_penalty(Spectrum *spec) const
	{
		const int charge = spec->get_charge();
		const int size_idx = config.calc_peptide_size_idx(charge,spec->get_org_pm_with_19());
		const int region_idx = config.calc_region_idx(spec->get_org_pm_with_19(),spec->get_org_pm_with_19());	

		return  regional_models[charge][size_idx][region_idx].get_missed_cleavage_score();
	}

	score_t get_missing_breakage_score(int charge, int size_idx, int region_idx) const
	{
		return  regional_models[charge][size_idx][region_idx].get_missed_cleavage_score();
	}

	
	void score_graph_edges(PrmGraph& prm) const
	{
		edge_model.score_graph_edges(prm);
	}

	float calc_seq_prob(const SeqPath& path, int charge, const PrmGraph& prm,
		  int denovo_rank, int score_rank, score_t top_score, bool from_denovo) const
	{
		return tag_model.calc_seq_prob(path,charge,prm,denovo_rank,
			score_rank,top_score,from_denovo);
	}

	int get_max_score_model_charge() const
	{
		return max_score_model_charge;
	}


protected:
	
	int max_score_model_charge;

	int num_peak_levels;          // the number of levels + 1 (for level 0)

	vector<double> q_rand;           // rank (rank 0 = no peak);
	vector< vector<double> > q_frag; // rank (rank 0 = no peak), frag_type_idx

	Spectrum *current_spectrum; // the current spectrum being ranked

	vector<int> level_thresholds; // rank levels for rank models


	vector< vector< vector<RegionalPepNovoModel> > > regional_models; // charge, size, region

	EdgeModel edge_model;

	TagModel  tag_model;


	// calcs the score of participating fragments that have istopic peak scores
	void add_breakage_iso_score(Spectrum *spec, Breakage *breakage) const;


	// creates models allowing each fragment to have upto two parents
	// (chooses the ones that give the best difference in probability
	// in terms of DKL). Parent fragments must appear before the current
	// fragment in the order of the regional fragment set
	void learn_parent_dependencies(const FileManager& fm, int charge = 2);

	// learns the probabilities of randomly observing peaks
	void learn_q_rand(const FileManager& fm, int charge);


	// this function is the interface for querying on peak's intensity level,
	// rank etc. The peak's level might depend on the type of fragment that
	// is inteded to be used, so there is an option to add this to the query
	// (though systems like regular rank won't use the second parameter).
	virtual int get_peak_level(int p_idx, int f_idx = -1) const =0;


	virtual int get_lowest_level_in_spectrum(Spectrum *spec) const =0;



	void train_score_model(const char *name, const FileManager& fm, int charge=0);


	// reads all relevant info: 
	// thresholds and tables
	void read_score_model(istream& is);

	// writes all relevant info:
	// thresholds and tables
	void write_score_model(ostream& os) const;


	// converts all the probabilities to socres in the tables
	void convert_probs_to_scores();


};


#endif



