#include "Fragmentation.h"
#include "Config.h"


void FragmentType::read_fragment(istream& is)
{
	char line[128];
	is.getline(line,128);
	istringstream iss(line);
	char direction;
	iss >> direction;
	if (direction == 'p')
	{
		orientation = PREFIX;
	}
	else if (direction == 's')
	{
		orientation = SUFFIX;
	}
	else
	{
		cout << "Error reading fragment: " << line << endl;
		exit(1);
	}
	iss >> charge >> offset >> label;
}

void FragmentType::write_fragment(ostream& os) const
{
	if (orientation == PREFIX)
	{
		os << "p ";
	}
	else
		os << "s ";

	os << charge << " ";
	os << fixed << setprecision(5) << offset << " ";
	os << label << endl;
}

/********************************************************
Creates a string label from the fragment information.
*********************************************************/
void FragmentType::make_frag_label(mass_t tolerance)
{
	

	label = (orientation == PREFIX ? "p" : "s");

	if (charge>1)
	{
		ostringstream os;
		os << charge;
		label+= os.str();
	}

	if (offset != 0)
	{
		ostringstream os;
		os << fixed << setprecision(1) << offset;
		if (offset>0)
			label += "+";
		label += os.str();
	}

	//const mass_t frag_offsets[] ={ -26.9871,  1.0078, 18.0343, 44.9976, 19.0183,  1.9918};
//const char frag_letters[] = { 'a', 'b', 'c', 'x', 'y', 'z' };


//typedef enum { NO_LOSS, NH3_LOSS, H2O_LOSS, NH3NH3_LOSS, NH3H2O_LOSS, H2OH2O_LOSS,
//			  NUM_NEUTRAL_LOSSES} NEUTRAL_LOSSES;
//const mass_t neutral_loss_offsets[]={0.0, 17.0265, 18.0105, 34.053, 35.037, 36.021,
//									97.976896};

	// use standard labels for known fragments
	const char *neutral_loss_labels[]={"","-H2O","-NH3","-H2OH2O","-H2ONH3","-NH3NH3"};
	const mass_t neutral_loss_offsets[]={0,-18.01,-17.0265,-36.02,-35.037,-34.053};
	int num_neutral_losses = sizeof(neutral_loss_labels)/sizeof(char *);

	const char  *prefix_base_labels[]={"a","b","c"};
	const mass_t prefix_base_offsets[]={-26.9871,1.0078,18.0343}; 
	int num_prefix_bases = sizeof(prefix_base_labels)/sizeof(char *);

	const char  *suffix_base_labels[]={"x","y","z"};
	const mass_t suffix_base_offsets[]={45.9976,19.0183,2.99};
	int num_suffix_bases = sizeof(suffix_base_labels)/sizeof(char *);

	char **base_labels;
	mass_t *base_offsets;
	int num_bases;

	if (orientation == PREFIX)
	{
		base_labels  = (char **)prefix_base_labels;
		base_offsets = (mass_t *)prefix_base_offsets;
		num_bases    = num_prefix_bases;
	}
	else
	{
		base_labels  = (char **)suffix_base_labels;
		base_offsets = (mass_t *)suffix_base_offsets;
		num_bases    = num_suffix_bases;
	}

	// check all possible fragments
	mass_t tol = tolerance;
	
	if (tolerance>=0.5)
	{
		tol = (0.8 * tolerance)/charge;
	}
	else if (tolerance>0.1)
		tol = tolerance / charge;
	
	int b;
	for (b=0; b<num_bases; b++)
	{
		int n;
		for (n=0; n<num_neutral_losses; n++)
		{
			mass_t calc_offset = (base_offsets[b] + charge - 1 + neutral_loss_offsets[n])/(mass_t)charge;
			if (fabs(calc_offset-offset)<tol)
			{
				label = base_labels[b];
				if (charge>1)
					label+=char('0'+charge);
				
				if (n>0)
					label+=neutral_loss_labels[n];

				break;
			}
		}
		if (n<num_neutral_losses)
			break;
	}
}	
	


/*	if (orientation == PREFIX)
	{
		if (charge == 1)
		{
			if (fabs(offset - 1.0078) < 0.25)
			{
				label = "b";
			}
			else if (fabs(offset + 17.0027) < 0.25)
			{
				label = "b-H2O";
			}
			else if (fabs(offset + 16.0187) < 0.25)
			{
				label = "b-NH3";
			}
			else if  (fabs(offset +26.9871) < 0.25)
			{
				label = "a";
			}
			else if  (fabs(offset +35.0132) < 0.25)
			{
				label = "b-H2OH2O"; 
			}
			else if  (fabs(offset +34.0292) < 0.25)
			{
				label = "b-NH3H2O";
			}
			else if  (fabs(offset +44.9976) < 0.25)
			{
				label = "a-H2O"; 
			}
			else if  (fabs(offset +44.0136) < 0.25)
			{
				label = "a-NH3";
			}
			else if  (fabs(offset +16.0181) < 0.1)
			{
				label = "c";
			}
		}
		else if (charge == 2)
		{
			if (fabs(offset - 1.0078) < 0.2)
			{
				label="b2";
			} 
			else if (fabs(offset + 7.9975) < 0.2)
			{
				label="b2-H2O";
			}
			else if (fabs(offset - 0.5039) < 0.2)
			{
				label="b2-H";
			}
		
		}
		else if (charge == 3)
		{
			if (fabs(offset - 1.0078) < 0.175)
			{
				label = "b3";
			}
		}
	}
	else   // SUFFIX fragments
	{
		if (charge == 1)
		{
			if (fabs(offset - 19.0183) < 0.25)
			{
				label = "y";
			}
			else if (fabs(offset - 1.0078) < 0.25)
			{
				label = "y-H2O";
			}
			else if (fabs(offset - 1.9918) < 0.25)
			{
				label = "y-NH3";
			}
			else if (fabs(offset + 17.0027) < 0.25)
			{
				label = "y-H2OH2O";
			}
			else if (fabs(offset + 16.0187) < 0.25)
			{
				label = "y-NH3H2O";
			}
			else if (fabs(offset + 4.0069) < 0.25)
			{
				label = "z";
			}
			else if (fabs(offset + 47.0127) < 0.25)
			{
				label = "x";
			}
			
		}
		else if (charge == 2)
		{
			if (fabs(offset - 10.01305) < 0.2)
			{
				label = "y2";
			} 
			if (fabs(offset - 9.509) < 0.05)
			{
				label = "y2-H";
			} 
			else if (fabs(offset - 1.0078) < 0.1)
			{
				label = "y2-H2O";
			}
			else if (fabs(offset - 1.498) < 0.1)
			{
				label = "y2-NH3";
			}
			else if (fabs(offset + 7.998) < 0.1)
			{
				label = "y2-H2O-H2O";
			}
			else if (fabs(offset + 7.507) < 0.1)
			{
				label = "y2-H2O-NH3";
			}
		}
		else if (charge == 3)
		{
			if (fabs(offset - 7.0113) < 0.175)
			{
				label = "y3";
			}
		}
	} 
} */

// removes fragments that appear to be isotopic peaks of previously
// selected fragments such as b+1, y+2 etc.
void FragmentTypeSet::remove_isotopic_fragments(mass_t tolerance)
{

	sort_fragments_according_to_probs();

	int f;
	for (f=0; f<this->fragments.size(); f++)
	{
		FragmentType& frag = fragments[f];

		if (frag.label.length()<1)
			frag.make_frag_label(tolerance);

		// don't check known labels
		if (frag.label[0] != 's' &&  frag.label[0] != 'p')
			continue;

		int j;
		for (j=0; j<f; j++)
		{
			FragmentType& previous = fragments[j];

			if (previous.orientation == frag.orientation &&
				previous.charge      == frag.charge)
			{
				mass_t offset_diff = frag.offset - previous.offset;

				offset_diff *= frag.charge;

				if (offset_diff<3.3) // we do not want unrecognized fragments with such
								     // a small distance from selected fragments
				{
					frag.prob = -1;
					frag.spec_count = 0;
					break;
				}
			}
		}
	}

	sort_fragments_according_to_probs();

	while (fragments.size()>0 && fragments[fragments.size()-1].prob<0)
		fragments.pop_back();

}


void FragmentTypeSet::sort_fragments_according_to_probs()
{
	sort(fragments.begin(),fragments.end());
}

/*************************************************
Outputs labels in single line
**************************************************/
void FragmentTypeSet::output_fragment_labels(ostream& os) const
{
	int i;
	for (i=0; i<fragments.size()-1; i++)
		os << fragments[i].label << " ";
	os << fragments[i].label << endl;
}


void FragmentTypeSet::print() const
{
	int i;
	
	cout << setw(4) << left << " "
		     << setw(10) << left << "Label " 
			 << setw(5) << "Pidx"
			 <<  "offset"
			 << endl;
	for (i=0; i<fragments.size(); i++)
		cout << setw(4) << left <<i 
		     << setw(10) << left << fragments[i].label
			 << setw(5) << fragments[i].parent_frag_idx
			 << fragments[i].offset_from_parent_frag
			 << endl;
}






/***************************************************************
For each fragment idx its parent fragment is defined as the
fragment with the highest probability that has the same charge
and oreientation as the fragment in question.
****************************************************************/
void FragmentTypeSet::set_parent_frag_idxs()
{
	int i;

	for (i=0; i<fragments.size(); i++)
	{
		int j;
		for (j=0; j<i; j++)
		{
			if (fragments[j].charge == fragments[i].charge &&
				fragments[j].orientation == fragments[i].orientation)
				break;
		}

		// found parent_frag_idx
		if (j<i)
		{
			fragments[i].parent_frag_idx = j;
			fragments[i].offset_from_parent_frag = fragments[i].offset - fragments[j].offset;
		}
		else
		{
			fragments[i].parent_frag_idx=-1;
			fragments[i].offset_from_parent_frag =0;
		}
	}
}






void RegionalFragments::select_fragments_with_minimum_prob(score_t min_prob, int max_num_frags)
{
	while (frag_type_idxs.size()>0 && frag_probs[frag_type_idxs.size()-1]<min_prob)
	{
		frag_type_idxs.pop_back();
		frag_probs.pop_back();
	}

	if (max_num_frags>0)
	{
		while (frag_type_idxs.size()>max_num_frags)
		{
			frag_type_idxs.pop_back();
			frag_probs.pop_back();
		}
	}

}


void FragmentCombo::print_combo(const Config *config, ostream & os) const
{
	int i;
	for (i=0; i<this->frag_inten_idxs.size(); i++)
		os << config->get_fragment(frag_inten_idxs[i]).label << " ";
	os << "|";
	for (i=0; i<this->frag_no_inten_idxs.size(); i++)
		os << " " << config->get_fragment(frag_no_inten_idxs[i]).label;
	os << endl;
}


void FragmentCombo::read_combo(Config *config, istream& is)
{
	frag_inten_idxs.clear();
	frag_no_inten_idxs.clear();

	// find pos of separator
	char buff[128];
	char buff1[128];
	is.getline(buff,128);
	int i,len = is.gcount();
	for (i=0; i<len; i++)
		if (buff[i] == '|')
			break;
	int b_pos = i;
	if (b_pos == len)
	{
		cout << "Error: couldn't find '|' for FragmentCombo : " << buff << endl;
		exit(1);
	}
	
	// get inten idxs
	strncpy(buff1,buff,b_pos);
	istringstream iss(buff1);
	while (1)
	{
		string label;
		iss >> label;
		int f_idx=config->get_frag_idx_from_label(label);
		if (f_idx>=0)
		{
			frag_inten_idxs.push_back(f_idx);
		}
		else
			break;
	}
	
	// get no inten idxs
	istringstream iss2(buff+b_pos+1);
	while (1)
	{
		string label;
		iss2 >> label;
		int f_idx=config->get_frag_idx_from_label(label);
		if (f_idx>=0)
		{
			frag_no_inten_idxs.push_back(f_idx);
		}
		else
			break;
	}
}




struct idx_prob {
	bool operator< (const idx_prob& other) const
	{
		return (prob>other.prob);
	}
	int idx;
	score_t prob;
};



void RegionalFragments::sort_by_prob()
{
	vector<idx_prob> ip;

	if (frag_type_idxs.size() != frag_probs.size())
	{
		cout << "Error: probs and frag_idxs mismatch!" << endl;
		exit(1);
	}
	
	ip.resize(frag_type_idxs.size());
	int i;
	for (i=0; i<frag_type_idxs.size(); i++)
	{
		ip[i].idx=frag_type_idxs[i];
		ip[i].prob= frag_probs[i];
	}
	sort(ip.begin(),ip.end());
	for (i=0; i<ip.size(); i++)
	{
		frag_type_idxs[i]=ip[i].idx;
		frag_probs[i]=ip[i].prob;
	}

	strong_frag_type_idxs.clear();
	frag_type_combos.clear();
}


/***************************************************
Sets default fragments for PepNovo charge 2
****************************************************/
void RegionalFragments::init_pepnovo_types(int charge, Config *config)
{
	const char* pep_labels[]={"y","b","a","y-H2O","b-H2O","a-H2O","y-NH3","b-NH3",
                           "a-NH3","y-H2OH2O","b-H2OH2O","y-NH3H2O","b-NH3H2O",
						   "y2","b2","y3","b3","y2-H2O","b2-H2O","y2-NH3","b2-NH3",
						   "y3-H2O","b3-H2O","y3-NH3","b3-NH3"};
	frag_type_idxs.resize(13);
	int i;

	for (i=0; i<13; i++)
	{
		string label(pep_labels[i]);
		frag_type_idxs[i]=config->get_frag_idx_from_label(label);
	}
		
	if (charge>1)
	{
		frag_type_idxs.resize(15);
		for (i=13; i<15; i++)
		{
			string label(pep_labels[i]);
			frag_type_idxs[i]=config->get_frag_idx_from_label(label);
		}
	}

	
	if (charge>2)
	{
		frag_type_idxs.resize(21);
		for (i=15; i<21; i++)
		{
			string label(pep_labels[i]);
			frag_type_idxs[i]=config->get_frag_idx_from_label(label);
		}
	}

//	cout << ">> ";
//	for (i=0; i<frag_type_idxs.size(); i++)
//		cout << frag_type_idxs[i] << " ";
//	cout << endl;

	frag_probs.resize(config->get_all_fragments().size(),0);
	set_parent_frag_idxs(config);
}




void RegionalFragments::init_with_all_types(int charge, Config *config)
{
	int i;
	frag_type_idxs.clear();
	const vector<FragmentType>& all_fragments = config->get_all_fragments();
	for (i=0; i<all_fragments.size(); i++)
		if (all_fragments[i].charge<=charge)
			frag_type_idxs.push_back(i);

	frag_probs.resize(all_fragments.size(),0);

	set_parent_frag_idxs(config);
}






struct idx_pair {
	bool operator< (const idx_pair& other) const
	{
		return (prob>other.prob);
	}
	int f_idx;
	score_t prob;
};

/******************************************************************
// makes the order of the fragments in the frag_idxs and frag_probs 
// vectors be in descending probability order
*******************************************************************/
void RegionalFragments::sort_according_to_frag_probs()
{
	int i;
	vector<idx_pair> pairs;
	if (frag_type_idxs.size() != frag_probs.size())
	{
		cout << "Error: mismtach in frag_idxs and frag_probs sizes!" << 
			frag_type_idxs.size() << " <=> " << frag_probs.size() << endl;
		exit(1);
	}

	pairs.resize(frag_type_idxs.size());
	for (i=0; i<frag_type_idxs.size(); i++)
	{
		pairs[i].f_idx=frag_type_idxs[i];
		pairs[i].prob =frag_probs[i];
	}
	sort(pairs.begin(),pairs.end());
	for (i=0; i<pairs.size(); i++)
	{
		frag_type_idxs[i]=pairs[i].f_idx;
		frag_probs[i]=pairs[i].prob;
	}
}


/*********************************************************************
// checks if the fragments have a parent fragment in the frag_idxs and 
// give its idx, otherwise -1
**********************************************************************/
void RegionalFragments::set_parent_frag_idxs(Config *config)
{
	int i;
	const vector<FragmentType>& all_fragments = config->get_all_fragments();
	parent_frag_positions.resize(frag_type_idxs.size(),-1);

	for (i=0; i<frag_type_idxs.size(); i++)
	{
		int frag_type_idx = frag_type_idxs[i];
	//	cout << frag_type_idx << " ";
		int parent_idx = all_fragments[frag_type_idx].parent_frag_idx;
		if (parent_idx>=0)
			parent_frag_positions[i]=get_position_of_frag_type_idx(parent_idx);
	}
}


/*********************************************************************
// chooses all fragments that have high enough probability to be strong
**********************************************************************/
void RegionalFragments::select_strong_fragments(score_t min_prob, int max_num_strong)
{
	int i;

	strong_frag_type_idxs.clear();
	for (i=0; i<frag_type_idxs.size() && i <max_num_strong; i++)
	{
		if (i>=2 && frag_probs[i]<min_prob)
			continue;

		strong_frag_type_idxs.push_back(frag_type_idxs[i]);
	}
}




