#ifndef __MODEL_H__
#define __MODEL_H__

/*************************************************************************

  This is the top contianer class for PepNovo's models.
  All models must inherit this class. The acutal model class that is used
  is RegularRankModel which is inherits from DiscretePeakModel, which inherits from Model.

  Training/reading/writing of all models is done through the functions of this class.

  The model class contains virtual functions for scoring breakages and edges.

  Before scoring a spectrum the model should be initialized for that spectrum.
**************************************************************************/
#include "Config.h"
#include "BasicDataStructs.h"
#include "Spectrum.h"
#include "FileManagement.h"
#include "includes.h"
#include "PMCSQS.h"


#define MINIMAL_NUMBER_SPECTRA_FOR_FRAGMENT_SELECTION 100
 
struct Edge; // forward declr
struct SeqPath;
class PrmGraph;

class Model {
public:
	

	string get_model_name() const { return model_name; }

	void set_model_name(string _name) { model_name = _name; config.set_model_name(_name); }

	Config * get_config() { return &config; }
	void set_config(Config *_config) { config= *_config; }


	// this function performs the entire training process of the model
	void full_train_model(const char *name, 
			const FileManager& fm, mass_t tolerance);

	void train_pmc(char *pos_list)
	{
		pmcsqs.train_pmc_models(&config,pos_list);
	}

	void train_sqs(char *pos_list, char *neg_list, float *weights=NULL)
	{
		pmcsqs.train_sqs_models(&config,pos_list,neg_list,weights);
	}

	float get_best_mz_charge(Config *config, const BasicSpectrum& bs, 
						   mass_t* mz1, int* charge1, float *prob1,
						   mass_t* mz2, int* charge2, float *prob2)
	{
		return pmcsqs.get_best_mz_charge(config,bs,mz1,charge1,prob1,mz2,charge2,prob2);
	}

	bool get_ind_pmcsqs_was_intialized() const { return (pmcsqs.get_ind_initialized_pmc() &&
		pmcsqs.get_ind_initialized_sqs()); }

	const PMCSQS_Scorer* get_pmcsqs_ptr() const { return &pmcsqs; }
	

	
	// uses the current model fragments to test what tolerance can be used that 
	// catches at least *percen_frags_caught* of the dominant fragments
	mass_t calculate_tolerance(const FileManager& fm, mass_t max_tolerance,
							   float percent_frags_caught = 0.95);
	
	// 
	void perform_offset_frequency_function(const FileManager& fm, mass_t tolerance,
										   int max_charge);

	void read_model(const char* model_name);

	void clone_charge_model(int source_charge, int target_charge);


	void write_model();


	virtual void init_score_model() = 0;

	// in case there are som preliminary actions the model needs to 
	// do before it can score a spectrum (most models might not need this)
	virtual void init_model_for_scoring_spectrum(Spectrum *spec) = 0;

	virtual void score_breakage(Spectrum *spec, Breakage *breakage, bool verbose=false) const =0;
	
	virtual void score_edge_and_breakages(Spectrum *spec, Edge *edge) const =0;

	virtual score_t get_prefix_score_penalty(Spectrum *spec) const =0;
	
	virtual score_t get_suffix_score_penalty(Spectrum *spec) const =0;

	virtual score_t get_missing_breakage_score(int charge, int size_idx, int region_idx) const =0;

	virtual void score_graph_edges(PrmGraph& prm) const =0;

	virtual float calc_seq_prob(const SeqPath& path, int charge, const PrmGraph& prm,
			int denovo_rank, int score_rank, score_t top_score, bool from_denovo) const =0;

	virtual int get_max_score_model_charge() const =0;


protected:

	string model_name;
	
	Config config;

	PMCSQS_Scorer pmcsqs;


	bool select_fragments(const char *name, const FileManager& fm,
		int max_num_frags =0, float min_prob = 0.5);


	// The actual training and writing of score models depends on the type 
	// of model that is used

	virtual void train_score_model(const char *name, const FileManager& fm, 
		int charge=0) =0;

	virtual void read_score_model(istream& is) =0;

	virtual void write_score_model(ostream& os) const =0;
};



/******************************************************************
Looks for the pm_with_19 that maximizes a score related objective..
Assumes that isotope socre were already calculated for all peaks.
Offers a secondary pm_with_19 if there is a close call.
*******************************************************************/
//void  calc_corrected_pm_with_19(Spectrum *spec, Model *model,
//							mass_t pm_tolerance, bool print = false);


// determines the tolerance for which *cuttoff_prob* of the abundant fragments
// are caught
mass_t calc_tolerance_distribution(Model *model,  const FileManager& fm, mass_t max_tolerance,
								   float cutoff_prob=0.96);

// determines the parent mass tolerance for which *cuttoff_prob* of the abundant fragments
// are caught
mass_t calc_parent_mass_tolerance_distribution(Model *model,  const FileManager& fm, 
											   float cutoff_prob=0.98);


#endif

