#include "PMCSQS.h"
#include "auxfun.h"


/********************************************************************
Returns the highest probability m/z value for the given charge 
according to the models. If there is another m/z that is far enough and 
also has a high probability, it is also returned;
*********************************************************************/
void PMCSQS_Scorer::find_best_mz_values(const BasicSpectrum& bs, int charge, PmcSqsChargeRes& res) const
{
	vector<ME_Regression_Sample> spec_samples;
	vector<float> probs;
	fill_fval_vectors_with_PMC(bs, charge, spec_samples);

	probs.resize(spec_samples.size(),-1);
	
	int best_idx=-1;
	float best_prob=-1;
	int i;
	for (i=0; i<spec_samples.size(); i++)
	{
		probs[i]=pmc_models[charge]->p_y_given_x(0,spec_samples[i]);
		if (probs[i]>best_prob)
		{
			best_prob=probs[i];
			best_idx = i;
		}
	}
	
	res.mz1 = curr_spec_pmc_tables[charge][best_idx].m_over_z;
	res.prob1 = best_prob;

	// look for additional m/z
	int second_best_idx=-1;
	float second_best_prob=-1;

	const mass_t mz_diff = curr_spec_pmc_tables[charge][1].m_over_z - 
						   curr_spec_pmc_tables[charge][0].m_over_z;

	const int idx_diff = (int)(0.8/(charge * mz_diff));
	for (i=0; i<spec_samples.size(); i++)
	{
		if (fabs(i-best_idx)<idx_diff)
			continue;

		if (probs[i]>second_best_prob)
		{
			second_best_prob=probs[i];
			second_best_idx = i;
		}

	}

	res.mz2 = curr_spec_pmc_tables[charge][second_best_idx].m_over_z;
	res.prob2 = second_best_prob;

//	cout << charge << " ]\t" << res.mz1 << "\t" << res.prob1 << "\t" << res.mz2 << "\t" << res.prob2 << endl;


}


/***************************************************************************
Gives detailed m/z and prob values for the different charges.
Returns the highest sqs probability found for a spectrum.
****************************************************************************/
float PMCSQS_Scorer::get_pmcsqs_results_for_spectrum(Config *config, const BasicSpectrum& bs,
						vector<PmcSqsChargeRes>& res)
{
	ME_Regression_Sample sqs_sam;

	init_for_current_spec(config,bs);

	calculate_curr_spec_pmc_values(bs,bin_increment);

	const int num_charges = pmc_models.size();
	res.resize(num_charges);
	float max_prob=-1;
	int max_prob_charge=0;

	vector<float> adjusted_probs;
	adjusted_probs.resize(num_charges,0);


	if (this->ind_initialized_sqs)
	{
		fill_fval_vector_with_SQS(bs, sqs_sam);
	
		int charge;
		for (charge=1; charge<num_charges; charge++)
		{
			const float prob = sqs_models[charge][0]->p_y_given_x(0,sqs_sam);
			adjusted_probs[charge]=prob;
			if (prob>max_prob)
			{
				max_prob=prob;
				max_prob_charge=charge;
			}
		}

		// copare the max_prob_charge with the rest, if it is beaten by anyone
		// that one becomes the max_prob_charge
		// there is no re-running of the comparison, so there is no danger of
		// endless loops where the max_charge_prob iterates between different values
		
		for (charge=1; charge<=max_charge; charge++)
		{
			if (charge == max_prob_charge)
				continue;

			float comp_prob = -1;
			if (charge<max_prob_charge)
			{
				if (! sqs_models[max_prob_charge][charge])
					continue;

				comp_prob=sqs_models[max_prob_charge][charge]->p_y_given_x(0,sqs_sam);
			}
			else
			{
				if (! sqs_models[charge][max_prob_charge])
					continue;

				comp_prob = 1.0 - sqs_models[charge][max_prob_charge]->p_y_given_x(0,sqs_sam);	
			}

			if (comp_prob < 0.5)
			{
				
				adjusted_probs[charge]=1.0 - comp_prob;
				adjusted_probs[max_prob_charge] = comp_prob;
				max_prob_charge = charge;
			}
			else
				adjusted_probs[charge]=1.0 - comp_prob;

		}
	}
	else // give the max prob charge to the input charge, the rest get 0
	{
		int max_prob_charge=bs.ssf->charge;
		if (max_prob_charge<=0)
		{
			cout << "Error: no SQS model was read, so charge must be supplied in the spectrum!" << endl;
			exit(1);
		}

		adjusted_probs[max_prob_charge]=1.0;
		max_prob=1;
	}


	int charge;
	for (charge=1; charge<num_charges; charge++)
	{
		const float prob = adjusted_probs[charge];
		if (prob>0.015)
		{
			find_best_mz_values(bs,charge,res[charge]);
			const float ratio =  res[charge].prob2 / res[charge].prob1;
			res[charge].prob1=prob;
			res[charge].prob2=prob*ratio; // give probabilities that are relative to the sqs probs
		}
		else // only give one guess, the second one will come from another charge
		{
			res[charge].mz1=bs.ssf->m_over_z;
			res[charge].prob1=prob;
		}
	}
	

	return max_prob;
}



/******************************************************************************
Selects the the two best values of charges 1,2,3
returns the max prob found
*******************************************************************************/
float PMCSQS_Scorer::get_best_mz_charge(Config *config, const BasicSpectrum& bs, 
						   mass_t* mz1, int* charge1, float *prob1,
						   mass_t* mz2, int* charge2, float *prob2)
{
	vector<PmcSqsChargeRes> res;

	if (! this->ind_initialized_pmc)
	{
		cout << "Error: no PMC model was read!" << endl;
		exit(1);
	}

	get_pmcsqs_results_for_spectrum(config,bs,res);

	float best_prob=-1;
	int best_charge=0;

	int charge;
	for (charge =1; charge<=this->max_charge; charge++)
		if (res[charge].prob1>best_prob)
		{
			best_charge = charge;
			best_prob = res[charge].prob1;
		}

	float second_best_prob=res[best_charge].prob2;
	int   second_best_charge = best_charge;
	
	for (charge =1; charge<=this->max_charge; charge++)
		if (charge != best_charge && res[charge].prob1>second_best_prob)
		{
			second_best_charge = charge;
			second_best_prob = res[charge].prob1;
		}

	*mz1 = (mass_t)res[best_charge].mz1;
	*charge1 = best_charge;
	*prob1 = best_prob;

	if (mz2 && charge2)
	{
		*charge2 = second_best_charge;
		*prob2   = second_best_prob;
		if (second_best_charge == best_charge)
		{
			*mz2 = (mass_t)res[best_charge].mz2;
		}
		else
			*mz2 = (mass_t)res[second_best_charge].mz1;
	}


	return best_prob;
}







const int DPColumnBytes = sizeof(int)*(Val+1);

// for each peak,aa holds the idx of the previous aa if they have a mass
// diff of that aa (within tolerance)
// entry 0 in each column holds an indicator if peak is in aa diff
// entry 1 in each column holds an indicator if peak has a 
void PMCSQS_Scorer::fill_SQS_DP(const BasicSpectrum& bs, vector<DPColumn>& dp ) const
{
	const QCPeak *peaks = bs.peaks;
	const int num_peaks = bs.num_peaks;
	const vector<mass_t>& aa2mass = config->get_aa2mass();
	const mass_t tag_tolerance = (config->get_tolerance()<0.15 ? 
									config->get_tolerance() : config->get_tolerance()*0.33);


	dp.resize(num_peaks);
	int i;
	for (i=0; i<num_peaks; i++)
	{
		dp[i].pointers[0]=0;
		int j;
		for (j=1; j<=Val; j++)
			dp[i].pointers[j]=-1;
	}

	int aa;
	for (aa=Ala; aa<=Val; aa++)
	{
		if (aa==Ile || aa==Xle)
			continue;

		const mass_t min_offset = aa2mass[aa]-tag_tolerance;
		const mass_t max_offset = aa2mass[aa]+tag_tolerance;

		int trail_idx=0;
		int lead_idx=1;

		while (lead_idx<num_peaks)
		{
			if (curr_spec_iso_levels[lead_idx]>0)
			{
				lead_idx++;
				continue;
			}

			while (peaks[lead_idx].mass-peaks[trail_idx].mass>max_offset)
				trail_idx++;

			if (curr_spec_iso_levels[trail_idx]==0 && 
				peaks[lead_idx].mass-peaks[trail_idx].mass>min_offset)
			{
				dp[lead_idx].pointers[aa]=trail_idx;
				dp[lead_idx].pointers[0]=1;
				dp[trail_idx].pointers[0]=1;

			//	cout << "Off: " << peaks[lead_idx].mass-peaks[trail_idx].mass - min_offset<< endl;
			}
			else 
				dp[lead_idx].pointers[aa]=-1;

			lead_idx++;
		}

	}
}


/*****************************************************************************
Does the raw calculations required for processing the spectrum.
******************************************************************************/
void PMCSQS_Scorer::init_for_current_spec(Config *_config, 
										  const BasicSpectrum& bs)
{
	config = _config;

	bs.calc_peak_isotope_levels(config->get_tolerance(),this->curr_spec_iso_levels);
	bs.select_strong_peak_idxs(this->curr_spec_iso_levels,this->curr_spec_strong_inds);


	curr_spec_total_intensity=0;
	curr_spec_strong_intensity=0;
	curr_spec_num_strong=0;
	int i;
	for (i=0; i<bs.num_peaks; i++)
	{
		curr_spec_total_intensity+=bs.peaks[i].intensity;
		if (curr_spec_strong_inds[i])
		{
			curr_spec_strong_intensity+=bs.peaks[i].intensity;
			curr_spec_num_strong++;
		}
	}


	this->curr_spec_pmc_tables.clear();
	this->curr_spec_pmc_tables.resize(max_charge+1);

	this->curr_spec_background_stats.clear();
	this->curr_spec_background_stats.resize(max_charge+1);

	this->curr_spec_maximal_values.clear();
	this->curr_spec_maximal_values.resize(max_charge+1);

}


/****************************************************************************
*****************************************************************************/
void PMCSQS_Scorer::fill_fval_vector_with_SQS(const BasicSpectrum& bs,
								   ME_Regression_Sample& sam) const
{
	const mass_t tolerance = config->get_tolerance();
	const QCPeak *peaks  = bs.peaks;
	const int  num_peaks = bs.num_peaks;
	const mass_t m_over_z = bs.ssf->m_over_z;
	const int num_strong=curr_spec_num_strong;
	const float total_intensity=curr_spec_total_intensity;



	sam.f_vals.push_back(fval(SQS_CONST,1.0));
	sam.f_vals.push_back(fval(SQS_NUM_PEAKS, (float)num_peaks));
	sam.f_vals.push_back(fval(SQS_SQR_NUM_PEAKS,sqrt((float)num_peaks)));

	int size_group=0;

	if (m_over_z<1000.0)
	{
		sam.f_vals.push_back(fval(SQS_IND_BELOW_1000,1.0));
		sam.f_vals.push_back(fval(SQS_M_OVER_Z_BELOW_1000, m_over_z*0.001));
	} 
	else
	{
		size_group=1;
		sam.f_vals.push_back(fval(SQS_IND_MZ_ABOVE_1000,1.0));
		sam.f_vals.push_back(fval(SQS_M_OVER_Z_ABOVE_1000, m_over_z*0.001));  
	}

	float grass_level_inten=NEG_INF;
	

	// calculate grass level peaks
	if (1)
	{
		int i;
		vector<float> peak_intens;
		peak_intens.resize(num_peaks);
		for (i=0; i<num_peaks; i++)
			peak_intens[i]=peaks[i].intensity;

		sort(peak_intens.begin(),peak_intens.end());

		int idx_G = num_peaks/3;
		grass_level_inten = peak_intens[idx_G];

		float cum_intensity2G=0;
		float inten_2G=2.0 * grass_level_inten;
		int idx_2G = idx_G;
		while (idx_2G<num_peaks && peak_intens[idx_2G]<inten_2G)
		{
			cum_intensity2G+=peak_intens[idx_2G];
			idx_2G++;
		}

		float cum_intensity5G=0;
		float inten_5G = 5.0 * grass_level_inten;
		int idx_5G = idx_2G;
		while (idx_5G<num_peaks && peak_intens[idx_5G]<inten_5G)
		{
			cum_intensity5G+=peak_intens[idx_5G];
			idx_5G++;
		}

		float inten_10G = 10.0 * grass_level_inten;
		int idx_10G = idx_5G;
		while (idx_10G<num_peaks && peak_intens[idx_10G]<inten_10G)
			idx_10G++;

		sam.f_vals.push_back(fval(SQS_PROP_UPTO2G,(float)idx_2G/(float)num_peaks));
		sam.f_vals.push_back(fval(SQS_PROP_UPTO5G,(float)(idx_5G-idx_2G)/(float)num_peaks));
		sam.f_vals.push_back(fval(SQS_PROP_UPTO10G,(float)(idx_10G-idx_5G)/(float)num_peaks)); 
		sam.f_vals.push_back(fval(SQS_PROP_MORE10G,(float)(num_peaks-idx_10G)/(float)num_peaks));
		sam.f_vals.push_back(fval(SQS_PROP_INTEN_UPTO2G,cum_intensity2G/total_intensity));
		sam.f_vals.push_back(fval(SQS_PROP_INTEN_UPTO5G,cum_intensity5G/total_intensity));
		sam.f_vals.push_back(fval(SQS_PROP_INTEN_MORE5G,
			(total_intensity-cum_intensity2G-cum_intensity5G)/total_intensity));

	/*	cout << "UPTO2G: " << (float)idx_2G/(float)num_peaks << endl;
		cout << "UPTO5G: " << (float)(idx_5G-idx_2G)/(float)num_peaks << endl;
		cout << "UPTO10G: " << (float)(idx_10G-idx_5G)/(float)num_peaks << endl;
		cout << "MORE10G: " << (float)(num_peaks-idx_10G)/(float)num_peaks << endl;

		cout << "INTEN2G: " << cum_intensity2G/total_intensity << endl;
		cout << "INTEN5G: " << cum_intensity5G/total_intensity << endl;
		cout << "INTENMORE: " << (total_intensity-cum_intensity2G-cum_intensity5G)/total_intensity << endl;*/

	}

	// isotope features
	if (1)
	{
		int i;
		int num_with_iso=0;
		int strong_with_iso=0;
		for (i=1; i<num_peaks; i++)
			if (curr_spec_iso_levels[i]>0)
			{
				num_with_iso++;
				if (curr_spec_iso_levels[i-1]==0 && curr_spec_strong_inds[i-1])
					strong_with_iso++;
			}

		sam.f_vals.push_back(fval(SQS_PROP_ISO_PEAKS,num_with_iso/(float)num_peaks)); 
		sam.f_vals.push_back(fval(SQS_PROP_STRONG_WITH_ISO_PEAKS,strong_with_iso/(float)num_strong)); 

	//	cout << "PROP WITH ISO  : " << num_with_iso/(float)num_peaks << endl;
	//	cout << "STRONG WITH IOS: " << strong_with_iso/(float)num_strong << endl;
	}


	// neutral loss features
	if (1)
	{
		const mass_t offsets[3]={18.0105, 17.0265, 27.9994};
		const int num_offsets = 3;
		
		int i;
		for (i=0; i<num_offsets; i++)
		{
			const mass_t min_offset = offsets[i]-tolerance;
			const mass_t max_offset = offsets[i]+tolerance;

			int num_pairs=0;
			int num_strong_pairs=0;

			int trail_idx=0;
			int lead_idx=1;

			while (lead_idx<num_peaks)
			{
				while (peaks[lead_idx].mass-peaks[trail_idx].mass>max_offset)
					trail_idx++;

				if (peaks[lead_idx].mass-peaks[trail_idx].mass>min_offset)
				{
					num_pairs++;
					if (curr_spec_strong_inds[lead_idx])
						num_strong_pairs++;
				}
				lead_idx++;
			}

			sam.f_vals.push_back(fval(SQS_PROP_ALL_WITH_H2O_LOSS+i,num_pairs/(float)num_peaks));
			sam.f_vals.push_back(fval(SQS_PROP_STRONG_WITH_H2O_LOSS+i,num_strong_pairs/(float)num_strong)); 

		//	cout << "OFF REG " << i << " " << num_pairs/(float)num_peaks << endl;
		//	cout << "OFF STR " << i << " " << num_strong_pairs/(float)num_strong << endl;
		}
	}

	// tag features
	if (1)
	{
		vector<DPColumn> dp;
		fill_SQS_DP(bs,dp);

		float inten_in_tags[3]={0,0,0};
		int num_peaks_in_tags[3]={0,0,0};
		int num_strong_peaks_in_tags[3]={0,0,0};

		vector<int> tag_lengths;
		tag_lengths.resize(num_peaks,0);

		int longest=0;
		int i;
		for (i=1; i<num_peaks; i++)
		{
			int max_tl=-1;
			int aa;
			for (aa=Ala; aa<=Val; aa++)
			{
				const int prev = dp[i].pointers[aa];
				if (prev>=0 && tag_lengths[prev]>max_tl)
					max_tl = tag_lengths[prev];
			}

			tag_lengths[i]=max_tl+1;
			if (tag_lengths[i]>longest)
				longest = tag_lengths[i];

			int k;
			for (k=0; k<3; k++)
			{
				if (tag_lengths[i]==k)
					break;
				
				inten_in_tags[k]+=peaks[i].intensity;
				num_peaks_in_tags[k]++;
				if (curr_spec_strong_inds[i])
					num_strong_peaks_in_tags[k]++;
			}		
		}

		

		if (longest>=4)
		{
			sam.f_vals.push_back(fval(SQS_IND_MAX_TAG_LENGTH_ABOVE_4,1.0));
			sam.f_vals.push_back(fval(SQS_MAX_TAG_LENGTH_ABOVE_4,(float)longest));
		}
		else
		{
			sam.f_vals.push_back(fval(SQS_IND_MAX_TAG_LENGTH_BELOW_4,1.0));
			sam.f_vals.push_back(fval(SQS_MAX_TAG_LENGTH_BELOW_4,(float)longest));
		}

		float inten_in_tags_both_sides=0;
		for (i=0; i<num_peaks; i++)
			if (dp[i].pointers[0])
				inten_in_tags_both_sides+=peaks[i].intensity;

		sam.f_vals.push_back(fval(SQS_PROP_INTEN_IN_TAGS,inten_in_tags_both_sides/total_intensity));

	//	cout << "MAX TAG: " << longest << endl;
	//	cout << "Inten in TAG: " << inten_in_tags_both_sides/total_intensity << endl;

		const float strong_threshes[]={0.3,0.2,0.1};
		int k;
		for (k=0; k<3; k++)
		{
			const int pos_off = 4*k;
			const float prop_strong = (float)num_strong_peaks_in_tags[k]/(float)num_strong;
			sam.f_vals.push_back(fval(SQS_PROP_TAGS1+pos_off,(float)num_peaks_in_tags[k]/(float)num_peaks));
			sam.f_vals.push_back(fval(SQS_PROP_STRONG_PEAKS_IN_TAG1+pos_off,prop_strong));
			sam.f_vals.push_back(fval(SQS_PROP_INTEN_TAG1+pos_off,inten_in_tags[k]/total_intensity));

			if (prop_strong<strong_threshes[k])
				sam.f_vals.push_back(fval(SQS_PROP_STRONG_BELOW30_TAG1+pos_off,1.0));

		//	cout << k+1 << " PROP  " << (float)num_peaks_in_tags[k]/(float)num_peaks << endl;
		//	cout << k+1 << " STRNG " << (float)num_strong_peaks_in_tags[k]/(float)num_strong << endl;
		//	cout << k+1 << " INTEN " << inten_in_tags[k]/total_intensity << endl;
		}
	}


	// density features
	if (1)
	{
		mass_t t1=m_over_z*0.6666;
		mass_t t2=m_over_z*1.3333;
		mass_t h1=m_over_z;

		float inten_t1=0,inten_t2=0, inten_h1=0;
		int	  num_peaks_t1=0, num_peaks_t2=0, num_peaks_h1=0;

		int i;
		for (i=0; i<num_peaks && peaks[i].mass<t1; i++)
			inten_t1+=peaks[i].intensity;
		
		num_peaks_t1=i;

		for ( ; i<num_peaks && peaks[i].mass<h1; i++)
			inten_h1+=peaks[i].intensity;

		num_peaks_h1=i;

		inten_t2=inten_h1;

		for ( ; i<num_peaks && peaks[i].mass<t2; i++)
			inten_t2+=peaks[i].intensity;

		num_peaks_t2=i-num_peaks_t1;

		int num_peaks_t3= num_peaks - num_peaks_t2 - num_peaks_t1;
		float inten_t3 = total_intensity - inten_t2 - inten_t1;

		const int group_offset = size_group * 10;
	
		sam.f_vals.push_back(fval(SQS_PEAK_DENSE_T1_I+group_offset,(float)(num_peaks_t1)/(float)num_peaks));
		sam.f_vals.push_back(fval(SQS_PEAK_DENSE_T2_I+group_offset,(float)(num_peaks_t2)/(float)num_peaks));
		sam.f_vals.push_back(fval(SQS_PEAK_DENSE_T3_I+group_offset,(float)(num_peaks_t3)/(float)num_peaks));

	//	cout << "T1 PEAKS: " << (float)(num_peaks_t1)/(float)num_peaks << endl;
	//	cout << "T2 PEAKS: " << (float)(num_peaks_t2)/(float)num_peaks << endl;
	//	cout << "T3 PEAKS: " << (float)(num_peaks_t3)/(float)num_peaks << endl;


		sam.f_vals.push_back(fval(SQS_INTEN_DENSE_T1_I+group_offset,inten_t1/total_intensity));
		sam.f_vals.push_back(fval(SQS_INTEN_DENSE_T2_I+group_offset,inten_t2/total_intensity));
		sam.f_vals.push_back(fval(SQS_INTEN_DENSE_T3_I+group_offset,inten_t3/total_intensity));

	//	cout << "T1 INTEN: " << inten_t1/total_intensity << endl;
	//	cout << "T2 INTEN: " << inten_t2/total_intensity << endl;
	//	cout << "T3 INTEN: " << inten_t3/total_intensity << endl;


		int num_peaks_h2 = num_peaks - num_peaks_h1;
		float inten_h2 = total_intensity - inten_h1;

		sam.f_vals.push_back(fval(SQS_PEAK_DENSE_H1_I+group_offset,(num_peaks_h1)/(float)num_peaks));
		sam.f_vals.push_back(fval(SQS_PEAK_DENSE_H2_I+group_offset,(num_peaks_h2)/(float)num_peaks));

		sam.f_vals.push_back(fval(SQS_INTEN_DENSE_H1_I+group_offset,inten_h1/total_intensity));
		sam.f_vals.push_back(fval(SQS_INTEN_DENSE_H2_I+group_offset,inten_h2/total_intensity));
	
		float inten33=0.333* total_intensity;
		float inten50=0.5  * total_intensity;
		float inten75=0.75 * total_intensity;
		float inten90=0.90 * total_intensity;


		const int other_offset = 4 * size_group;

		float cum_inten=0;

		for (i=0; i<num_peaks && cum_inten<inten33; i++)
			cum_inten+=peaks[i].intensity;

		if (i==num_peaks)
			i--;

		sam.f_vals.push_back(fval(SQS_PROP_MZ_RANGE_WITH_33_INTEN_I+other_offset,
								peaks[i].mass/m_over_z));

		for ( ; i<num_peaks && cum_inten<inten50; i++)
			cum_inten+=peaks[i].intensity;

		if (i==num_peaks)
			i--;

		sam.f_vals.push_back(fval(SQS_PROP_MZ_RANGE_WITH_50_INTEN_I+other_offset,
								peaks[i].mass/m_over_z));

		for ( ; i<num_peaks && cum_inten<inten75; i++)
			cum_inten+=peaks[i].intensity;

		if (i==num_peaks)
			i--;

		sam.f_vals.push_back(fval(SQS_PROP_MZ_RANGE_WITH_75_INTEN_I+other_offset,
								peaks[i].mass/m_over_z));

		for ( ; i<num_peaks && cum_inten<inten90; i++)
			cum_inten+=peaks[i].intensity;

		if (i==num_peaks)
			i--;

		sam.f_vals.push_back(fval(SQS_PROP_MZ_RANGE_WITH_90_INTEN_I+other_offset,
								peaks[i].mass/m_over_z));		
	}

	// pmc features
	if (1)
	{
		vector< vector<float> > sqs_features;
		
		get_sqs_features_from_pmc_tables(bs,sqs_features);

		vector<float> max_vals;
		max_vals.resize(4,0.000001);

		// first give absolute counts
		int charge;
		for (charge=1; charge<=3; charge++)
		{
			vector<float>& vals = sqs_features[charge];
			const int idx_off = 4 *( charge-1);

			int i;
			for (i=0; i<4; i++)
			{
				sam.f_vals.push_back(fval(SQS_NUM_FRAG_PAIRS_1+idx_off+i,vals[i]));			
				if (vals[i]>max_vals[i])
					max_vals[i]=vals[i];
			}
				
		}

		// add features for prop of max
		for (charge=1; charge<=3; charge++)
		{
			vector<float>& vals = sqs_features[charge];
			const int idx_off = 4 *( charge-1);

			int i;
			for (i=0; i<4; i++)
				sam.f_vals.push_back(fval(SQS_PROP_OF_MAX_FRAG_PAIRS_1+idx_off+i,vals[i]/max_vals[i]));			
		}

		// conver to proportions by dividing by the number of peaks/strong_peaks
		// and subtract the background levels (first)
		const float one_over_peaks = 1.0/(float)num_peaks;
		const float one_over_strong = 1.0/(float)num_strong;

		
	
		for (charge=1; charge<=3; charge++)
		{
			vector<float>& vals = sqs_features[charge];
			const int idx_off = 4 *( charge-1);

			// normalize to get proportions of total number of peaks/strong
			vals[0]-= this->curr_spec_background_stats[charge].num_frag_pairs;
			vals[1]-= this->curr_spec_background_stats[charge].num_strong_frag_pairs;
			vals[2]-= this->curr_spec_background_stats[charge].num_c2_frag_pairs;
			vals[3]-= this->curr_spec_background_stats[charge].num_strong_c2_frag_pairs;

			vals[0]*=one_over_peaks;
			vals[1]*=one_over_strong;
			vals[2]*=one_over_peaks;
			vals[3]*=one_over_strong;

			int i;
			for (i=0; i<4; i++)
				sam.f_vals.push_back(fval(SQS_PROP_FRAG_PAIRS_1+idx_off+i,vals[i]));			
		}
	}

	

/*	
	SQS_AVG_NUM_PEAKS_IN_2DA_INTERVAL, 
	SQS_AVG_NUM_PEAKS_IN_10DA_INTERVAL_FROM_STRONG, */


	sort(sam.f_vals.begin(),sam.f_vals.end());
}





PmcStats calc_pmc_stats_for_mass(const QCPeak *peaks, int num_peaks, mass_t single_charge_pair_sum,
								 mass_t tolerance, mass_t increment, const vector<float>& iso_levels,
								 const vector<bool>& strong_inds)
{
	const mass_t min_single_sum = single_charge_pair_sum - tolerance;
	const mass_t max_single_sum = single_charge_pair_sum + tolerance;
	const mass_t min_double_sum = min_single_sum + 1.0;
	const mass_t max_double_sum = max_single_sum + 1.0;
	const mass_t double_charge_pair_sum = single_charge_pair_sum +1.0;
	PmcStats pmc_stats;

	int forward_idx = -1;
	int back_idx = num_peaks-1;

	pmc_stats.tol_frag_pairs=0;
	pmc_stats.tol_strong_frag_pairs=0;
	pmc_stats.tol_c2_frag_pairs=0;
	pmc_stats.tol_strong_c2_frag_pairs=0;

	pmc_stats.tol_sqr_frag_pairs=0;
	pmc_stats.tol_strong_sqr_frag_pairs=0;
	pmc_stats.tol_c2_sqr_frag_pairs=0;
	pmc_stats.tol_strong_c2_sqr_frag_pairs=0;

	// find pairs of b/y
	while (forward_idx<back_idx)
	{
		forward_idx++;
		if (iso_levels[forward_idx]>0)
			continue;

		while (back_idx>=0 && peaks[forward_idx].mass + peaks[back_idx].mass>max_single_sum)
			back_idx--;

		if (back_idx>=0 && peaks[forward_idx].mass + peaks[back_idx].mass > min_single_sum)
		{
			if (iso_levels[back_idx]>0)
				continue;

			mass_t tol = fabs(peaks[forward_idx].mass + peaks[back_idx].mass - single_charge_pair_sum);
			if (tol<increment)
				tol = increment;
			const mass_t tol_sqr = tol*tol;
			const float inten_sum = peaks[forward_idx].intensity + peaks[back_idx].intensity;
					
			pmc_stats.num_frag_pairs++;
			pmc_stats.inten_frag_pairs += inten_sum;
			pmc_stats.tol_frag_pairs += tol;
			pmc_stats.tol_sqr_frag_pairs += tol_sqr;

			if (strong_inds[forward_idx] || strong_inds[back_idx])
			{
				pmc_stats.num_strong_frag_pairs++;
				pmc_stats.inten_strong_pairs += inten_sum;
				pmc_stats.tol_strong_frag_pairs += tol;
				pmc_stats.tol_strong_sqr_frag_pairs += tol_sqr;
			}
		}
	}

	// normalize tolerances
	if (pmc_stats.num_frag_pairs>0)
	{
		float one_over = 1.0 / (float)pmc_stats.num_frag_pairs;
		pmc_stats.tol_frag_pairs *= one_over;
		pmc_stats.tol_sqr_frag_pairs *= one_over;
	}

	if (pmc_stats.num_strong_frag_pairs>0)
	{
		float one_over = 1.0 / (float)pmc_stats.num_strong_frag_pairs;
		pmc_stats.tol_strong_frag_pairs *= one_over;
		pmc_stats.tol_strong_frag_pairs *= one_over;
	}
		

	// find pairs b/y2
	forward_idx = -1;
	back_idx = num_peaks-1;

	const int last_idx =num_peaks-1;
	while (forward_idx<last_idx)
	{
		forward_idx++;
		if (iso_levels[forward_idx]>0)
			continue;
			
		mass_t sum = 2*peaks[forward_idx].mass + peaks[back_idx].mass;
		while (back_idx>=0 && sum>max_double_sum)
		{
			back_idx--;
			if (back_idx<0)
				break;
			sum = 2*peaks[forward_idx].mass + peaks[back_idx].mass;
		}

		if (back_idx>=0 && sum > min_double_sum)
		{
			if (iso_levels[back_idx]>0)
				continue;

			mass_t tol = fabs(sum - double_charge_pair_sum);
			if (tol<increment)
				tol = increment;

			const mass_t tol_sqr = tol*tol;
			const float inten_sum = peaks[forward_idx].intensity + peaks[back_idx].intensity;
				
			pmc_stats.num_c2_frag_pairs++;
			pmc_stats.inten_c2_pairs += inten_sum;
			pmc_stats.tol_c2_frag_pairs += tol;
			pmc_stats.tol_c2_sqr_frag_pairs += tol_sqr;

			if (strong_inds[forward_idx] || strong_inds[back_idx])
			{
				pmc_stats.num_strong_c2_frag_pairs++;
				pmc_stats.inten_c2_strong_pairs += inten_sum;
				pmc_stats.tol_strong_c2_frag_pairs += tol;
				pmc_stats.tol_strong_c2_sqr_frag_pairs += tol_sqr;
			}
		}
	}

	// normalize tolerances
	if (pmc_stats.num_c2_frag_pairs>0)
	{
		float one_over = 1.0 / (float)pmc_stats.num_c2_frag_pairs;
		pmc_stats.tol_c2_frag_pairs *= one_over;
		pmc_stats.tol_c2_sqr_frag_pairs *= one_over;
	}

	if (pmc_stats.num_strong_c2_frag_pairs>0)
	{
		float one_over = 1.0 / (float)pmc_stats.num_strong_c2_frag_pairs;
		pmc_stats.tol_strong_c2_frag_pairs *= one_over;
		pmc_stats.tol_strong_c2_sqr_frag_pairs *= one_over;
	}

	return pmc_stats;
}


/*******************************************************************************

  Calculates the average statistics observed for the background using different
  mass offsets

********************************************************************************/
void calc_background_stats(const mass_t single_charge_pair_sum, // the sum of b+y or c+z
						  mass_t increment,
						  Config *config,
						  const QCPeak *peaks, 
						  const int num_peaks,
						  const vector<bool>& strong_inds,
						  const vector<float>& iso_levels,
						  PmcStats& pmc_stats_total)
{
	const mass_t tolerance = config->get_tolerance();
	const mass_t offsets[]={-22.0,-10.0,-8.5,8.5,12.0,22.5};
	const int num_offsets = sizeof(offsets)/sizeof(mass_t);
	const float one_over_num_offsets = 1.0 / (float)num_offsets;

	int i;
	for (i=0; i<num_offsets; i++)
	{
		PmcStats pmc_stats = calc_pmc_stats_for_mass(peaks,num_peaks,
			single_charge_pair_sum+offsets[i], 1.5*tolerance,increment,iso_levels,strong_inds);

		if (i==0)
		{
			pmc_stats_total = pmc_stats;
		}
		else
		{
			pmc_stats_total.num_c2_frag_pairs        += pmc_stats.num_c2_frag_pairs;
			pmc_stats_total.num_frag_pairs           += pmc_stats.num_frag_pairs;
			pmc_stats_total.num_strong_c2_frag_pairs += pmc_stats.num_strong_c2_frag_pairs;
			pmc_stats_total.num_strong_frag_pairs	 += pmc_stats.num_strong_frag_pairs;

			pmc_stats_total.inten_frag_pairs		 += pmc_stats.inten_frag_pairs;
			pmc_stats_total.inten_strong_pairs		 += pmc_stats.inten_strong_pairs;
			pmc_stats_total.inten_c2_pairs			 += pmc_stats.inten_c2_pairs;
			pmc_stats_total.inten_c2_strong_pairs	 += pmc_stats.inten_c2_strong_pairs;

			pmc_stats_total.tol_c2_frag_pairs		 += pmc_stats.tol_c2_frag_pairs;
			pmc_stats_total.tol_c2_sqr_frag_pairs	 += pmc_stats.tol_c2_sqr_frag_pairs;
			pmc_stats_total.tol_frag_pairs			 += pmc_stats.tol_frag_pairs;
			pmc_stats_total.tol_sqr_frag_pairs		 += pmc_stats.tol_sqr_frag_pairs;
			pmc_stats_total.tol_strong_c2_frag_pairs += pmc_stats.tol_strong_c2_frag_pairs;
			pmc_stats_total.tol_strong_c2_sqr_frag_pairs += pmc_stats.tol_strong_c2_sqr_frag_pairs;
			pmc_stats_total.tol_strong_frag_pairs    += pmc_stats.tol_strong_frag_pairs;
			pmc_stats_total.tol_strong_sqr_frag_pairs+= pmc_stats.tol_strong_sqr_frag_pairs;		
		}
	}

	pmc_stats_total.num_c2_frag_pairs		 *= one_over_num_offsets;
	pmc_stats_total.num_frag_pairs			 *= one_over_num_offsets;
	pmc_stats_total.num_strong_c2_frag_pairs *= one_over_num_offsets;
	pmc_stats_total.num_strong_frag_pairs	 *= one_over_num_offsets;

	pmc_stats_total.inten_frag_pairs		 *= one_over_num_offsets;
	pmc_stats_total.inten_strong_pairs		 *= one_over_num_offsets;
	pmc_stats_total.inten_c2_pairs			 *= one_over_num_offsets;
	pmc_stats_total.inten_c2_strong_pairs	 *= one_over_num_offsets;

	pmc_stats_total.tol_c2_frag_pairs		 *= one_over_num_offsets;
	pmc_stats_total.tol_c2_sqr_frag_pairs	 *= one_over_num_offsets;
	pmc_stats_total.tol_frag_pairs			 *= one_over_num_offsets;
	pmc_stats_total.tol_sqr_frag_pairs		 *= one_over_num_offsets;
	pmc_stats_total.tol_strong_c2_frag_pairs *= one_over_num_offsets;
	pmc_stats_total.tol_strong_c2_sqr_frag_pairs *= one_over_num_offsets;
	pmc_stats_total.tol_strong_frag_pairs    *= one_over_num_offsets;
	pmc_stats_total.tol_strong_sqr_frag_pairs*= one_over_num_offsets;
}



/**********************************************************************
Tests the pm in increments of 0.1
***********************************************************************/
void fill_pmc_table_stats(int charge,
						  const mass_t single_charge_pair_sum, // the sum of b+y or c+z
						  mass_t minus_range, 
						  mass_t plus_range,
						  mass_t increment,
						  Config *config,
						  const BasicSpectrum& bs, 
						  const vector<bool>& strong_inds,
						  const vector<float>& iso_levels,
						  vector<PmcStats>& pmc_stats_vec)
{
	const mass_t org_m_over_z = bs.ssf->m_over_z;
	const mass_t tolerance = config->get_tolerance();
	const int num_bins_per_Da = (int)(1.0/increment);
	const int num_reserve_bins = (int)((plus_range-minus_range)*num_bins_per_Da)+3;
	const mass_t one_over_charge = 1.0/(mass_t)charge;

	pmc_stats_vec.clear();
	pmc_stats_vec.reserve(num_reserve_bins);

	mass_t delta;
	for (delta = minus_range; delta<=plus_range; delta+=increment)
	{
		PmcStats pmc_stats;
		

		pmc_stats = calc_pmc_stats_for_mass(bs.peaks,bs.num_peaks,single_charge_pair_sum+delta,
			tolerance,increment,iso_levels,strong_inds);

		pmc_stats.m_over_z = org_m_over_z + one_over_charge * (delta+0.5);

		pmc_stats_vec.push_back(pmc_stats);
	}
}



/*********************************************************************************
Fills the PMC sample features
Assumes the frag_pair_sum_offset is set.
Sets the PMC features and also creates vetors of features that are to be added
to the SQS sampels.
**********************************************************************************/
void PMCSQS_Scorer::calculate_curr_spec_pmc_values(const BasicSpectrum& bs,
												   mass_t increment)
{
	const mass_t m_over_z = bs.ssf->m_over_z;


	if (frag_pair_sum_offset<-999)
	{
		cout << "Error: must first set the expected frag pair offset!" << endl;
		exit(1);
	}

	int charge;
	for (charge=1; charge<=this->max_charge; charge++)
	{
		mass_t org_pm_with_19 = bs.ssf->m_over_z * charge - 1.0025*(charge - 1);

		mass_t frag_pair_sum = org_pm_with_19 + frag_pair_sum_offset;
	
		fill_pmc_table_stats( charge,
							  frag_pair_sum,
							 -3*charge, 
							  5*charge,
							  increment,
							  config,
							  bs,
							  curr_spec_strong_inds,
							  curr_spec_iso_levels,
							  curr_spec_pmc_tables[charge]);

		calc_background_stats(frag_pair_sum,0.1,config,bs.peaks,bs.num_peaks, 
			curr_spec_strong_inds, curr_spec_iso_levels, curr_spec_background_stats[charge]);

		// find maximal values
		int i;
		PmcStats& maximal = curr_spec_maximal_values[charge];
		maximal = curr_spec_pmc_tables[charge][0];

		for (i=1; i<curr_spec_pmc_tables[charge].size(); i++)
		{
			const PmcStats& curr_stats = curr_spec_pmc_tables[charge][i];

			// mathced pairs look for maximum number

			if (curr_stats.num_c2_frag_pairs>maximal.num_c2_frag_pairs)
				maximal.num_c2_frag_pairs = curr_stats.num_c2_frag_pairs;

			if (curr_stats.num_frag_pairs>maximal.num_frag_pairs)
				maximal.num_frag_pairs = curr_stats.num_frag_pairs;

			if (curr_stats.num_strong_frag_pairs>maximal.num_strong_frag_pairs)
				maximal.num_strong_frag_pairs = curr_stats.num_strong_frag_pairs;

			if (curr_stats.num_strong_c2_frag_pairs > maximal.num_strong_c2_frag_pairs)
				maximal.num_strong_c2_frag_pairs = curr_stats.num_strong_c2_frag_pairs;

			// intensity look for maximal numbers

			if (curr_stats.inten_frag_pairs > maximal.inten_frag_pairs)
				maximal.inten_frag_pairs = curr_stats.inten_frag_pairs;
			
			if (curr_stats.inten_strong_pairs> maximal.inten_strong_pairs)
				maximal.inten_strong_pairs = curr_stats.inten_strong_pairs;

			if (curr_stats.inten_c2_pairs> maximal.inten_c2_pairs)
				maximal.inten_c2_pairs = curr_stats.inten_c2_pairs;

			if (curr_stats.inten_c2_strong_pairs > maximal.inten_c2_strong_pairs)
				maximal.inten_c2_strong_pairs = curr_stats.inten_c2_strong_pairs;


			// tolerance sums look for minimum number

			if (curr_stats.tol_c2_frag_pairs<maximal.tol_c2_frag_pairs)
				maximal.tol_c2_frag_pairs = curr_stats.tol_c2_frag_pairs;

			if (curr_stats.tol_c2_sqr_frag_pairs<maximal.tol_c2_sqr_frag_pairs)
				maximal.tol_c2_sqr_frag_pairs = curr_stats.tol_c2_sqr_frag_pairs;

			if (curr_stats.tol_frag_pairs < maximal.tol_frag_pairs)
				maximal.tol_frag_pairs = curr_stats.tol_frag_pairs;

			if (curr_stats.tol_sqr_frag_pairs < maximal.tol_sqr_frag_pairs)
				maximal.tol_sqr_frag_pairs = curr_stats.tol_sqr_frag_pairs;

			if (curr_stats.tol_strong_c2_frag_pairs < maximal.tol_strong_c2_frag_pairs)
				maximal.tol_strong_c2_frag_pairs = curr_stats.tol_strong_c2_frag_pairs;

			if (curr_stats.tol_strong_c2_sqr_frag_pairs < maximal.tol_strong_c2_sqr_frag_pairs)
				maximal.tol_strong_c2_sqr_frag_pairs = curr_stats.tol_strong_c2_sqr_frag_pairs;

			if (curr_stats.tol_strong_frag_pairs < maximal.tol_strong_frag_pairs)
				maximal.tol_strong_frag_pairs = curr_stats.tol_strong_frag_pairs ;

			if (curr_stats.tol_strong_sqr_frag_pairs < maximal.tol_strong_sqr_frag_pairs)
				maximal.tol_strong_sqr_frag_pairs = curr_stats.tol_strong_sqr_frag_pairs;
		}

		// find indicators for min tolerance and max pairs
		float tol_pairs =           999999999;
		float tol_strong_pairs =    999999999;
		float tol_c2_pairs =        999999999;
		float tol_c2_strong_pairs = 999999999;

		int   idx_pairs=-1;
		int	  idx_strong_pairs=-1;
		int   idx_c2_pairs=-1;
		int   idx_c2_strong_pairs=-1;

		for (i=0; i<curr_spec_pmc_tables[charge].size(); i++)
		{
			const PmcStats& curr_stats = curr_spec_pmc_tables[charge][i];

			if (curr_stats.num_frag_pairs == maximal.num_frag_pairs &&
				curr_stats.tol_frag_pairs < tol_pairs)
			{
				idx_pairs=i;
				tol_pairs=curr_stats.tol_frag_pairs;
			}

			if (curr_stats.num_strong_frag_pairs == maximal.num_strong_frag_pairs &&
				curr_stats.tol_strong_frag_pairs < tol_strong_pairs)
			{
				idx_strong_pairs=i;
				tol_strong_pairs=curr_stats.tol_strong_frag_pairs;
			}

			if (curr_stats.num_c2_frag_pairs == maximal.num_c2_frag_pairs &&
				curr_stats.tol_c2_frag_pairs < tol_c2_pairs)
			{
				idx_c2_pairs=i;
				tol_c2_pairs=curr_stats.tol_c2_frag_pairs;
			}

			if (curr_stats.num_strong_c2_frag_pairs == maximal.num_strong_c2_frag_pairs &&
				curr_stats.tol_strong_c2_frag_pairs < tol_c2_strong_pairs)
			{
				idx_c2_strong_pairs=i;
				tol_c2_strong_pairs=curr_stats.tol_strong_c2_frag_pairs;
			}

		}

		curr_spec_pmc_tables[charge][idx_pairs].ind_pairs_with_min_tol=true;
		curr_spec_pmc_tables[charge][idx_strong_pairs].ind_strong_pairs_with_min_tol=true;
		curr_spec_pmc_tables[charge][idx_c2_pairs].ind_c2_pairs_with_min_tol=true;
		curr_spec_pmc_tables[charge][idx_c2_strong_pairs].ind_c2_strong_pairs_with_min_tol=true;


		static vector<float> log_distances;
		if (log_distances.size()<curr_spec_pmc_tables[charge].size())
		{
			log_distances.resize(curr_spec_pmc_tables[charge].size(),0);
			int i;
			for (i=1; i<log_distances.size(); i++)
				log_distances[i]=log(1.0+(float)i);
		}

		for (i=0; i<curr_spec_pmc_tables[charge].size(); i++)
		{
			PmcStats& curr_stats = curr_spec_pmc_tables[charge][i];
			curr_stats.log_dis_from_pairs_min_tol = log_distances[abs(i-idx_pairs)];
			curr_stats.log_dis_from_strong_pairs_min_tol = log_distances[abs(i-idx_strong_pairs)];
			curr_stats.log_dis_from_c2_pairs_min_tol = log_distances[abs(i-idx_c2_pairs)];
			curr_stats.log_dis_from_c2_strong_pairs_min_tol = log_distances[abs(i-idx_c2_strong_pairs)];
		}

	}
}
	


/*****************************************************************************************

******************************************************************************************/
void PMCSQS_Scorer::get_sqs_features_from_pmc_tables(const BasicSpectrum& bs,
						vector< vector<float> >& sqs_features) const
{
	const mass_t org_m_over_z = bs.ssf->m_over_z;
	float max_num_strong_pairs=0;
	int   best_table_idx=0;
	mass_t mz_diff = 99999;

	sqs_features.resize(this->max_charge+1);

	int charge;
	for (charge=1; charge<=this->max_charge; charge++)
	{
		// find entry which has the maximal number of strong pairs, while maining the minimial
		// m/z distance shift from the original
		int i;
		for (i=0; i<this->curr_spec_pmc_tables[charge].size(); i++)
		{
			const float curr_num_strong = curr_spec_pmc_tables[charge][i].num_strong_frag_pairs;
			
			if (curr_num_strong>=max_num_strong_pairs)
			{
				mass_t distance = fabs(curr_spec_pmc_tables[charge][i].m_over_z - org_m_over_z);
				if (curr_num_strong == max_num_strong_pairs && distance>= mz_diff)
					continue;
				
				max_num_strong_pairs = curr_num_strong;
				mz_diff = distance;
				best_table_idx = i;
			}
		}

		const PmcStats& best_stats = curr_spec_pmc_tables[charge][best_table_idx];
		int idx_off = (charge -1) * 4;

		sqs_features[charge].clear();

		sqs_features[charge].push_back(best_stats.num_frag_pairs);

		sqs_features[charge].push_back(best_stats.num_strong_frag_pairs);

		sqs_features[charge].push_back(best_stats.num_c2_frag_pairs);

		sqs_features[charge].push_back(best_stats.num_strong_c2_frag_pairs);
	}
}


	


/***********************************************************************

************************************************************************/
void PMCSQS_Scorer::fill_fval_vectors_with_PMC(const BasicSpectrum& bs, 
											   int charge,
											   vector<ME_Regression_Sample>& samples) const
{
	
	const vector<PmcStats>& pmc_stats = curr_spec_pmc_tables[charge];
	const PmcStats& best =  curr_spec_maximal_values[charge];
	const mass_t org_mz = bs.ssf->m_over_z;
	const int num_peaks = bs.num_peaks;
	const int num_strong = curr_spec_num_strong;


	samples.resize(pmc_stats.size());

	int i;
	for (i=0; i<pmc_stats.size(); i++)
	{
		const PmcStats& curr = pmc_stats[i];
		const mass_t curr_mz = pmc_stats[i].m_over_z;
		samples[i].label=0;
		samples[i].f_vals.clear();

		vector<fval>& fvals = samples[i].f_vals;

		fvals.push_back(fval(PMC_CONST,1.0));

		mass_t mass_diff = (curr_mz-org_mz)*charge;

		fvals.push_back(fval(PMC_DIFF_FROM_MEASURED_MZ,mass_diff));
		fvals.push_back(fval(PMC_ABS_DIFF_FROM_MEASURED_MZ,fabs(mass_diff)));

		if (mass_diff>-0.2 && mass_diff<0.2)
		{
			fvals.push_back(fval(PMC_IND_IS_P0,1.0));
		}
		else if (mass_diff>0.8 && mass_diff<1.2)
		{
			fvals.push_back(fval(PMC_IND_IS_P1,1.0));
		}
		else if (mass_diff>1.8 && mass_diff<2.2)
		{
			fvals.push_back(fval(PMC_IND_IS_P2,1.0)); 
		}
		else if (mass_diff>2.8 && mass_diff<3.2)
		{
			fvals.push_back(fval(PMC_IND_IS_P3,1.0));
		}
		else if (mass_diff<-0.8 && mass_diff>-1.2)
		{
			fvals.push_back(fval(PMC_IND_IS_M1,1.0));
		}
		else if (mass_diff<-1.8 && mass_diff>-2.2)
		{
			fvals.push_back(fval(PMC_IND_IS_M2,1.0));
		}
		else if (mass_diff<-2.8 && mass_diff>-3.2)
		{
			fvals.push_back(fval(PMC_IND_IS_M3,1.0));
		}
		else if (mass_diff<-3.8 && mass_diff>-4.2)
		{
			fvals.push_back(fval(PMC_IND_IS_M4,1.0));
		}
		


		// tolerance features
		if (1)
		{
			if (curr.num_frag_pairs<=0)
			{
				fvals.push_back(fval(PMC_IND_HAS_NO_PAIRS,1.0));
			}
			else
			{
				fvals.push_back(fval(PMC_IND_HAS_PAIRS,1.0));
				fvals.push_back(fval(PMC_NUM_FRAG_PAIRS,curr.num_frag_pairs));
				fvals.push_back(fval(PMC_AVG_TOL_OVER_PAIRS,curr.tol_frag_pairs));
				fvals.push_back(fval(PMC_AVG_TOL_SQR_OVER_PAIRS,curr.tol_sqr_frag_pairs));
				fvals.push_back(fval(PMC_AVG_DIFF_TOL_OVER_PAIRS,
					(curr.tol_frag_pairs - best.tol_frag_pairs) ));
				fvals.push_back(fval(PMC_AVG_DIFF_TOL_SQR_OVER_PAIRS,
					(curr.tol_sqr_frag_pairs - best.tol_sqr_frag_pairs)));
	
				fvals.push_back(fval(PMC_PROP_NUM_PAIRS,(float)curr.num_frag_pairs/num_peaks));
				fvals.push_back(fval(PMC_PROP_INTEN_PAIRS,curr.inten_frag_pairs/curr_spec_total_intensity));
	
				fvals.push_back(fval(PMC_REL_PROP_NUM_PAIRS,(float)curr.num_frag_pairs/(float)best.num_frag_pairs));
				fvals.push_back(fval(PMC_REL_PROP_INTEN_PAIRS,curr.inten_frag_pairs/best.inten_frag_pairs));
				
				if (curr.num_frag_pairs == best.num_frag_pairs)
					fvals.push_back(fval(PMC_IND_BEST_NUM_FRAG_PAIRS,1.0));
				
				if (curr.ind_pairs_with_min_tol)
				{
					fvals.push_back(fval(PMC_IND_PAIRS_AND_MIN_TOLERANCE,1.0));
				}
				else
				{
					fvals.push_back(fval(PMC_LOG_DIS_PAIRS_MIN_TOL,curr.log_dis_from_pairs_min_tol));
				}
			}

			if (curr.num_strong_frag_pairs<=0)
			{
				fvals.push_back(fval(PMC_IND_HAS_NO_STRONG_PAIRS,1.0));
			}
			else
			{
				fvals.push_back(fval(PMC_IND_HAS_STRONG_PAIRS,1.0));
				fvals.push_back(fval(PMC_NUM_STRONG_FRAG_PAIRS,curr.num_strong_frag_pairs));
				fvals.push_back(fval(PMC_AVG_TOL_OVER_STRONG_PAIRS,curr.tol_strong_frag_pairs));
				fvals.push_back(fval(PMC_AVG_TOL_SQR_OVER_STRONG_PAIRS,curr.tol_strong_sqr_frag_pairs));
				fvals.push_back(fval(PMC_AVG_DIFF_TOL_OVER_STRONG_PAIRS,
					(curr.tol_strong_frag_pairs - best.tol_strong_frag_pairs) ));
				fvals.push_back(fval(PMC_AVG_DIFF_TOL_SQR_OVER_STRONG_PAIRS,
					(curr.tol_strong_sqr_frag_pairs - best.tol_strong_sqr_frag_pairs) ));
				
				fvals.push_back(fval(PMC_PROP_NUM_STRONG_PAIRS,(float)curr.num_strong_frag_pairs/num_strong));
				fvals.push_back(fval(PMC_PROP_INTEN_STRONG_PAIRS,curr.inten_strong_pairs/curr_spec_strong_intensity));

				fvals.push_back(fval(PMC_REL_PROP_NUM_STRONG_PAIRS,(float)curr.num_strong_frag_pairs/(float)best.num_frag_pairs));
				fvals.push_back(fval(PMC_REL_PROP_INTEN_STRONG_PAIRS,curr.inten_strong_pairs/best.inten_strong_pairs));

				if (best.num_strong_frag_pairs == curr.num_strong_frag_pairs)
				{
					fvals.push_back(fval(PMC_IND_BEST_NUM_STRONG_FRAG_PAIRS,1.0));
					if (curr.num_frag_pairs == best.num_frag_pairs)
						fvals.push_back(fval(PMC_IND_BEST_BOTH_PAIRS,1.0));
				}

				if (curr.ind_strong_pairs_with_min_tol)
				{
					fvals.push_back(fval(PMC_IND_STRONG_PAIRS_AND_MIN_TOLERANCE,1.0));
				}
				else
				{
					fvals.push_back(fval(PMC_LOG_DIS_STRONG_PAIRS_MIN_TOL,curr.log_dis_from_strong_pairs_min_tol));
				}
			}

			if (curr.num_c2_frag_pairs<=0)
			{
				fvals.push_back(fval(PMC_IND_HAS_NO_C2_PAIRS,1.0));
			}
			else
			{
				if (curr.num_frag_pairs>0)
					fvals.push_back(fval(PMC_IND_HAS_BOTH_PAIRS,1.0));

				fvals.push_back(fval(PMC_IND_HAS_C2_PAIRS,1.0));
				fvals.push_back(fval(PMC_NUM_C2_FRAG_PAIRS,curr.num_c2_frag_pairs));
				fvals.push_back(fval(PMC_AVG_TOL_OVER_C2_PAIRS,curr.tol_c2_frag_pairs));
				fvals.push_back(fval(PMC_AVG_TOL_SQR_OVER_C2_PAIRS,curr.tol_c2_sqr_frag_pairs));
				fvals.push_back(fval(PMC_AVG_DIFF_TOL_OVER_C2_PAIRS,
					(curr.tol_c2_frag_pairs - best.tol_c2_frag_pairs) ));
				fvals.push_back(fval(PMC_AVG_DIFF_TOL_SQR_OVER_C2_PAIRS,
					(curr.tol_c2_sqr_frag_pairs - best.tol_c2_sqr_frag_pairs) ));

				fvals.push_back(fval(PMC_PROP_NUM_C2_PAIRS,(float)curr.num_c2_frag_pairs/(float)num_peaks));
				fvals.push_back(fval(PMC_PROP_INTEN_C2_PAIRS,curr.inten_c2_pairs/curr_spec_total_intensity));

				fvals.push_back(fval(PMC_REL_PROP_NUM_C2_PAIRS,(float)curr.num_c2_frag_pairs/(float)best.num_c2_frag_pairs));
				fvals.push_back(fval(PMC_REL_PROP_INTEN_C2_PAIRS,curr.inten_c2_pairs/best.inten_c2_pairs));

				if (curr.num_c2_frag_pairs == best.num_c2_frag_pairs)				
					fvals.push_back(fval(PMC_IND_BEST_NUM_C2_FRAG_PAIRS,1.0));

				if (curr.ind_c2_pairs_with_min_tol)
				{
					fvals.push_back(fval(PMC_IND_C2_PAIRS_AND_MIN_TOLERANCE,1.0));
				}
				else
				{
					fvals.push_back(fval(PMC_LOG_DIS_C2_PAIRS_MIN_TOL,curr.log_dis_from_c2_pairs_min_tol));	
				}
			}

			if (curr.num_strong_c2_frag_pairs<=0)
			{
				fvals.push_back(fval(PMC_IND_HAS_NO_C2_STRONG_PAIRS,1.0));
			}
			else
			{
				if (curr.num_strong_frag_pairs>0)
					fvals.push_back(fval(PMC_IND_HAS_BOTH_STRONG_PAIRS,1.0));


				fvals.push_back(fval(PMC_IND_HAS_C2_STRONG_PAIRS,1.0));
				fvals.push_back(fval(PMC_NUM_STRONG_C2_FRAG_PAIRS,curr.num_strong_c2_frag_pairs));
				fvals.push_back(fval(PMC_AVG_TOL_OVER_C2_STRONG_PAIRS,curr.tol_strong_c2_frag_pairs));
				fvals.push_back(fval(PMC_AVG_TOL_SQR_OVER_C2_STRONG_PAIRS,curr.tol_strong_c2_sqr_frag_pairs));
				fvals.push_back(fval(PMC_AVG_DIFF_TOL_OVER_C2_STRONG_PAIRS,
					(curr.tol_strong_c2_frag_pairs - best.tol_strong_c2_frag_pairs) ));
				fvals.push_back(fval(PMC_AVG_DIFF_TOL_SQR_OVER_C2_STRONG_PAIRS,
					(curr.tol_strong_c2_sqr_frag_pairs - best.tol_strong_c2_sqr_frag_pairs) ));
				
				fvals.push_back(fval(PMC_PROP_NUM_C2_STRONG_PAIRS,(float)curr.num_strong_c2_frag_pairs/curr_spec_num_strong));
				fvals.push_back(fval(PMC_PROP_INTEN_C2_STRONG_PAIRS,curr.inten_c2_strong_pairs/curr_spec_strong_intensity));

				fvals.push_back(fval(PMC_REL_PROP_NUM_C2_STRONG_PAIRS,(float)curr.num_strong_c2_frag_pairs/(float)best.num_strong_c2_frag_pairs));
				fvals.push_back(fval(PMC_REL_PROP_INTEN_C2_STRONG_PAIRS,curr.inten_c2_strong_pairs/best.inten_c2_strong_pairs));

				if (curr.num_strong_c2_frag_pairs == best.num_strong_c2_frag_pairs)
				{
					fvals.push_back(fval(PMC_IND_BEST_NUM_STRONG_C2_FRAG_PAIRS,1.0));
					if (curr.num_c2_frag_pairs == best.num_c2_frag_pairs)
						fvals.push_back(fval(PMC_IND_BEST_BOTH_C2_PAIRS,1.0));				
				}

				
				if (curr.ind_c2_strong_pairs_with_min_tol)
				{
					fvals.push_back(fval(PMC_IND_C2_STRONG_PAIRS_AND_MIN_TOLERANCE,1.0));
				}
				else
				{
					fvals.push_back(fval(PMC_LOG_DIS_C2_STRONG_PAIRS_MIN_TOL,curr.log_dis_from_c2_strong_pairs_min_tol));
				}
			}
		}


	//	"PMC_IND_PAIRS_AND_MIN_TOLERANCE",		"PMC_IND_STRONG_PAIRS_AND_MIN_TOLERANCE",
	//	"PMC_IND_C2_PAIRS_AND_MIN_TOLERANCE",	"PMC_IND_C2_STRONG_PAIRS_AND_MIN_TOLERANCE",

		sort(fvals.begin(),fvals.end());
	}


	

	/*


	// values in raw numbers
	PMC_NUM_FRAG_PAIRS,				PMC_NUM_STRONG_FRAG_PAIRS,
	PMC_INTEN_FRAG_PAIRS,			PMC_INTEN_STRONG_FRAG_PAIRS,
	PMC_NUM_C2_FRAG_PAIRS,			PMC_NUM_STRONG_C2_FRAG_PAIRS,
	PMC_INTEN_C2_FRAG_PAIRS,		PMC_INTEN_STRONG_C2_FRAG_PAIRS,

	// indicators
	PMC_IND_BEST_NUM_FRAG_PAIRS,		PMC_IND_BEST_NUM_STRONG_FRAG_PAIRS,
	PMC_IND_BEST_BOTH_PAIRS,
	
	PMC_IND_BEST_NUM_C2_FRAG_PAIRS,		PMC_IND_BEST_NUM_STRONG_C2_FRAG_PAIRS,
	PMC_IND_BEST_BOTH_C2_PAIRS,


	// values as proportions (of num peaks / num strong peaks)
	PMC_PROP_FRAG_PAIRS,				PMC_PROP_STRONG_FRAG_PAIRS,
	PMC_PROP_INTEN_FRAG_PAIRS,			PMC_PROP_INTEN_STRONG_FRAG_PAIRS,
	PMC_PROP_C2_FRAG_PAIRS,				PMC_PROP_STRONG_C2_FRAG_PAIRS,
	PMC_PROP_INTEN_C2_FRAG_PAIRS,		PMC_PROP_INTEN_C2_STRONG_FRAG_PAIRS,

	// compared with optimal value measured for all mass values

	PMC_REL_PROP_FRAG_PAIRS,				PMC_REL_PROP_STRONG_FRAG_PAIRS,
	PMC_REL_PROP_INTEN_FRAG_PAIRS,			PMC_REL_PROP_INTEN_STRONG_FRAG_PAIRS,
	PMC_REL_PROP_C2_FRAG_PAIRS,				PMC_REL_PROP_STRONG_C2_FRAG_PAIRS,
	PMC_REL_PROP_INTEN_FRAG_C2_PAIRS,		PMC_REL_PROP_INTEN_STRONG_C2_FRAG_PAIRS,

	PMC_NUM_FIELDS 
	*/

}



