#include "QuickClustering.h"
#include "auxfun.h"
#include "AnnotatedSpectrum.h"




void mark_top_peaks_with_sliding_window(const QCPeak *peaks, 
										int num_peaks, 
										mass_t window_size, 
										int num_peaks_per_window, 
										vector<bool>& indicators) 
{
	// filter low intensity noise
	// and mark those that are good peaks
	const mass_t half_window_size = 0.5 * window_size;
	int max_peak_idx = num_peaks -1;
	int min_idx=1;
	int max_idx=1;
	int p_idx =1;

	indicators.clear();
	indicators.resize(num_peaks,false);

	bool use_scaled_intensity = true;

	int i;
	for (i=0; i<5 && i<max_peak_idx; i++)
	{
		if (peaks[i].scaled_intensity<=0)
			break;
	}
	if (i<5 && i<max_peak_idx)
		use_scaled_intensity=false;

	indicators[0] = true;
	for (i=1; i<max_peak_idx; i++)
	{
		const mass_t& peak_mass=peaks[i].mass;
		mass_t min_mass = peaks[min_idx].mass;
		mass_t max_mass = peaks[max_idx].mass;

		// advance min/max pointers
		while (peak_mass-min_mass > half_window_size)
			min_mass=peaks[++min_idx].mass;

		while (max_idx < max_peak_idx && max_mass - peak_mass <= half_window_size)
			max_mass=peaks[++max_idx].mass;

		if (max_mass - peak_mass > half_window_size)
			max_idx--;

		// if there are less than the maximum number of peaks in the window, keep it.
		if (max_idx-min_idx < num_peaks_per_window)
		{
			indicators[i]=true;
			continue;
		}

		// check if this is one of the top peaks in the window
		int higher_count=0;
		if (use_scaled_intensity)
		{
			for (int j=min_idx; j<=max_idx; j++)
				if (peaks[j].scaled_intensity > peaks[i].scaled_intensity)
					higher_count++;
		}
		else
			for (int j=min_idx; j<=max_idx; j++)
				if (peaks[j].intensity > peaks[i].intensity)
					higher_count++;

		if (higher_count < num_peaks_per_window)
			indicators[i]=true;
	}
	indicators[i]=true;
}




void BasicSpectrum::output_to_mgf(ostream& mgf, Config *config, const char *seq) const
{
	mgf << "BEGIN IONS" << endl;
	mgf << "TITLE=" <<  ssf->single_name << endl;
	
	if (ssf->peptide.get_num_aas()>0)
	{
		mgf << "SEQ=" << ssf->peptide.as_string(config) << endl;
	}
	else if (seq && strlen(seq)>2)
		mgf << "SEQ=" << seq << endl;
	
	if (ssf->type == MZXML)
	{
		MZXML_single *mzxml_single = (MZXML_single *)ssf;
		if (mzxml_single->scan_number>=0)
			mgf << "SCAN=" <<mzxml_single->scan_number << endl;

//		if (mzxml_single->retention_time>=0)
//			mgf << "RT=" << mzxml_single->retention_time << endl;
	}

	mgf << "CHARGE=+" << ssf->charge << endl;
		
	mgf << "PEPMASS=" << ssf->m_over_z << endl;
	
	int i;
	for (i=0; i<this->num_peaks; i++)
		mgf << fixed << setprecision(3) << peaks[i].mass << " " << peaks[i].intensity << endl;

	mgf << "END IONS" << endl << endl;
}






