#include "spectrum.h"
#include "label.h"
#include "batch.h"

#include <iostream>
#include <fstream>
#include <cmath>
#include <cstdio>
#include <list>

//
// **************************************************************
//    Spectrum methods
// **************************************************************
//

Spectrum::~Spectrum() { 
  if(info!=(char *)0) free(info); info=(char *)0; 
}

//
//  Finds peaks in the spectrum inside a [-tolerance, tolerance] window around
//    targetMass. startIdx is a hint about where to start looking for targetMass
//    to avoid having to do binary search to find the peak closest to mass.
//    If the return set is empty then minIdx=-1 and maxIdx=-2;
//
inline void Spectrum::locateWithinTolerance(int startIdx, float targetMass, float tolerance, int &minIdx, int &maxIdx) {
	minIdx=-1; maxIdx=-2;
	if(peakList[0][0]>=targetMass-tolerance) minIdx=0; else {
		// Find first peak after targetMass
		for(minIdx=startIdx; minIdx<(int)peakList.size() && peakList[minIdx][0]<targetMass-tolerance; minIdx++);
		if(minIdx==(int)peakList.size()) { minIdx=-1; return; }
		// Now find leftmost peak >= targetMass-tolerance
		for(; minIdx>0 && peakList[minIdx-1][0]>=targetMass-tolerance; minIdx--);
	}
	if(peakList[minIdx][0]>targetMass+tolerance) { minIdx=-1; return; }
	if(peakList[peakList.size()-1][0]<=targetMass+tolerance) maxIdx=peakList.size()-1;
	else for(maxIdx=minIdx; maxIdx<(int)peakList.size()-1 && peakList[maxIdx+1][0]<=targetMass+tolerance; maxIdx++);
}

inline void Spectrum::weightedAverage(int minIdx, int maxIdx, float &avgMass, float &totScore) {
	int i=0; totScore=0; avgMass=0;
	for(i=minIdx; i<=maxIdx; i++) totScore+=peakList[i][1];
	if(totScore<=0) for(i=minIdx; i<=maxIdx; i++) avgMass+=peakList[i][0]/(maxIdx-minIdx+1);  // regular average
	else for(i=minIdx; i<=maxIdx; i++) avgMass+=peakList[i][0]*peakList[i][1]/totScore;       // weighted average
}

Spectrum &Spectrum::operator=(const Spectrum &other) {
	parentMass=other.parentMass;   parentCharge=other.parentCharge;   scan=other.scan;
	resolution=other.resolution;   idDist=other.idDist;
	peakList.resize(other.peakList.size());
	for (unsigned int i=0; i<other.peakList.size(); i++) peakList[i]=other.peakList[i];
	return(*this);
}

void Spectrum::changeResolution(float newResolution, bool enforceRounding) { 
	resolution=newResolution;   idDist=1/resolution; 
	parentMass = enforceRounding?round(parentMass/resolution):parentMass/resolution;
	if(!enforceRounding) for(unsigned int i=0; i<peakList.size(); i++) peakList[i][0]=peakList[i][0]/resolution;
	else for(unsigned int i=0; i<peakList.size(); i++) peakList[i][0]=round(peakList[i][0]/resolution);
}

// addZPMpeaks - Ensures that the spectrum contains the ions b0/bk/y0/yk where k is the
//                 number of amino acids in the corresponding peptide. New peaks are added
//                 with a score of 0.1 if the b/y masses are not already present.
//
// tolerance - mass tolerance used when looking for b0/bk/y0/yk in the current spectrum
// ionOffest - used to specify b0/bk/y0/yk mass offset, depending on type of spectrum.
//               Use 0 for PRM spectra and AAJumps::massHion for MS/MS spectra
// includeY0k - If true then y0,yk are also added to the spectrum
// labels    - if !=NULL then the labels are shifted to match the correct peaks after any peak insertions
//
void Spectrum::addZPMpeaks(float tolerance, float ionOffset, bool includeY0k, SpectrumPeakLabels *labels){
	float massB0 = ionOffset*idDist, 
	      massBk = parentMass-(AAJumps::massMH-ionOffset)*idDist, 
	      massY0 = (AAJumps::massH2O+ionOffset)*idDist, 
	      massYk = parentMass-(AAJumps::massHion-ionOffset)*idDist;

	list<TwoValues<float> > newPeaks;
	list<int> prevIndices;
	TwoValues<float> tmp(0,0);
	unsigned int pivot;   bool valIsPresent=false;
	for(pivot=0; pivot<peakList.size() and peakList[pivot][0]<=massB0+tolerance; pivot++) 
		{ newPeaks.push_back(peakList[pivot]); prevIndices.push_back(pivot); valIsPresent=true; }
	if(!valIsPresent) { newPeaks.push_back(tmp.set(massB0,0.1)); prevIndices.push_back(-1); } valIsPresent=false;
	if(includeY0k) {
		for(; pivot<peakList.size() and peakList[pivot][0]<massY0-tolerance; pivot++) { newPeaks.push_back(peakList[pivot]); prevIndices.push_back(pivot); }
		for(; pivot<peakList.size() and peakList[pivot][0]<=massY0+tolerance; pivot++) { newPeaks.push_back(peakList[pivot]); prevIndices.push_back(pivot); valIsPresent=true; }
		if(!valIsPresent) { newPeaks.push_back(tmp.set(massY0,0.1)); prevIndices.push_back(-1); } valIsPresent=false;
	}
	for(; pivot<peakList.size() and peakList[pivot][0]<massBk-tolerance; pivot++) { newPeaks.push_back(peakList[pivot]); prevIndices.push_back(pivot); }
	for(; pivot<peakList.size() and peakList[pivot][0]<=massBk+tolerance; pivot++) { newPeaks.push_back(peakList[pivot]); prevIndices.push_back(pivot); valIsPresent=true; }
	if(!valIsPresent) { newPeaks.push_back(tmp.set(massBk,0.1)); prevIndices.push_back(-1); } valIsPresent=false;
	if(includeY0k) {
		for(; pivot<peakList.size() and peakList[pivot][0]<massYk-tolerance; pivot++) { newPeaks.push_back(peakList[pivot]); prevIndices.push_back(pivot); }
		for(; pivot<peakList.size() and peakList[pivot][0]<=massYk+tolerance; pivot++) { newPeaks.push_back(peakList[pivot]); prevIndices.push_back(pivot); valIsPresent=true; }
		if(!valIsPresent) { newPeaks.push_back(tmp.set(massYk,0.1)); prevIndices.push_back(-1); } valIsPresent=false;
	}

	if(labels!=0) { labels->resize(newPeaks.size());   int labelsIdx=labels->size()-1;
		for(list<int>::reverse_iterator iter = prevIndices.rbegin();iter != prevIndices.rend(); iter++, labelsIdx--)
			if(*iter>=0) (*labels)[labelsIdx]=(*labels)[*iter]; else (*labels)[labelsIdx].set(0,0,0,0,1);
	}

	peakList.resize(newPeaks.size());   list<TwoValues<float> >::iterator iter = newPeaks.begin();
	for(unsigned int i=0; i<peakList.size(); i++,iter++) peakList[i]=*iter;
}

void Spectrum::maximizeZPMpeaks(float tolerance, float ionOffset, bool includeY0k){
	float massB0 = ionOffset*idDist, 
	      massBk = parentMass-(AAJumps::massMH-ionOffset)*idDist, 
	      massY0 = (AAJumps::massH2O+ionOffset)*idDist, 
	      massYk = parentMass-(AAJumps::massHion-ionOffset)*idDist;

	unsigned int pivot;
	float maxScore=0; for(pivot=0; pivot<peakList.size(); pivot++) if(peakList[pivot][1]>maxScore) maxScore=peakList[pivot][1];
	for(pivot=0; pivot<peakList.size() and peakList[pivot][0]<=massB0+tolerance; pivot++) peakList[pivot][1] = maxScore;
	if(includeY0k) {
		for(; pivot<peakList.size() and peakList[pivot][0]<massY0-tolerance; pivot++);
		for(; pivot<peakList.size() and peakList[pivot][0]<=massY0+tolerance; pivot++) peakList[pivot][1] = maxScore;
	}
	for(; pivot<peakList.size() and peakList[pivot][0]<massBk; pivot++);
	for(; pivot<peakList.size() and peakList[pivot][0]<=massBk; pivot++) peakList[pivot][1] = maxScore;
	if(includeY0k) {
		for(; pivot<peakList.size() and peakList[pivot][0]<massYk-tolerance; pivot++);
		for(; pivot<peakList.size() and peakList[pivot][0]<=massYk+tolerance; pivot++) peakList[pivot][1] = maxScore;
	}
}

void Spectrum::normalize(bool removeNegatives) { // Normalizes total intensity to 100
	float totalIntensity=0;
	for(unsigned int i=0; i<peakList.size(); i++) totalIntensity+=peakList[i][1];
	for(unsigned int i=0; i<peakList.size(); i++) peakList[i][1] = 100*peakList[i][1]/totalIntensity;
}

//
// findMatches - Find all peaks within tolerance of baseMass
//
//  If startIdx is >=0 then the search starts at startIdx. Function returns the number of peaks within tolerance.
//
short Spectrum::findMatches(float baseMass, float peakTol, vector<int> &matchesIdx, int startIdx){
	unsigned int baseIdx=(startIdx>=0 and startIdx<peakList.size())?(unsigned int)startIdx:0,
	             numPeaks = peakList.size(), matchCount=0;
	matchesIdx.resize(numPeaks);
	for(; baseIdx>0 and peakList[baseIdx][0]>=baseMass-peakTol; baseIdx--);  
	for(; baseIdx<numPeaks and peakList[baseIdx][0]<baseMass-peakTol; baseIdx++);  // Get to first peak above the lower end of the tolerance window
	for(; baseIdx<numPeaks and peakList[baseIdx][0]<=baseMass+peakTol; baseIdx++)
		matchesIdx[matchCount++]=baseIdx;
	matchesIdx.resize(matchCount);   return matchCount;
}

//
//  findClosest - Finds the spectrum peak with peak mass closest to mass
// 
int Spectrum::findClosest(float mass) {
	if(peakList.size()==0) return -1;

	int lowerLim=0, upperLim=(int)peakList.size()-1, middle;
	while(upperLim>lowerLim+1) {  // Set lowerLim to the index of the largest monoisotopic mass < mass
		middle=(lowerLim+upperLim)/2;
		if(peakList[middle][0]>mass) upperLim=middle; else lowerLim=middle;
	}
	if(fabs(peakList[lowerLim][0]-mass)<fabs(peakList[upperLim][0]-mass)) return lowerLim; else return upperLim;
}


void Spectrum::mergePeakList(vector<TwoValues<float> > &newPeaks, Spectrum *putHere) {
    int idxUpdated, idxOld, idxNew;
    vector<TwoValues<float> > &updatedPeakList = (putHere==0)?peakList:putHere->peakList;
    vector<TwoValues<float> > *oldPeakList;

    if(putHere==0) {
        oldPeakList = new vector<TwoValues<float> >;
        oldPeakList->resize(peakList.size()); for(int i=0; i<peakList.size(); i++) (*oldPeakList)[i]=peakList[i];
    } else {
        oldPeakList = &peakList;  // No copying necessary if results goes into another object (putHere)
        putHere->parentMass = parentMass;   putHere->parentCharge = parentCharge;
    }

    // Sorted merge
    updatedPeakList.resize(oldPeakList->size()+newPeaks.size());
    idxUpdated=0; idxOld=0; idxNew=0;
    while(idxOld<oldPeakList->size() or idxNew<newPeaks.size()) {
        if(idxOld==oldPeakList->size()) { updatedPeakList[idxUpdated++]=newPeaks[idxNew++]; continue; }
        if(idxNew==newPeaks.size()) { updatedPeakList[idxUpdated++]=(*oldPeakList)[idxOld++]; continue; }

        if(fabs((*oldPeakList)[idxOld][0]-newPeaks[idxNew][0])<0.0001) {
            updatedPeakList[idxUpdated][0]=(*oldPeakList)[idxOld][0];
            updatedPeakList[idxUpdated++][1]=(*oldPeakList)[idxOld++][1]+newPeaks[idxNew++][1];
        } else if ((*oldPeakList)[idxOld][0]<newPeaks[idxNew][0]) updatedPeakList[idxUpdated++]=(*oldPeakList)[idxOld++];
        else updatedPeakList[idxUpdated++]=newPeaks[idxNew++];
    }
    updatedPeakList.resize(idxUpdated);  // Just in case some peaks get merged along the way

    if(putHere==0) delete oldPeakList;
}

void Spectrum::mergePeakListRev(vector<TwoValues<float> > &newPeaks, Spectrum *putHere) {
    int idxUpdated, idxOld, idxNew;
    vector<TwoValues<float> > &updatedPeakList = (putHere==0)?peakList:putHere->peakList;
    vector<TwoValues<float> > *oldPeakList;

    if(putHere==0) {
        oldPeakList = new vector<TwoValues<float> >;
        oldPeakList->resize(peakList.size()); for(int i=0; i<peakList.size(); i++) (*oldPeakList)[i]=peakList[i];
    } else {
        oldPeakList = &peakList;  // No copying necessary if results goes into another object (putHere)
        putHere->parentMass = parentMass;   putHere->parentCharge = parentCharge;
    }

    float aaMass = parentMass-19;

    // Sorted merge of reversed peaks / endpoints
    updatedPeakList.resize(oldPeakList->size()+newPeaks.size());
    idxUpdated=0; idxOld=oldPeakList->size()-1; idxNew=newPeaks.size()-1;
    while(idxOld>=0 or idxNew>=0) {
        if(idxOld<0) { updatedPeakList[idxUpdated++].set(aaMass-newPeaks[idxNew][0],newPeaks[idxNew--][1]); continue; }
        if(idxNew<0) { updatedPeakList[idxUpdated++].set(aaMass+18-(*oldPeakList)[idxOld][0],(*oldPeakList)[idxOld--][1]); continue; }  // +18 because PRMs are assumed to be from y-ions
        if(fabs((aaMass+18-(*oldPeakList)[idxOld][0])-(aaMass-newPeaks[idxNew][0]))<0.0001) {
            updatedPeakList[idxUpdated++].set(max((float)0,aaMass+18-(*oldPeakList)[idxOld][0]),(*oldPeakList)[idxOld--][1]+newPeaks[idxNew--][1]);
        } else if ((aaMass+18-(*oldPeakList)[idxOld][0])<(aaMass-newPeaks[idxNew][0]))
                { updatedPeakList[idxUpdated++].set(max((float)0,aaMass+18-(*oldPeakList)[idxOld][0]),(*oldPeakList)[idxOld--][1]); }
        else { updatedPeakList[idxUpdated++].set(max((float)0,aaMass-newPeaks[idxNew][0]),newPeaks[idxNew--][1]); }
    }
    updatedPeakList.resize(idxUpdated);  // Just in case some peaks get merged along the way

    if(putHere==0) delete oldPeakList;
}

// Converts a list of masses to a list of corresponding spectrum peak indices (closest mass)
//  masses is assumed to be sorted by increasing mass values.
//
void Spectrum::massesToIndices(vector<TwoValues<float> > &masses, vector<int> &indices, float peakTol) {
	int idxClosest, idxPeaks=0, idxMasses;   float distClosest=peakTol+1; // Supremum for distance to closest peak
	indices.resize(masses.size());
	for(idxMasses=0; idxMasses<(int)masses.size(); idxMasses++) {
		while(idxPeaks>0 and peakList[idxPeaks][0]>masses[idxMasses][0]-peakTol) idxPeaks--;
		while(idxPeaks<(int)peakList.size() and peakList[idxPeaks][0]<masses[idxMasses][0]-peakTol) idxPeaks++;
		idxClosest=-1; distClosest=peakTol+1;
		while(idxPeaks<(int)peakList.size() and abs(peakList[idxPeaks][0]-masses[idxMasses][0])<=peakTol) {
			if (abs(peakList[idxPeaks][0]-masses[idxMasses][0])<distClosest)
				{ idxClosest=idxPeaks; distClosest=abs(peakList[idxPeaks][0]-masses[idxMasses][0]); }
			idxPeaks++;
		}
		if(idxClosest>=0) indices[idxMasses]=idxClosest; else indices[idxMasses]=-1;
		idxPeaks = min(idxPeaks,(int)peakList.size()-1);
	}
}

void Spectrum::selectIndices(vector<int> &idx) {
	unsigned int i;
	for(i=0; i<idx.size(); i++)
		if(idx[i]<peakList.size()) peakList[i]=peakList[idx[i]]; else break;
	peakList.resize(i);
}

void Spectrum::output(ostream &output) {
    output << parentMass << " " << parentCharge << endl;
    for(int i=0; i<peakList.size(); i++) output << peakList[i][0] << " " << peakList[i][1] << endl;
}

void Spectrum::output_ms2(ostream &output) {
    output << ":0.0.0\n";
    output << parentMass << " " << parentCharge << endl;
    for(int i=0; i<peakList.size(); i++) output << peakList[i][0] << " " << peakList[i][1] << endl;
}

//
//  Computes the reversed spectrum and returns it in putHere (or reverses current if putHere==NULL)
//    The reverse of a peak with mass m is parentMass-1+pmOffset-m :
//    use pmOffset=0 for PRM spectra and pmOffset=2 for MS/MS spectra.
//
void Spectrum::reverse(float pmOffset, Spectrum *putHere) {
    vector<TwoValues<float> > &updatedPeakList = (putHere==0)?peakList:putHere->peakList;
    vector<TwoValues<float> > *oldPeakList;

    if(putHere==0) {
        oldPeakList = new vector<TwoValues<float> >;
        oldPeakList->resize(peakList.size()); for(unsigned int i=0; i<peakList.size(); i++) (*oldPeakList)[i]=peakList[i];
    } else {
        oldPeakList = &peakList;  // No copying necessary if results goes into another object (putHere)
        (*putHere) = *this;
    }
    
    float totMass = parentMass-AAJumps::massHion+pmOffset;   unsigned int numPeaks = oldPeakList->size();
	for(unsigned int i=0; i<numPeaks; i++)
		updatedPeakList[numPeaks-i-1].set(totMass-(*oldPeakList)[i][0],(*oldPeakList)[i][1]);

    if(putHere==0) delete oldPeakList;
}

//
//  Selects the top k peaks in the spectrum.
//
void Spectrum::selectTopK(unsigned int topK, Spectrum *putHere) {
    vector<TwoValues<float> > &updatedPeakList = (putHere==0)?peakList:putHere->peakList;
    vector<TwoValues<float> > *oldPeakList;
	unsigned int peakIdx;

    if(putHere==0) {
        oldPeakList = new vector<TwoValues<float> >;
        oldPeakList->resize(peakList.size()); for(unsigned int i=0; i<peakList.size(); i++) (*oldPeakList)[i]=peakList[i];
    } else {
        oldPeakList = &peakList;  // No copying necessary if results goes into another object (putHere)
        (*putHere) = *this;
    }
	vector<TwoValues<float> > sortedInts(oldPeakList->size());

    if(topK<oldPeakList->size()) {
		for(peakIdx=0; peakIdx<oldPeakList->size(); peakIdx++) sortedInts[peakIdx].set((*oldPeakList)[peakIdx][1],peakIdx);
		sort(sortedInts.begin(),sortedInts.end());
		
		updatedPeakList.resize(topK);
		for(peakIdx=1; peakIdx<=topK; peakIdx++)
			updatedPeakList[peakIdx-1] = (*oldPeakList)[(unsigned int)sortedInts[sortedInts.size()-peakIdx][1]];
		
		sort(updatedPeakList.begin(),updatedPeakList.end());
    }
    
    if(putHere==0) delete oldPeakList;
}

//  a) Rounds peak masses to the closest integer (after dividing by resolution)
//  b) Resulting peak intensities are added for peak with the same integer mass
//  c) Integer peak masses are converted back to float by multiplying by resolution
void Spectrum::roundMasses(float resolution) {
	unsigned int peakIdx, pivot;
	
	// Step a)
	for(peakIdx=0; peakIdx<peakList.size(); peakIdx++) 
		peakList[peakIdx][0]=(float)round(peakList[peakIdx][0]/resolution);
	
	// Step b)
	for(peakIdx=0, pivot=1; pivot<peakList.size(); pivot++)
		if(fabs(peakList[peakIdx][0]-peakList[pivot][0])<0.00001) peakList[peakIdx][1]+=peakList[pivot][1];
		else peakList[++peakIdx]=peakList[pivot];
	peakList.resize(peakIdx+1);
	
	// Step c)
	for(peakIdx=0; peakIdx<peakList.size(); peakIdx++) 
		peakList[peakIdx][0]*=resolution;
}

//
//  Computes the sets of pairs in the spectrum and returns (in pairs) a list of
//    pairs sorted by minimum distance to the closest endpoint. Two peaks are
//    considered a pair if their masses add up to parentMass-idDist+pmOffset -> 
//    use pmOffset=0 for PRM spectra and pmOffset=2 for MS/MS spectra.
//
//  pairs    - returns the masses of the peaks in the pairs (per pair)
//  pairsIdx - returns the indices of the peaks in the pairs (per pair)
//
void Spectrum::getPairs(float pmOffset, float tolerance, vector<vector<float> > &pairs, vector<vector<int> > &pairsIdx) {
	float aaMass = parentMass+(pmOffset-idDist)*AAJumps::massHion;
	int curPair,  // Index of the current pair being generated
		  i,j,k;      // iterators over the prefix/suffix peaks
	vector<float> peakScores;  // Peak scores including scores of neigboring peaks within tolerance
	vector<bool> processed;    // Keeps track of which peaks have already been used in some pair

	peakScores.resize(peakList.size());   processed.resize(peakList.size());
	for(k=0;k<(int)peakList.size();k++) {
		processed[k]=false;
		peakScores[k]=peakList[k][1];
	}

	unsigned int maxNumPairs = (unsigned int) round(((double)peakList.size()+1.0)*(double)peakList.size()/2);
	pairs.resize(maxNumPairs);   pairsIdx.resize(maxNumPairs);
	i=0; j=peakList.size()-1; curPair=0;
	while (i<=j) {
		if (peakList[i][0]<=aaMass-peakList[j][0]) {
			for(k=j; k>i && peakList[k][0]>=aaMass-peakList[i][0]-2*tolerance; k--) {
				pairs[curPair].resize(3);           pairsIdx[curPair].resize(2);
				pairs[curPair][2]=peakScores[i]+peakScores[k];
				pairs[curPair][0]=peakList[i][0];   pairs[curPair][1]=peakList[k][0];
				pairsIdx[curPair][0]=i;             pairsIdx[curPair][1]=k;
				curPair++;                          processed[k]=true;
			}
			if(k==j && !processed[i]) {
				pairs[curPair].resize(3);           pairsIdx[curPair].resize(2);
				pairs[curPair][2]=peakScores[i];
				pairs[curPair][0]=peakList[i][0];   pairs[curPair][1]=aaMass-peakList[i][0];
				pairsIdx[curPair][0]=i;             pairsIdx[curPair][1]=-1;
				curPair++; 
			}
			processed[i]=true; i++;
		} else {
			for(k=i; k<j && peakList[k][0]<=aaMass-peakList[j][0]+2*tolerance; k++) {
				pairs[curPair].resize(3);           pairsIdx[curPair].resize(2);
				pairs[curPair][2]=peakScores[j]+peakScores[k];
				pairs[curPair][0]=peakList[k][0];   pairs[curPair][1]=peakList[j][0];
				pairsIdx[curPair][0]=k;             pairsIdx[curPair][1]=j;
				curPair++;                          processed[k]=true;
			}
			if(k==i && !processed[j]) {
				pairs[curPair].resize(3);           pairsIdx[curPair].resize(2);
				pairs[curPair][2]=peakScores[j];
				pairs[curPair][0]=aaMass-peakList[j][0];   pairs[curPair][1]=peakList[j][0];
				pairsIdx[curPair][0]=-1;            pairsIdx[curPair][1]=j;
				curPair++; 
			}
			processed[j]=true; j--;
		}
	}
	pairs.resize(curPair);   pairsIdx.resize(curPair);
}

//
//  makeSymmetric - Forces a spectrum to be symmetric by adding symmetric peaks
//    whenever missing from the spectrum.
//
//  pmOffset - use 0 for PRM spectra, 2 for MS/MS spectra
//
void Spectrum::makeSymmetric(float pmOffset, float tolerance) {
	// Find all symmetric pairs in the spectrum
	vector<vector<float> > pairs; vector<vector<int> > pairsIdx;
	getPairs(pmOffset, tolerance, pairs, pairsIdx);
	
	// Check which peaks already have a symmetric peak in the spectrum
	vector<bool> needsPair(peakList.size());
	unsigned int peakIdx;
	for(peakIdx=0; peakIdx<needsPair.size(); peakIdx++) needsPair[peakIdx]=true;
	for(unsigned int pairIdx=0; pairIdx<pairsIdx.size(); pairIdx++)
		if(pairsIdx[pairIdx][0]>=0 and pairsIdx[pairIdx][1]>=0)
			{ needsPair[pairsIdx[pairIdx][0]]=false; needsPair[pairsIdx[pairIdx][1]]=false; }
	
	// Count how many new peaks are necessary
	unsigned int curCount=peakList.size(), newCount=0;
	for(peakIdx=0; peakIdx<needsPair.size(); peakIdx++) newCount+=(unsigned int)needsPair[peakIdx];
	peakList.resize(curCount+newCount);
	
	// Insert new symmetric peaks
	unsigned int newIdx=curCount;
	for(peakIdx=0; peakIdx<needsPair.size(); peakIdx++)
		if(needsPair[peakIdx]) {
			peakList[newIdx].set(parentMass+(pmOffset-1)*AAJumps::massHion-peakList[peakIdx][0],peakList[peakIdx][1]);
			if(peakList[newIdx][0]>0) newIdx++;
		}
	peakList.resize(newIdx);
	sort(peakList.begin(),peakList.end());
}

bool Spectrum::compare(Spectrum &toSpec) {
	if(peakList.size()!=toSpec.peakList.size() || parentMass!=toSpec.parentMass) return false;
	for(int i=0; i<peakList.size(); i++)
		if(peakList[i][0]!=toSpec.peakList[i][0] || peakList[i][1]!=toSpec.peakList[i][1])
			return false;
	return true;
}

//
//  merge - merge the peaks in the current spectrum with those in withSpec and
//          output the result to toSpec.
//
//  scores1, scores2, scoresMerged are vectors with the same sizes as peakList,
//    withSpec.peakList and toSpec.peakList; scoresMerged is computed to add
//    the values in scores1,scores2 if the corresponding peaks with the same
//    indices (in peakList/withSpec.peakList) are merged into toSpec
//
void Spectrum::merge(vector<TwoValues<float> > &withPeaks, Spectrum &toSpec, vector<float> &scores1, vector<float> &scores2, vector<float> &scoresMerged) {
	toSpec.copyNP(*this);
	toSpec.peakList.resize(peakList.size()+withPeaks.size());
	scoresMerged.resize(peakList.size()+withPeaks.size());
	unsigned int i=0,j=0,k=0;
	while(k<toSpec.size() && (i<peakList.size() || j<withPeaks.size())) {
		if(i==peakList.size()) { toSpec.peakList[k] = withPeaks[j]; scoresMerged[k++]=scores2[j++]; continue; }
		if(j==withPeaks.size()) { toSpec.peakList[k] = peakList[i]; scoresMerged[k++]=scores1[i++]; continue; }
		if(peakList[i][0]==withPeaks[j][0]) {
			toSpec.peakList[k].set(peakList[i][0],peakList[i][1]+withPeaks[j][1]);
			scoresMerged[k++] = scores1[i++] + scores2[j++];
		} else {
			if(peakList[i][0]<withPeaks[j][0]) { toSpec.peakList[k] = peakList[i]; scoresMerged[k++]=scores1[i++]; }
			else { toSpec.peakList[k] = withPeaks[j]; scoresMerged[k++]=scores2[j++]; }
		}
	}
	toSpec.peakList.resize(k); scoresMerged.resize(k);
}

//
//  filterAdd - constructs a new spectrum (output) where each peak p in the current
//      object is added to output if it also exists in other (within tolerance).
//      The peak score in output is the score of p _plus_ the maximum score of any
//      matching peak in other _minus_ the absolute distance between them (in % of tolerance)
//
//  otherScores   - scores of the peaks selected from other
//  otherEPScores - summed scores of the peaks in other matched to the endpoints
//                   of the current spectrum (guaranteed matches)
//
void Spectrum::filterAdd(Spectrum &other, float shift, float tolerance, Spectrum &output, vector<float> &otherScores, float &otherEPScores) {
	int low=0, high=0;  // Indices of the potential matching peaks in other
	int i, otherIdx, outputIdx=0;
	
	output.copyNP(*this);
	
	// Check for matches of internal PRMs
	output.peakList.resize(peakList.size());   otherScores.resize(peakList.size());
	for(i=0; i<peakList.size(); i++) {
		while(low<other.size() && other[low][0]+shift<peakList[i][0]-tolerance-0.00001) low++;
		while(high<other.size() && other[high][0]+shift<=peakList[i][0]+tolerance+0.00001) high++; // high is index of first unreachable peak
		if(low==high) continue;
		output[outputIdx].set(peakList[i][0],peakList[i][1]+other[low][1]-fabs(peakList[i][0]-other[low][0]-shift)/tolerance);
		otherScores[outputIdx]=other[low][1];
		for(otherIdx=low+1; otherIdx<high; otherIdx++) 
			if(output[outputIdx][1]<peakList[i][1]+other[otherIdx][1]-fabs(peakList[i][0]-other[otherIdx][0]-shift)/tolerance) {
				output[outputIdx][1] = peakList[i][1]+other[otherIdx][1]-fabs(peakList[i][0]-other[otherIdx][0]-shift)/tolerance;
				otherScores[outputIdx]=other[otherIdx][1];
			}
		outputIdx++;
	}
	output.peakList.resize(outputIdx);   otherScores.resize(outputIdx);
}

//
//  denovoLR - find a de-novo intepretation of the spectrum by finding the
//    highest scoring left-to-right path in the spectrum (does not exclude
//    forbidden pairs).
//
float Spectrum::denovoLR(AAJumps &jumps, float peakTol, vector<int> &matchedIdx) {
	vector<TwoValues<float> > scores;  // Pos[0] has current score, Pos[1] has best score so far
	vector<TwoValues<int> > indices;   // Pos[0] has index of predecessor, Pos[1] has index of PRM with best path score so far
	int idxPeak,  // Index of current peak
		idxJump,  // Index of current jump
		idxPred;  // Index of tentative predecessor
	
	scores.resize(peakList.size());   indices.resize(peakList.size());
	for(idxPeak=0; idxPeak<peakList.size(); idxPeak++) { scores[idxPeak].set(peakList[idxPeak][1],0); indices[idxPeak].set(-1,idxPeak); }

	for(idxPeak=0; idxPeak<peakList.size(); idxPeak++) {
		if(idxPeak==0) scores[0].set(peakList[idxPeak][1],peakList[idxPeak][1]);
		else if (peakList[idxPeak][1]<scores[idxPeak-1][1])
			{ indices[idxPeak][1]=indices[idxPeak-1][1]; scores[idxPeak][1]=scores[idxPeak-1][1]; }
		
		for(idxJump=0; idxJump<jumps.size() and jumps[idxJump]<=0; idxJump++);
		for(idxPred=idxPeak-1; idxPred>=0 and peakList[idxPred][0]>peakList[idxPeak][0]-jumps[idxJump]+peakTol; idxPred--);
		while (idxPred>=0 and idxJump<jumps.size()) {
			for(; idxPred>=0 and peakList[idxPred][0]>peakList[idxPeak][0]-jumps[idxJump]-peakTol; idxPred--) {

				if(scores[idxPred][0]+peakList[idxPeak][1]>scores[idxPeak][0]) {
					scores[idxPeak][0] = scores[idxPred][0]+peakList[idxPeak][1];
					indices[idxPeak][0] = idxPred;
				} 
			}
			idxJump++;
			for(; idxJump<jumps.size() and idxPred>=0 and peakList[idxPred][0]>peakList[idxPeak][0]-jumps[idxJump]+peakTol; idxPred--);
		}

		// Check if path should link to highest scoring PRM with mass farther than the largest jump
		if(idxPred>=0 and scores[idxPred][1]+peakList[idxPeak][1]>scores[idxPeak][0]) {
			scores[idxPeak][0] = scores[idxPred][1]+peakList[idxPeak][1];
			indices[idxPeak][0] = indices[idxPred][1];
		}		

		if(scores[idxPeak][0]>scores[idxPeak][1]) 
			{ scores[idxPeak][1]=scores[idxPeak][0];   indices[idxPeak][1]=idxPeak; }
	}

	// Fill matchedIdx
	int mIdx=0;  matchedIdx.resize(indices.size());
	idxPeak=indices[indices.size()-1][1];
	while(idxPeak>=0) { 
		matchedIdx[mIdx++]=idxPeak;   idxPeak=indices[idxPeak][0]; }
	matchedIdx.resize(mIdx);
	
	return scores[scores.size()-1][1];
}


//
// **************************************************************
//    SpecSet methods
// **************************************************************
//

vector <list<int> > &SpecSet::getMassesHistogram(vector <list<int> > &results, float resolution) {
    int i;
    double t,massIdx=0;  // Value is always integer but max is (double,double)

    for(i=0 ; i<specs.size() ; i++) 
    	{ t = round(specs[i].parentMass/resolution); if(massIdx<t) massIdx = t; }
    results.resize((int)massIdx+1);

    for(i=0 ; i<specs.size() ; i++) results[(int)(round(specs[i].parentMass/resolution))].push_front(i);
    return results;
}

template<class T> unsigned int SpecSet::extract(vector<T> &features, T featureValue, SpecSet &output) {
	output.resize(max(features.size(),specs.size()));
	unsigned int keptSpectra=0, pivot;

	for(pivot=0; pivot<output.size(); pivot++)
		if(features[pivot]==featureValue)
			output[keptSpectra++] = specs[pivot];
			
	output.resize(keptSpectra);
	return keptSpectra;
}

void instantiate_SpecSet_extract() {
	SpecSet specs, specsOut;
	
	vector<int> features;   specs.extract(features, (int) 0, specsOut);
}

//
//  LoadSpecSet_mgf: Loads a set of spectra from a .mgf file. Recognizes and processes
//    these MGF tags: BEGIN IONS, END IONS, CHARGE, PEPMASS
//
//  Note: If a spectrum has more than 1 header then this function only uses the last header.
//
unsigned int SpecSet::LoadSpecSet_mgf(char *filename) {
	BufferedLineReader blr;   resize(0);
	if(blr.Load(filename)<=0 or blr.size()<3) return 0;  // A valid file must have at least 3 lines: BEGIN_IONS, END_IONS and one mass/intensity peak

	unsigned int lineIdx, specIdx, peakIdx;

	// Counts number of spectra and number of peaks per spectrum
	list<int> peaksPerSpec;   int first;
	int numPeaks=0;  // Counts number of peaks in the spectrum
	for(lineIdx=0; lineIdx<blr.size(); lineIdx++) {
		if(numPeaks>=0 and strcmp("END IONS",blr.getline(lineIdx))==0) { peaksPerSpec.push_back(numPeaks); numPeaks=-1; continue; }
		if(strcmp("BEGIN IONS",blr.getline(lineIdx))==0) { numPeaks=0; continue; }
		first = (int)blr.getline(lineIdx)[0];
		if(numPeaks>=0 and first>=48 and first<=57) numPeaks++;
	}
	
	// Parse spectra
	resize(peaksPerSpec.size());   lineIdx=0;   char *token, *line;
	for(specIdx=0; specIdx<specs.size(); specIdx++) {
		// Skip empty lines
		while(lineIdx<blr.size() and blr.getline(lineIdx)[0]==0) lineIdx++;
		if(lineIdx==blr.size()) { cerr<<"Error loading "<<filename<<" - "<<specIdx<<" spectra instead of "<<specs.size()<<"?\n"; resize(0); return 0; }

		// Start of spectrum
		if(strcmp("BEGIN IONS",blr.getline(lineIdx))!=0) { cerr<<"ERROR: Expected BEGIN IONS, found '"<<blr.getline(lineIdx)<<"' (line "<<lineIdx+1<<")\n"; return 0; }
		else lineIdx++;
		
		// Read peaks/charge/parent mass
		specs[specIdx].resize(peaksPerSpec.front());   peaksPerSpec.pop_front();   peakIdx=0;
		specs[specIdx].parentCharge=0;
		while(lineIdx<blr.size() and strcmp("END IONS",blr.getline(lineIdx))!=0) {
			line = blr.getline(lineIdx++);
			if(line[0]>=48 and line[0]<=57) {
				token = strtok(line," \t"); if(!token) { cerr<<"Error loading "<<filename<<" - could not parse peak mass on line "<<lineIdx<<"!\n"; resize(0); return 0; }
				specs[specIdx][peakIdx][0] = atof(token);
				token = strtok(NULL," \t"); if(!token) { cerr<<"Error loading "<<filename<<" - could not parse peak intensity on line "<<lineIdx<<"!\n"; resize(0); return 0; }
				specs[specIdx][peakIdx][1] = atof(token);
				peakIdx++; continue;
			}
			
			if(strncmp("CHARGE=+",line,8)==0) { specs[specIdx].parentCharge = (short)atof(&line[8]); continue; }
			else if(strncmp("CHARGE=",line,7)==0) { specs[specIdx].parentCharge = (short)atof(&line[7]); continue; }
			
			if(strncmp("PEPMASS=",line,8)==0) specs[specIdx].parentMass = atof(&line[8]);
		}
		if(specs[specIdx].parentCharge>0) specs[specIdx].parentMass=specs[specIdx].parentMass*specs[specIdx].parentCharge-specs[specIdx].parentCharge+1;
		lineIdx++;
	}
	
	return specs.size();
}

//
//  LoadSpecSet_ms2: Loads a set of spectra from a .ms2 file. Spectra must be separated
//    by at least one empty line.
//
//  Note: If a spectrum has more than 1 header then this function only uses the last header.
//
unsigned int SpecSet::LoadSpecSet_ms2(char *filename) {
	BufferedLineReader blr;   resize(0);
	if(blr.Load(filename)<=0 or blr.size()<3) return 0;  // A valid file must have at least 3 lines: 1 header (2 lines) + 1 (m/z,intensity) pair
	
	unsigned int lineIdx, specIdx, peakIdx;

	// Counts number of spectra and number of peaks per spectrum
	list<int> peaksPerSpec;
	int numPeaks=1;  // Counts number of peaks in the spectrum
	for(lineIdx=0; lineIdx<blr.size(); lineIdx++)
		if(blr.getline(lineIdx)[0]==':') { if(numPeaks>0) peaksPerSpec.push_back(numPeaks); numPeaks=-1; }
		else if(blr.getline(lineIdx)[0]!=0) numPeaks++;
	peaksPerSpec.push_back(numPeaks);  // Number of peaks in the last spectrum
	peaksPerSpec.pop_front();          // First element is just the '1' used to initialize numPeaks

	// Parse spectra
	resize(peaksPerSpec.size());   lineIdx=0;   char *token;
	for(specIdx=0; specIdx<specs.size(); specIdx++) {
		// Skip empty lines
		while(blr.getline(lineIdx)[0]==0 and lineIdx<blr.size()) lineIdx++;
		if(lineIdx==blr.size()) { cerr<<"Error loading "<<filename<<" - "<<specIdx<<" spectra instead of "<<specs.size()<<"?\n"; resize(0); return 0; }
		
		// Parse header(s)
		while(blr.getline(lineIdx)[0]==':') {
			lineIdx++;
			token = strtok(blr.getline(lineIdx++)," \t"); if(!token) { cerr<<"Error loading "<<filename<<" - could not parse parent mass for spectrum "<<specIdx+1<<"?\n"; resize(0); return 0; }
			specs[specIdx].parentMass = (float)atof(token);
			token = strtok(NULL," \t"); if(!token) { cerr<<"Error loading "<<filename<<" - could not parse parent charge for spectrum "<<specIdx+1<<"?\n"; resize(0); return 0; }
			specs[specIdx].parentCharge = (short)atof(token);
			if(lineIdx==blr.size()) { cerr<<"Error loading "<<filename<<" - "<<specIdx<<" spectra instead of "<<specs.size()<<"?\n"; resize(0); return 0; }
		}

		// Read spectrum peaks
		specs[specIdx].resize(peaksPerSpec.front());   peaksPerSpec.pop_front();
		for(peakIdx=0; peakIdx<specs[specIdx].size(); peakIdx++) {
			token = strtok(blr.getline(lineIdx++)," \t"); if(!token) { cerr<<"Error loading "<<filename<<" - could not parse peak mass for spectrum "<<specIdx+1<<", peak "<<peakIdx+1<<"?\n"; resize(0); return 0; }
			specs[specIdx][peakIdx][0] = (float)atof(token);
			token = strtok(NULL," \t"); if(!token) { cerr<<"Error loading "<<filename<<" - could not parse peak intensity for spectrum "<<specIdx+1<<", peak "<<peakIdx+1<<"?\n"; resize(0); return 0; }
			specs[specIdx][peakIdx][1] = (float)atof(token);
			if(lineIdx==blr.size() and peakIdx<specs[specIdx].size()-1) { cerr<<"Error loading "<<filename<<" - end of file before end of spectrum "<<specIdx+1<<" ended, got only "<<peakIdx+1<<" peaks instead of "<<specs.size()<<"?\n"; resize(0); return 0; }
		}
	}
	
	return specs.size();
}

//
//  LoadSpecSet_prms: Loads a set of spectra from a .prms file, as output but the
//     current version of pepnovo_prms (2007/04/09). Text file with multiple spectra,
//     each delimited in the following format:
//
//  - ">> <original msms file index> <scan/index # in msms file> <single-spectrum file name>"
//      -- scan/index # in msms file is stored in the Spectrum.scan field.
//      -- The whole info line is stored in the Spectrum.info field.
//  - Set of peaks in "<peak mass> <peak intensity/score>" format
//  - Spectrum terminated by empty line
//  - Last pair of mass/intensity corresponds to the corrected sum-of-amino-acid-masses parent mass
//
//  Note: If a spectrum has more than 1 header then this function only uses the last header.
//
unsigned int SpecSet::LoadSpecSet_prms(char *filename) {
	unsigned int specIdx, lineIdx, peakIdx;
	list<unsigned int> specSizes;
	char *line;
	BufferedLineReader blr;
	
	// Load whole file into memory
	if(blr.Load(filename)<=0) return 0;
	
	// Count # spectra in the file
	bool inSpectrum=false;
	peakIdx = 0;
	for(lineIdx=0; lineIdx<blr.size(); lineIdx++) {
		line=blr.getline(lineIdx); 
		if(line[0]!=0 and line[0]=='>' and line[1]!=0 and line[1]=='>') { inSpectrum=true; peakIdx=0; continue; }
		if(line[0]==0) { if(inSpectrum) specSizes.push_back(peakIdx); inSpectrum=false; }
		else if(inSpectrum) peakIdx++;
	}
	if(inSpectrum) specSizes.push_back(peakIdx);  // In case there is no empty line after the last spectrum
	specs.resize(specSizes.size());
	
	// Parse the text
	peakIdx = 0; specIdx = 0;
	list<unsigned int>::iterator sizesIter = specSizes.begin();
	char *token;
	for(lineIdx=0; lineIdx<blr.size(); lineIdx++) {
		line=blr.getline(lineIdx);
		
		if(line[0]==0) { 
			if(inSpectrum) {  // End of current spectrum
				if(peakIdx>0 and peakIdx<=specs[specIdx].size()) specs[specIdx].parentMass = specs[specIdx][peakIdx-1][0]+AAJumps::massMH;
				specIdx++;   inSpectrum=false;
			}
			continue;
		}
		
		// Check for start of a new spectrum
		if(line[0]!=0 and line[0]=='>' and line[1]!=0 and line[1]=='>') {
			specs[specIdx].info = (char *)malloc(strlen(line)-1);   strcpy(specs[specIdx].info,&line[2]);
			char *tok=strtok(line," \t");   tok=strtok(NULL," \t");   // Skip ">> " and "<file_index> "
			tok=strtok(NULL," \t");    specs[specIdx].scan=(unsigned int)strtoul(tok,NULL,10);
			specs[specIdx].resize(*sizesIter);   sizesIter++;
			peakIdx=0;   inSpectrum=true;   continue;
		}
		
		// Peak <mass> <intensity> pair
		token = strtok(line," \t");  if(token[0]==0) { cerr<<"ERROR reading peak mass for peak "<<peakIdx<<" for the spectrum entitled ("<<specs[specIdx].info<<") in file "<<filename<<"!\n"; specs.resize(0); return 0; }
		specs[specIdx][peakIdx][0] = atof(token);
		token = strtok(NULL," \t");  if(token[0]==0) { cerr<<"ERROR reading peak intensity for peak "<<peakIdx<<" for the spectrum entitled ("<<specs[specIdx].info<<") in file "<<filename<<"!\n"; specs.resize(0); return 0; }
		specs[specIdx][peakIdx][1] = atof(token);
		peakIdx++;
	}
	
	return specs.size();
}

unsigned int SpecSet::LoadSpecSet_pkl(char *filename) {
    int numSpecs=1;                     // Assume file contains at least one spectrum and count one additional spectrum per blank line
    ifstream input(filename);
    if (!input) { cerr << "ERROR: cannot open " << filename << "\n";   specs.resize(0);  return 0; }

    list<int> specSizes;  // Register number of peaks per spectrum

    char *lineBuffer = (char *)malloc(1025);
    int numPeaks = -1;  // zero only after reading first tuple of (parent mass, intensity, charge)
    while (!input.eof() && !input.fail()) {
        input.getline(lineBuffer,1024,'\n');
        if (lineBuffer[0]==0 or lineBuffer[0]=='\n' or lineBuffer[0]=='\r')
           { if(numPeaks>=0) { numSpecs++; specSizes.push_back(numPeaks); numPeaks=-1;} }
        else { if(((int)lineBuffer[0]>=(int)'0' and (int)lineBuffer[0]<=(int)'9') or lineBuffer[0]=='-') numPeaks++; }
    }
    if (numPeaks>=0) specSizes.push_back(numPeaks); else numSpecs--; // In case there was no empty line at the end of the pkl file

    specs.resize(numSpecs);
    input.close();
    ifstream input2(filename);

    float foo;
    for (int i=0; i<numSpecs ; i++) {
        input2 >> specs[i].parentMass >> foo >> specs[i].parentCharge ; // Read precursor mass, total intensity, charge state
        if (specs[i].parentCharge>0)
            specs[i].parentMass = specs[i].parentMass * specs[i].parentCharge - specs[i].parentCharge + 1; // Convert to PM+19

        specs[i].peakList.resize(specSizes.front());   specSizes.pop_front();
        for (int j=0 ; j<specs[i].size() ; j++)
            input2 >> specs[i][j][0] >> specs[i][j][1];
        input2.getline(lineBuffer,1024,'\n');  // Read intermediate newline
    }

    input2.close();
    free(lineBuffer);
    return 1;
}

short SpecSet::SaveSpecSet_info(char *filename){
    ofstream output(filename);
    if (!output) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    for (int i=0; i<specs.size(); i++) output << specs[i].info << "\n";

    output.close();
    return 1;
}

short SpecSet::SaveSpecSet_pkl(char *filename){
    ofstream output(filename);
    if (!output) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }

    for (int i=0; i<specs.size(); i++) {
        output << specs[i].parentMass << " -1 " << specs[i].parentCharge << "\n";
        for (int j=0; j<specs[i].size(); j++) {
            output << specs[i][j][0] << " " << specs[i][j][1] << endl;
        }
        if (i<specs.size()-1) { output << endl; }
    }

    output.close();
    return 1;
}

//
//  SaveSpecSet_pklbin - saves a set of spectra in binary format. File format
//    1 int - number of spectra in the file
//    numSpecs shorts - number of peaks per spectrum in the file
//    arrays of numPeaks+1 pairs of floats - [parentMass charge] + array of [mass intensity]
//
short SpecSet::SaveSpecSet_pklbin(char *filename){
    FILE *fp;   unsigned int numSpecs = specs.size();
    unsigned int i,p;
    
    fp = fopen(filename,"w");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return -1; }
    
    fwrite(&numSpecs,sizeof(int),1,fp);  // Number of spectra in the file
    
    unsigned short *numPeaks=(unsigned short *)malloc(sizeof(short)*numSpecs); unsigned short maxNumPeaks=0;
    for(i=0; i<numSpecs; i++) { numPeaks[i]=specs[i].size(); maxNumPeaks=max(maxNumPeaks,numPeaks[i]); }
    fwrite(numPeaks,sizeof(short),numSpecs,fp);  // Number of peaks per spectrum in the file
    
    float *peaksBuffer = (float *)malloc(2*(maxNumPeaks+1)*sizeof(float));   unsigned int pbIdx;
    for(i=0; i<numSpecs; i++) {
    	peaksBuffer[0]=specs[i].parentMass;
    	peaksBuffer[1]=(float)specs[i].parentCharge;
    	for (pbIdx=2,p=0; p<numPeaks[i]; p++) {
	    	peaksBuffer[pbIdx++]=specs[i][p][0];
	    	peaksBuffer[pbIdx++]=specs[i][p][1];
    	}
	    fwrite(peaksBuffer,sizeof(float),2*(numPeaks[i]+1),fp);  // [parentMass charge] followed by [masses intensities]
    }
    
    free(peaksBuffer); free(numPeaks);
    fclose(fp);
    return 1;
}

//
//  LoadSpecSet_pklbin - loads a set of spectra in binary format. File format
//    1 int - number of spectra in the file
//    numSpecs shorts - number of peaks per spectrum in the file
//    arrays of numPeaks+1 pairs of floats - [parentMass charge] + array of [mass intensity]
//
unsigned int SpecSet::LoadSpecSet_pklbin(char *filename){
    FILE *fp;   unsigned int numSpecs=0;   
    unsigned short *numPeaks;  // Pointer to array containing 1+number of peaks per spectrum
    float *data;               // Pointer to array containing spectrum data
    unsigned int dataSize=0;   // Size of the data buffer
    unsigned int i,p,dataIdx,count,numValues;
    
    fp = fopen(filename,"r");
    if (fp==0) { cerr << "ERROR: cannot open " << filename << "\n";   return 0; }
    
    count = fread(&numSpecs,sizeof(unsigned int),1,fp);   // Number of spectra in the file
    if(count!=1) { cerr<<"ERROR reading "<<filename<<"!\n"; return 0; }

    numPeaks = (unsigned short *)malloc(sizeof(unsigned short)*numSpecs);  // Number of peaks per spectrum
    if(numPeaks==(unsigned short *)0) { cerr<<"ERROR: Not enough memory for "<<numSpecs<<" spectra.\n"; fclose(fp); return 0; }
    count=fread(numPeaks,sizeof(unsigned short),numSpecs,fp);
    if(count!=numSpecs) { cerr<<"ERROR reading "<<filename<<"!\n"; free(numPeaks); return 0; }
    for(i=0;i<numSpecs;i++) { if(numPeaks[i]>dataSize) dataSize=numPeaks[i]; }
    dataSize = 2*dataSize+2;   // Each spectrum 'line' has 2 values and there's one additional 'line' with parent mass/charge
	data = (float *) malloc(sizeof(float)*dataSize);
    
    specs.resize(numSpecs);
    for(i=0; i<numSpecs; i++) {
    	numValues = 2*numPeaks[i]+2;   
		count = fread(data,sizeof(float),numValues,fp);
		if(count!=numValues) { cerr<<"ERROR reading "<<filename<<"!\n"; free(numPeaks); free(data); return 0; }

    	specs[i].parentMass = data[0];    specs[i].parentCharge = (int)data[1];    specs[i].resize(numPeaks[i]);
    	for (p=0,dataIdx=2; dataIdx<numValues; dataIdx+=2,p++) specs[i][p].set(data[dataIdx],data[dataIdx+1]);
    }
    
    free(numPeaks); free(data);
    fclose(fp);
    return 1;
}

//
//  MakeSymmetric - convert a list of masses/scores to a symmetric list of masses/scores,
//    i.e. outSpec[i]+outSpec[n-i-1]=parentMass+18, adding entries with score zero if necessary.
//  Notes:
//    1 - if outSpec==NULL or outSpec==inSpec then the resulting contents replace those of inSpec
//    2 - if indices!=NULL then each position contains the inSpec index of the entry in outSpec (or -1 if not in inSpec)
//    3 - Current implementation is tailored to PRM spectra. Set parameter parentMass to sum of aa masses+18+3 
//          to apply this function to MS/MS spectra.
//
void MakeSymmetric(vector<TwoValues<float> > *inSpec, float parentMass, float peakTol, vector<TwoValues<float> > *outSpec, vector<int> *indices) {
	bool replaceContents = (inSpec==outSpec || outSpec==0);
	if(replaceContents) outSpec=new vector<TwoValues<float> >;   
	
	outSpec->resize(2*inSpec->size());   if(indices) indices->resize(2*inSpec->size());
	
	int inBottom=0, inTop=inSpec->size()-1,     // Used to traverse inSpec
	    outBottom=0, outTop=outSpec->size()-1;  // Used to fill in the symmetric outSpec
	float symMass;

	while(inBottom<=inTop) {
		if(abs((*inSpec)[inBottom][0]-(parentMass-AAJumps::massHion-(*inSpec)[inTop][0]))<=peakTol) {  // symmetric peaks
			(*outSpec)[outBottom++]=(*inSpec)[inBottom++];      if(indices) (*indices)[outBottom-1]=inBottom-1;
			if(inBottom<=inTop)       // if condition avoids the case where a peak is its own symmetric
				{ (*outSpec)[outTop--]=(*inSpec)[inTop--];      if(indices) (*indices)[outTop+1]=inTop+1;  }
		} else if((*inSpec)[inBottom][0]<(parentMass-AAJumps::massHion-(*inSpec)[inTop][0])) {
			symMass = parentMass-AAJumps::massHion-(*inSpec)[inBottom][0];
			(*outSpec)[outBottom++]=(*inSpec)[inBottom++];   (*outSpec)[outTop--].set(symMass,0);
			if(indices) { (*indices)[outBottom-1]=inBottom-1;   (*indices)[outTop+1]=-1; }
		} else {
			symMass = parentMass-AAJumps::massHion-(*inSpec)[inTop][0];
			(*outSpec)[outBottom++].set(symMass,0);          (*outSpec)[outTop--]=(*inSpec)[inTop--];			
			if(indices) { (*indices)[outBottom-1]=-1;           (*indices)[outTop+1]=inTop+1; }
		}
	}
	
	// Remove the possible empty positions in the middle of the array (by shifting the top elements down)
	int difference = outTop+1-outBottom;
	if(difference>0) 
		for(int pivot=outTop+1; pivot<=outSpec->size(); pivot++) 
			{ (*outSpec)[pivot-difference]=(*outSpec)[pivot]; if (indices) (*indices)[pivot-difference]=(*indices)[pivot]; }
	outSpec->resize(outSpec->size()-difference);   if (indices) indices->resize(outSpec->size()-difference);
	
	if(replaceContents) { inSpec->resize(outSpec->size()); for(int i=0; i<outSpec->size(); i++) (*inSpec)[i]=(*outSpec)[i];  delete outSpec; }
}
