#ifndef SPECTRUM_H
#define SPECTRUM_H

#include <vector>
#include <list>
#include <iostream>

#include "aminoacid.h"
#include "twovalues.h"

using namespace std;

class SpectrumPeakLabels;  // Defined in label.h

class Spectrum {
	void locateWithinTolerance(int startIdx, float targetMass, float tolerance, int &minIdx, int &maxIdx);
	void weightedAverage(int minIdx, int maxIdx, float &avgMass, float &totScore);
public:
    vector< TwoValues<float> > peakList;

    float parentMass;    // Monoisotopic parent mass (sum of AA masses + 19)
    short parentCharge;  // Precursor charge
    unsigned int scan;   // Scan number for this spectrum
    char *info;          // Miscelaneous information (e.g. spectrum filename)
    float resolution,    // How many daltons between peaks that are 1 integer unit apart 
    	  idDist;        // Inter-Dalton distance: number of integer units between
    	                 // peaks that are 1 Da apart (idDist=1/resolution), e.g. 
						 // 1 for regular spectra and 1/RESOLUTION for spectra with 
						 // masses=round(mass/RESOLUTION), e.g. idDist=10 for RESOLUTION=0.1

    Spectrum() { parentMass = 0; parentCharge = 0; scan=0; peakList.resize(0); resolution=1; idDist=1; info=(char *)0; }
    ~Spectrum();

    TwoValues<float> &operator[](unsigned int i) { return peakList[i]; }
    unsigned int size() { return peakList.size(); }
    unsigned int resize(unsigned int newSize) { peakList.resize(newSize); return peakList.size(); }
    
    Spectrum &operator=(const Spectrum &other);
    Spectrum &copyNP(const Spectrum &other) // Copy all member fields except peakList
    { parentMass=other.parentMass; parentCharge=other.parentCharge;   scan = other.scan;
      resolution=other.resolution; idDist=other.idDist; return(*this); }
    
	void changeResolution(float newResolution, bool enforceRounding);
	void setResolution(float newResolution) { resolution=newResolution; idDist=1/resolution; }

	// Add peaks for b0/bk/y0/yk
	void addZPMpeaks(float tolerance, float ionOffset, bool includeY0k, SpectrumPeakLabels *labels=0);
	void maximizeZPMpeaks(float tolerance, float ionOffset, bool includeY0k=false);
	void normalize(bool removeNegatives=true); // Normalizes total intensity to 100

	// Find all peaks within tolerance of baseMass
	short findMatches(float baseMass, float peakTol, vector<int> &matchesIdx, int startIdx=-1);
	int findClosest(float mass);

    void mergePeakList(vector<TwoValues<float> > &newPeaks, Spectrum *putHere);
    void mergePeakListRev(vector<TwoValues<float> > &newPeaks, Spectrum *putHere);
	void reverse(float pmOffset, Spectrum *putHere=0);
	void selectTopK(unsigned int topK, Spectrum *putHere=0);
	void roundMasses(float resolution=0.1);

	// Converts a list of masses to a list of corresponding spectrum peak indices (closest mass)
	void massesToIndices(vector<TwoValues<float> > &masses, vector<int> &indices, float peakTol);
	// Retain only the peaks with indices in idx (idx MUST be sorted)
	void selectIndices(vector<int> &idx);

    void output(ostream &output);
	void output_ms2(ostream &output);
    void getPairs(float pmOffset, float tolerance, vector<vector<float> > &pairs, vector<vector<int> > &pairsIdx);
	void makeSymmetric(float pmOffset, float tolerance);

	// Generic functions but initially defined for alignment of consensus to PRM spectra (batch.cpp::getPairAlignsPAext_aux)
	bool compare(Spectrum &toSpec);
	void merge(vector<TwoValues<float> > &withPeaks, Spectrum &toSpec, vector<float> &scores1, vector<float> &scores2, vector<float> &scoresMerged);
	void filterAdd(Spectrum &other, float shift, float tolerance, Spectrum &output, vector<float> &otherScores, float &otherEPScores);
	
	// De-novo intepretation functions
	float denovoLR(AAJumps &jumps, float peakTol, vector<int> &matchedIdx);
};

class SpecSet {
public:
    vector<Spectrum> specs;
    
    SpecSet(unsigned int sz=0) { specs.resize(sz); }
    SpecSet(char *filename) { LoadSpecSet_pkl(filename); }

    Spectrum &operator[](unsigned int i) { return specs[i]; }
    unsigned int size() { return specs.size(); }
    unsigned int resize(unsigned int newSize) { specs.resize(newSize); return specs.size(); }

    vector <list<int> > &getMassesHistogram(vector <list<int> > &results, float resolution=1.0);
	void changeResolution(float newResolution, bool enforceRounding)
		{ for(unsigned int i=0; i<specs.size(); i++) specs[i].changeResolution(newResolution,enforceRounding); }
	void addZPMpeaks(float tolerance, float ionOffset, bool includeY0k)
		{ for(unsigned int i=0; i<specs.size(); i++) specs[i].addZPMpeaks(tolerance, ionOffset, includeY0k); }

	void normalize() { for(unsigned int i=0; i<specs.size(); i++) specs[i].normalize(); } 
	template<class T> unsigned int extract(vector<T> &features, T featureValue, SpecSet &output);

    unsigned int LoadSpecSet_pkl(char *filename);
    unsigned int LoadSpecSet_ms2(char *filename);
	unsigned int LoadSpecSet_mgf(char *filename);
    unsigned int LoadSpecSet_prms(char *filename);
	unsigned int LoadSpecSet_pklbin(char *filename);
	short SaveSpecSet_info(char *filename);
    short SaveSpecSet_pkl(char *filename);
    short SaveSpecSet_pklbin(char *filename);
};

void MakeSymmetric(vector<TwoValues<float> > *inSpec, float parentMass, float peakTol, vector<TwoValues<float> > *outSpec=0, vector<int> *indices=0);

#endif
