% Copyright 2007,2008,2009 The Regents of the University of California
% All Rights Reserved
%
% Permission to use, copy, modify and distribute any part of this
% program for educational, research and non-profit purposes, by non-profit 
% institutions only, without fee, and without a written agreement is hereby 
% granted, provided that the above copyright notice, this paragraph and 
% the following three paragraphs appear in all copies.
%
% Those desiring to incorporate this work into commercial
% products or use for commercial purposes should contact the Technology
% Transfer & Intellectual Property Services, University of California,
% San Diego, 9500 Gilman Drive, Mail Code 0910, La Jolla, CA 92093-0910,
% Ph: (858) 534-5815, FAX: (858) 534-7345, E-MAIL:invent@ucsd.edu.
%
% IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
% FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
% INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE, EVEN
% IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
% OF SUCH DAMAGE.
%
% THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
% OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
% ENHANCEMENTS, OR MODIFICATIONS.  THE UNIVERSITY OF CALIFORNIA MAKES NO
% REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER IMPLIED OR
% EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
% MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR THAT THE USE OF
% THE SOFTWARE WILL NOT INFRINGE ANY PATENT, TRADEMARK OR OTHER RIGHTS.

%
%  sps.m - top-level script to run Shotgun Protein Sequencing analysis
%
function res = sps(paramsFN)

if nargin<1
    fprintf(1,'Syntax: specnets parameters_file_name\n\n');
    res=0; return;
end

%
% Reading parameters
%
[status, params] = sn_read_params_file(paramsFN);  if status==0 res=-1; return; end;

mandatory_params = {'INPUT_SPECS_MS';'EXE_DIR'};    params_names = fieldnames(params);
for idx=1:size(mandatory_params,1)
    if max(strcmp(mandatory_params{idx},params_names))==0 | isempty(getfield(params,mandatory_params{idx}))
        fprintf(1,'ERROR: Parameter %s is undefined in %s\n',mandatory_params{idx},paramsFN);    res=-1; return;
    end;
end;

pmTol = str2num(params.TOLERANCE_PM);    peakTol = str2num(params.TOLERANCE_PEAK);
idx = max(find(params.INPUT_SPECS_MS=='.'));
if ~isempty(idx) input_file_type=lower(params.INPUT_SPECS_MS(idx+1:length(params.INPUT_SPECS_MS))); else input_file_type=''; end;
if isempty(input_file_type) | (strcmp(input_file_type,'mgf')==0 & strcmp(input_file_type,'ms2')==0 & strcmp(input_file_type,'mzxml')==0 & strcmp(input_file_type,'pkl')==0 & strcmp(input_file_type,'pklbin')==0)
    fprintf(1,'ERROR: Unknown file type %s (parameter INPUT_SPECS_MS in %s)\n',input_file_type,paramsFN); res=-1; return;
end;
input_file_name = params.INPUT_SPECS_MS(1:idx-1);

STAGE_SCORING = 1;
STAGE_FILTERPAIRS = 2;
STAGE_ALIGNMENT = 3;
STAGE_FILTERSTARPAIRS = 4;
STAGE_ASSEMBLY = 5;
STAGE_REPORT = 6;
stage_str = lower(params.INITIAL_STAGE);    initial_stage=0;
if strcmp(stage_str,'scoring')         initial_stage=STAGE_SCORING; end;
if strcmp(stage_str,'filterpairs')     initial_stage=STAGE_FILTERPAIRS; end;
if strcmp(stage_str,'alignment')       initial_stage=STAGE_ALIGNMENT; end;
if strcmp(stage_str,'filterstarpairs') initial_stage=STAGE_FILTERSTARPAIRS; end;
if strcmp(stage_str,'assembly')        initial_stage=STAGE_ASSEMBLY; end;
if strcmp(stage_str,'report')          initial_stage=STAGE_REPORT; end;

%
% Initialize environment
%
c=clock;
curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
fprintf(1,'Shotgun Protein Sequencing 2.0.0: session started on %s %s\n',curDate,curTime);
dirContents = dir('.'); dirContents = {dirContents(:).name}';
if isempty(find(strcmp('spectra',dirContents))) status = dos('mkdir spectra'); end;
if isempty(find(strcmp('aligns',dirContents))) status = dos('mkdir aligns'); end;
if isempty(find(strcmp('assembly',dirContents))) status = dos('mkdir assembly'); end;
if isempty(find(strcmp('report',dirContents))) status = dos('mkdir report'); end;

global AAletters AAmasses AAcodes AAnames;
global bProfile yProfile bNames yNames;
load(sprintf('%s/msparams.mat',params.EXE_DIR));

%
% Loading MS/MS data (mgf/ms2/mzxml/pkl with multiple spectra per file)
%
if initial_stage==0
    if(strcmp(input_file_type,'pklbin'))
        specs_raw = sn_load_pklbin(params.INPUT_SPECS_MS,1);  if isempty(specs_raw) fprintf(1,'ERROR reading %s!\n',params.INPUT_SPECS_MS); res=-1; return; end;
        sn_save_pklbin(specs_raw, 'spectra/specs_ms.pklbin');
    else
        status = -1;
        if(strcmp(input_file_type,'ms2'))    status = dos(sprintf('%s/convert ms2   %s spectra/specs_ms.pklbin',params.EXE_DIR,params.INPUT_SPECS_MS)); end;
        if(strcmp(input_file_type,'mgf'))    status = dos(sprintf('%s/convert mgf   %s spectra/specs_ms.pklbin',params.EXE_DIR,params.INPUT_SPECS_MS)); end;
        if(strcmp(input_file_type,'mzxml'))  status = dos(sprintf('%s/convert mzxml %s spectra/specs_ms.pklbin',params.EXE_DIR,params.INPUT_SPECS_MS)); end;
        if(strcmp(input_file_type,'pkl'))    status = dos(sprintf('%s/convert pkl   %s spectra/specs_ms.pklbin',params.EXE_DIR,params.INPUT_SPECS_MS)); end;
        if status~=0 fprintf(1,'ERROR reading %s!\n',params.INPUT_SPECS_MS); res=-1; return; end;
    end;
end;
cd('spectra');
specs_raw = sn_load_pklbin('specs_ms.pklbin',1);  if isempty(specs_raw) fprintf(1,'ERROR reading %s!\n',params.INPUT_SPECS_MS); res=-1; return; end;
filenames = cell(size(specs_raw,1),1); for specIdx=1:size(filenames,1) filenames{specIdx}=sprintf('%s:%d',params.INPUT_SPECS_MS,specIdx); end;

%
% Pepnovo filtering/scoring
%
if initial_stage <= STAGE_SCORING
	c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
	curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
    fprintf(1,'Starting stage: scoring (%s %s)\n',curDate,curTime);

    sn_save_mgf(specs_raw(:,2), specs_raw(:,3), specs_raw(:,4), 'specs_ms.mgf');
    if ~isempty(params.MIN_SPECTRUM_QUALITY) pepnovoMode = sprintf('-pmcsqs_and_prm %s',params.MIN_SPECTRUM_QUALITY); else pepnovoMode = '-prm_only'; end;
    status = dos(sprintf('%s/PepNovo_bin -model_dir %s/Models -model LTQ_LOW_TRYP -file specs_ms.mgf -digest NON_SPECIFIC %s > specs_scored.prms',params.EXE_DIR,params.EXE_DIR,pepnovoMode)); if status~=0 fprintf(1,'ERROR executing %s/Pepnovo_bin!\n',params.EXE_DIR); res=-1; return; end;
	status = dos(sprintf('%s/convert prms specs_scored.prms specs_scored.pklbin',params.EXE_DIR)); if status~=0 fprintf(1,'ERROR reading Pepnovo output!\n'); res=-1; return; end;
end;
specs = sn_load_pklbin('specs_scored.pklbin',1);    specs_name = 'specs_scored';
idxKept = double(sn_load_binarray('kept_indices.bin','uint32'))+1;
specs_raw = specs_raw(idxKept,:);    filenames = filenames(idxKept,:);    clear idxKept;

%
% Alignment and spectral-stars consensus
%
cd('../aligns');

if initial_stage <= STAGE_FILTERPAIRS
	c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
	curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
    fprintf(1,'Starting stage: filterpairs (%s %s)\n',curDate,curTime);

	fid = fopen('filterpairs.params','w');    if fid<=0 fprintf(1,'ERROR opening filterpairs.params!\n'); res=-1; return; end;
	fprintf(fid,'10\nINPUT_SPECS_PKLBIN=../spectra/%s.pklbin\nOUTPUT_ALIGNS=pairs.dat\n',specs_name);
	fprintf(fid,'OUTPUT_NUM_MATCHED_PEAKS=pairs_mp.dat\nOUTPUT_MEANS=pairs_means.dat\nOUTPUT_VARIANCE=pairs_vars.dat\n');
	fprintf(fid,'TOLERANCE_PEAK=%s\nTOLERANCE_PM=%s\nMIN_RATIO=%s\nMIN_NUM_MATCHED_PEAKS=%s\nMIN_OVERLAP_AREA=%s\n',params.TOLERANCE_PEAK,params.TOLERANCE_PM,params.MIN_RATIO,params.MIN_MATCHED_PEAKS,params.MIN_OVERLAP_AREA);
	fclose(fid);
	status = dos(sprintf('%s/filterpairs filterpairs.params',params.EXE_DIR));   if status~=0 fprintf(1,'ERROR executing %s/filterpairs!\n',params.EXE_DIR); res=-1; return; end;

    [pairs_sp, pairs_asp, pairs_po, pairs_all] = sps_splitpairs(pmTol, str2num(params.MIN_MATCHED_PEAKS), str2num(params.MIN_PVALUE), str2num(params.SELECT_TRIANGLES));
    if isempty(pairs_all) fprintf(1,'No significant spectral alignments were found for your choice of alignment parameters\n'); res=0; return; end;
    
    sn_save_aligns('pairs_asp.bin',pairs_asp(:,1:2),pairs_asp(:,3:5));
    pairs=[pairs_sp;pairs_asp]; [foo,idxS]=sortrows(pairs(:,1:2)); pairs=pairs(idxS,:);
    sn_save_aligns('pairs.bin',pairs(:,1:2),pairs(:,3:5));
    sps_save_aligns('pairs_po.dat',pairs_po);
    save pairs pairs pairs_asp pairs_po;
else load pairs;
end;

% specalign
if initial_stage <= STAGE_ALIGNMENT
	c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
	curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
    fprintf(1,'Starting stage: alignment (%s %s)\n',curDate,curTime);

    fid = fopen('specalign.params','w');    if fid<=0 fprintf(1,'ERROR opening specalign.params!\n'); res=-1; return; end;
	fprintf(fid,'8\nINPUT_SPECS_PKLBIN=../spectra/%s.pklbin\nINPUT_ALIGNS=pairs_asp.bin\nOUTPUT_SPECS=pairs.pklbin\nOUTPUT_MODPOS=pairs_modpos.bin\n',specs_name);
	fprintf(fid,'TOLERANCE_PEAK=%s\nMAX_AA_JUMP=%s\nPENALTY_PTM=%s\nPENALTY_SAME_VERTEX=%s\n',params.TOLERANCE_PEAK,params.MAX_AA_JUMP,params.PENALTY_PTM,params.PENALTY_SAME_VERTEX);
	fclose(fid);

	status = dos(sprintf('%s/specalign specalign.params',params.EXE_DIR));    if status~=0 fprintf(1,'ERROR executing %s/specalign!\n',params.EXE_DIR); res=-1; return; end;
    [stars, stars_idx] = sps_genstars(specs, pairs_asp, pairs_po, peakTol, pmTol, str2num(params.RESOLUTION));
    sn_save_pklbin(stars,'../spectra/stars.pklbin');
else stars = sn_load_pklbin('../spectra/stars.pklbin');
end;

% filterstarpairs
if initial_stage <= STAGE_FILTERSTARPAIRS
	c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
	curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
    fprintf(1,'Starting stage: filterstarpairs (%s %s)\n',curDate,curTime);

    fid = fopen('filterstarpairs.params','w');   if fid<=0 fprintf(1,'ERROR opening filterstarpairs.params!\n'); res=-1; return; end;
	fprintf(fid,'13\nINPUT_SPECS_PKLBIN=../spectra/stars.pklbin\nINPUT_ALIGNS=pairs.bin\nINPUT_ALIGNSPA=pairs_po.dat\n');
	fprintf(fid,'OUTPUT_RATIOS=ratios.bin\nOUTPUT_RATIOSPA=ratios_po.bin\n');
	fprintf(fid,'TOLERANCE_PEAK=%s\nMAX_AA_JUMP=%s\nPENALTY_PTM=%s\nPENALTY_SAME_VERTEX=%s\nMIN_RATIO=-1.0\nGRAPH_TYPE=2\n',params.TOLERANCE_PEAK,params.MAX_AA_JUMP,params.PENALTY_PTM,params.PENALTY_SAME_VERTEX);
	fprintf(fid,'OUTPUT_SPECS=none\nOUTPUT_MODPOS=none\n');
	fclose(fid);
	status = dos(sprintf('%s/masab filterstarpairs.params',params.EXE_DIR));    if status~=0 fprintf(1,'ERROR executing %s/filterstarpairs!\n',params.EXE_DIR); res=-1; return; end;
    
    % Load and filter
    ratios = sn_load_binarray('ratios.bin','float');   ratios_po = sn_load_binarray('ratios_po.bin','float');
    stars_pairs = pairs(find(min(ratios(:,1:2)')'>=str2num(params.MIN_RATIO) & ratios(:,3)>=str2num(params.MIN_MATCHED_PEAKS)),:);
    stars_pairs_po = pairs_po(find(min(ratios_po(:,1:2)')'>=str2num(params.MIN_RATIO) & ratios_po(:,3)>=str2num(params.MIN_MATCHED_PEAKS)),:);
    [idx, idx_po] = sps_filter_samepep([specs{:,3}]', stars_pairs(:,1:2), stars_pairs(:,3:5), stars_pairs_po, 5);
    stars_pairs = stars_pairs(idx,:);   stars_pairs_po = stars_pairs_po(idx_po,:);
    sn_save_aligns('stars_pairs.bin',stars_pairs(:,1:2),stars_pairs(:,3:5));
    sps_save_aligns('stars_pairs_po.dat',stars_pairs_po);
    save stars_pairs stars_pairs stars_pairs_po;
else load stars_pairs;
end;

%
% Assembly and sequencing stage
%
cd('../assembly');
if initial_stage <= STAGE_ASSEMBLY
	c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
	curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
    fprintf(1,'Starting stage: assembly (%s %s)\n',curDate,curTime);
    
    fid = fopen('assembly.params','w');   if fid<=0 fprintf(1,'ERROR opening assembly.params!\n'); res=-1; return; end;
	fprintf(fid,'11\nINPUT_SPECS_PKLBIN=../spectra/stars.pklbin\nINPUT_ALIGNS=../aligns/stars_pairs.bin\nINPUT_ALIGNSPA=../aligns/stars_pairs_po.dat\n');
	fprintf(fid,'TOLERANCE_PEAK=%s\nMAX_AA_JUMP=%s\nPENALTY_PTM=%s\nPENALTY_SAME_VERTEX=%s\nMIN_RATIO=%s\nGRAPH_TYPE=2\n',params.TOLERANCE_PEAK,params.MAX_AA_JUMP,params.PENALTY_PTM,params.PENALTY_SAME_VERTEX,params.MIN_RATIO);
	fprintf(fid,'OUTPUT_SPECS=sps_seqs.pklbin\nOUTPUT_MODPOS=sps_seqs_modpos.bin\n');
	fclose(fid);
	status = dos(sprintf('%s/masab assembly.params',params.EXE_DIR));    if status~=0 fprintf(1,'ERROR executing %s/masab!\n',params.EXE_DIR); res=-1; return; end;
end

%
% Report results
%
if initial_stage <= STAGE_REPORT
	c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
	curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
    fprintf(1,'Starting stage: report (%s %s)\n',curDate,curTime);

    sps_seqs = sn_load_pklbin('sps_seqs.pklbin',1);
    sps_seqs_info  = sps_load_abinfo('component_info.bin');
    sps_seqs_comps = sps_load_bla('components.bla','int32');

    cd('..');
    sps_report(sps_seqs, sps_seqs_comps, str2num(params.TAG_LEN), stars, sps_seqs_info, peakTol, 'report/');
end;

res=0;

%
% De novo sequencing and evaluation
%
% cd('../spectra');
% if ~isempty(params.DENOVO)
%     if initial_stage <= STAGE_FILTERSTARPAIRS
%         if strcmp(params.DENOVO,'stars')
%             fid = fopen('denovo.params','w');   if fid<=0 fprintf(1,'ERROR opening denovo.params!\n'); res=-1; return; end;
% 			fprintf(fid,'11\nINPUT_SPECS_PKLBIN=stars_only.pklbin\nINPUT_ALIGNS=\nOUTPUT_SPECS=denovo.pklbin\n');
% 			fprintf(fid,'TOLERANCE_PEAK=%s\nTOLERANCE_PM=%s\nRESOLUTION=%s\nSPEC_TYPE_MSMS=0\nALL_SPECS_MS2=1\nMAX_AA_JUMP=3\nMISSING_PEAK_PENALTY=-4\nSTRICT_PM_TOLERANCE=0\n',params.TOLERANCE_PEAK,params.TOLERANCE_PM,params.RESOLUTION);
% 			fclose(fid);
% 			status = dos(sprintf('%s/ms3den denovo.params',params.EXE_DIR));    if status~=0 fprintf(1,'ERROR executing %s/tagsearch!\n',params.EXE_DIR); res=-1; return; end;
%         end
%         if strcmp(params.DENOVO,'multiple')
%             fid = fopen('denovo.params','w');   if fid<=0 fprintf(1,'ERROR opening denovo.params!\n'); res=-1; return; end;
% 			fprintf(fid,'11\nINPUT_SPECS_PKLBIN=%s.pklbin\nINPUT_ALIGNS=../aligns/pairs_stars.bin\nOUTPUT_SPECS=denovo.pklbin\n',specs_name);
% 			fprintf(fid,'TOLERANCE_PEAK=%s\nTOLERANCE_PM=%s\nRESOLUTION=%s\nSPEC_TYPE_MSMS=0\nALL_SPECS_MS2=1\nMAX_AA_JUMP=3\nMISSING_PEAK_PENALTY=-4\nSTRICT_PM_TOLERANCE=0\n',params.TOLERANCE_PEAK,params.TOLERANCE_PM,params.RESOLUTION);
% 			fclose(fid);
% 			status = dos(sprintf('%s/ms3den denovo.params',params.EXE_DIR));    if status~=0 fprintf(1,'ERROR executing %s/tagsearch!\n',params.EXE_DIR); res=-1; return; end;
%         end
%     end
%     denovo = sn_load_pklbin('denovo.pklbin');   % should replace the stars_only.pklbin input to tagsearch?
%     
%     if ~isempty(params.DENOVO_EVAL)
%         prevPeps = cell(size(specs,1),1);  for pivot=1:size(stars_indices,1) prevPeps{pivot}=''; end;
%         peptides = inspect_loadAnnots3(prevPeps, params.DENOVO_EVAL, '', '', 'mgf');
%         denovo(:,5) = peptides(stars_indices);
%         
%         stage82(denovo,'prm',.5,'',[],100,0,0,'denovo',0);
% 		load denovo_stage82 denovo_bypercs denovo_expPeaks;
% 		for i=1:size(denovo,1)
%             if denovo_bypercs(i,3)>denovo_bypercs(i,1)
%                 idx = 2:size(denovo{i,2},1)-1;   % Endpoints are always b-ions don't reverse them
%                 denovo{i,2}(idx,1)=denovo{i,3}-1-denovo{i,2}(idx,1);
%                 [foo,idxS]=sort(denovo{i,2}(:,1));
%                 denovo{i,2}=denovo{i,2}(idxS,:);
%             end;
% 		end;
% 		fprintf(1,'Flipped %d sequences.\n',size(find(denovo_bypercs(:,3)>denovo_bypercs(:,1)),1));
% 		
% 		denovo_dnstatsF = report_denovo(denovo, denovo(:,5), peakTol, str2num(params.MAX_MOD_MASS));
% 		denovo_tags = findLongestTags2(denovo,2,[],peakTol,'',[],str2num(params.MAX_MOD_MASS));
% %       masses = zeros(size(omannA3a_ms2_seqs,1),1); for i=1:size(masses,1) masses(i)=sum(getmasses4(omannA3a_ms2_seqs{i,5},'',[],100))+19; end;
%         denovo_stats = [100*denovo_dnstatsF(:,3)./(denovo_dnstatsF(:,3)+denovo_dnstatsF(:,5)) denovo_dnstatsF(:,3) denovo_tags-1]; 
%         [100*mean(denovo_dnstatsF(:,3)./(denovo_dnstatsF(:,3)+denovo_dnstatsF(:,5))) length(denovo) sum(denovo_dnstatsF(:,3)) mean(tags_ms2-1)]
%     end
% end
% 
% %
% % Tag generation + database search
% %
% cd('../spectra');
% if ~isempty(params.FASTA_DATABASE)
% 	if initial_stage <= STAGE_TAGSEARCH
% 		c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
% 		curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
%         fprintf(1,'Starting stage: tagsearch (%s %s)\n',curDate,curTime);
% 	
%         fid = fopen('tagsearch.params','w');   if fid<=0 fprintf(1,'ERROR opening tagsearch.params!\n'); res=-1; return; end;
% 		fprintf(fid,'9\nINPUT_SPECS_PKLBIN=../spectra/stars_only.pklbin\nINPUT_FASTA=../%s\nOUTPUT_PEPTIDES=peptides_tagsearch.txt\n',params.FASTA_DATABASE);
% 		fprintf(fid,'TOLERANCE_PEAK=%s\nTOLERANCE_PM=%s\nTAG_LEN=%s\nDOUBLE_AA_JUMPS=%s\nMATCH_TAG_FLANKING_MASSES=%s\nMAX_PARSIMONY=1\n',params.TOLERANCE_PEAK,params.TOLERANCE_PM,params.TAG_LEN,params.DOUBLE_AA_JUMPS,params.MATCH_TAG_FLANKING_MASSES);
% 		fclose(fid);
% 		status = dos(sprintf('%s/tagsearch tagsearch.params',params.EXE_DIR));    if status~=0 fprintf(1,'ERROR executing %s/tagsearch!\n',params.EXE_DIR); res=-1; return; end;
%     end;
%     
% 	tagsearch_peptides = sn_load_tagsearch(stars(stars_indices,:),'peptides_tagsearch.txt');
% 	proteins = cell(size(stars,1),1); for pepIdx=1:size(stars,1) proteins{pepIdx}=''; end;
% 	for pivot=1:length(stars_indices) if tagsearch_peptides{pivot,2}==1 stars{stars_indices(pivot),5}=tagsearch_peptides{pivot,3}{1}; proteins{stars_indices(pivot)}=tagsearch_peptides{pivot,1}{1,5}; end; end;
% 	peptides = stars(:,5);
% else
%     proteins = cell(size(stars,1),1);    for i=1:size(proteins,1) proteins{i}=''; end;    peptides = proteins;
% end;
% 
% %
% % Spectral Networks: propagation of annotations
% %
% cd('../specnets');    minEI = str2num(params.MIN_PERC_EXPINT);    minTP = str2num(params.MIN_PERC_TP);
% if initial_stage <= STAGE_SPECNETS
% 	c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
% 	curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
%     fprintf(1,'Starting stage: specnets (%s %s)\n',curDate,curTime);
% 
%     [stars_dbannot, eitp] = sn_enforceannots(stars, peptides, 'prm', peakTol, str2num(params.MAX_MOD_MASS), minEI, minTP);
% 	sn_save_binarray('../spectra/stars_dbannot.bin', round(10000*eitp), 'uint16');
% 	sn_save_pklbin(stars_dbannot,'../spectra/stars_dbannot.pklbin');    peptides_dbannot=stars_dbannot(:,5);   save peptides_dbannot peptides_dbannot;
% 	
% 	fid = fopen('pathproj.params','w');  if fid<=0 fprintf(1,'ERROR opening pathproj.params!\n'); res=-1; return; end;
% 	fprintf(fid,'6\nINPUT_SPECS_PKLBIN=../spectra/stars_dbannot.pklbin\nINPUT_ALIGNS=../aligns/pairs_stars.bin\nINPUT_ANNOTATED=../spectra/stars_dbannot.bin\nOUTPUT_ANNOTINFO=nets_annots.bin\n');
% 	fprintf(fid,'MIN_PERC_EXPINT=%.0f\nMIN_PERC_TP=%.0f\n',10000*minEI, 10000*minTP);
% 	fclose(fid);
% 	status = dos(sprintf('%s/pathproj pathproj.params',params.EXE_DIR));    if status~=0 fprintf(1,'ERROR executing %s/pathproj!\n',params.EXE_DIR); res=-1; return; end;
% else
%     stars_dbannot = sn_load_pklbin('../spectra/stars_dbannot.pklbin'); if size(stars_dbannot,1)~=size(peptides,1) fprintf(1,'ERROR loading necessary data (stars_dbannot) - rerun with INITIAL_STAGE=specnets or earlier (in %s).\n',paramsFN); res=-1; return; end;
%     if exist('peptides_dbannot.mat')==2 load peptides_dbannot; else peptides_dbannot=[]; end;
%     if size(stars_dbannot,1)~=size(peptides_dbannot,1) fprintf(1,'ERROR loading necessary data (peptides_dbannot) - rerun with INITIAL_STAGE=specnets or earlier (in %s).\n',paramsFN); res=-1; return; end;
%     stars_dbannot(:,5) = peptides_dbannot;
%     eitp = double(sn_load_binarray('../spectra/stars_dbannot.bin','uint16'))/10000; if size(stars_dbannot,1)~=size(eitp,1) fprintf(1,'ERROR loading necessary data (eitp) - rerun with INITIAL_STAGE=specnets or earlier (in %s).\n',paramsFN); res=-1; return; end;
% end
% 
% if initial_stage <= STAGE_REPORT
% 	c=clock;   curDate=sprintf('%4.0f/%2.0f/%2.0f',c(1),c(2),c(3));    curDate(find(curDate==' '))='0';
% 	curTime=sprintf('%2.0f:%2.0f:%2.0f',c(4),c(5),c(6));    curTime(find(curTime==' '))='0';
%     fprintf(1,'Starting stage: report (%s %s)\n',curDate,curTime);
% 
%     [stars_dbannot, pp_matches, pp_all] = sn_load_pathproj(stars_dbannot, minEI, minTP, str2num(params.MAX_MOD_MASS), 'nets_annots.bin', 'stars');
% 	idx = min(find(pp_matches(:,2)>0));
%     if ~isempty(idx) for matchIdx=idx:size(pp_matches,1) proteins{pp_matches(matchIdx,1)}=proteins{pp_matches(matchIdx,3)}; end; end;
% 
% 	% Recover previous unreliable peptide annotations
% 	for specIdx=1:size(stars_dbannot,1) if isempty(stars_dbannot{specIdx,5}) stars_dbannot{specIdx,5}=peptides{specIdx}; end; end;
% 	
% 	% Compute % b/y-ions and explained intensity
% 	if ~isempty(specs_raw)
%         sn_evalspecs([specs_raw(:,1:4) stars_dbannot(:,5)],'msms', peakTol,'',[],str2num(params.MAX_MOD_MASS),0,1,'specs_raw_stats',0);
%         load specs_raw_stats_sn_evalspecs expInt;
%         eiMSMS = sum(expInt(:,1:4)')';
% 	else eiMSMS = []; end;
% 	
%     sn_evalspecs(stars_dbannot,'prm', peakTol,'',[],str2num(params.MAX_MOD_MASS),0,1,'stars_dbannot',0);
% 	load stars_dbannot_sn_evalspecs expInt bypercs;
% 	eiPRM = [sum(expInt(:,1:3)')' max(bypercs(:,1:4)')'];
% 	
%     cd('..');
%     sn_report_specnets(stars_dbannot, pairs_stars, pp_matches, eiPRM, minEI, minTP, params.GRAPHVIZ_CMD, filenames, proteins, eiMSMS);
% end;
% 
% res=0;
