#include "tags.h"
#include "aminoacid.h"

bool cmp_tag(Tag t1, Tag t2) { return t1.score>t2.score; } // Used to sort tags by descending tag score

// FindMassNeighbors - indices[i] contains the indices of the peaks in spec
//    such that abs( mass - jumps[i] - spec[any peak in indices[i]][0] ) <= peakTol
//    idxSupremum - index of a peak in spec with a mass higher than mass-min(jumps) (optional)
void FindMassNeighbors(float mass, Spectrum &spec, AAJumps &jumps,
                         vector<list<short> > &indices, float peakTol, int idxSupremum=-1) {
	if(idxSupremum<0 or idxSupremum>=(int)spec.size()) idxSupremum=(int)spec.size()-1; if(idxSupremum<0) return;
	
	unsigned int jumpIdx; 
	int peakIdx=idxSupremum;
	indices.resize(jumps.size()); for(jumpIdx=0; jumpIdx<jumps.size(); jumpIdx++) indices[jumpIdx].clear();
	if(spec.size()==0) return;
	float curMass;

	for(jumpIdx=0; jumpIdx<jumps.size(); jumpIdx++) {
	  curMass=mass-jumps[jumpIdx];   if(curMass<-peakTol) break;   if(peakIdx<0) peakIdx=0;
		while(peakIdx<(int)spec.size() and spec[peakIdx][0]<=curMass+peakTol+0.0001) peakIdx++; if(peakIdx>=(int)spec.size()) peakIdx=(int)spec.size()-1;
		while(peakIdx>=0 and spec[peakIdx][0]>curMass+peakTol+0.0001) peakIdx--;
		while(peakIdx>=0 and spec[peakIdx][0]>curMass-peakTol-0.0001) indices[jumpIdx].push_back(peakIdx--); 
	}
}

unsigned int ExtractTags(Spectrum &spec, list<Tag> &tags, float peakTol,
                           unsigned int tagLen, unsigned int maxNumJumps) {
	vector<vector<vector<list<Tag> > > > partialTags; // Table containing all partial tags
						// Position [i,j,k] contains all the tags of length k 
						//   with j di-peptide jumps and ending on the i-th spectrum peak
		                // dimension 1 - Index of the spectrum peak where the tags end
		                // dimension 2 - Number of used di-peptide jumps (usually 0-2)
		                // dimension 3 - Tag length
	AAJumps jumps(1);
	unsigned int diIdx, peakIdx, lenIdx, aaIdx, 
	             numPeaks=spec.size(), numJumps=jumps.size();
	vector<list<short> > aaNeighs(numJumps),  // Peaks with mass one aa to the left of current peak
	                     diNeighs(numJumps);  // Peaks with mass two aa to the left of current peak
	Tag curTag(tagLen);

	// Initializations
	partialTags.resize(numPeaks);   tags.clear();
	for(peakIdx=0; peakIdx<numPeaks; peakIdx++) {
		partialTags[peakIdx].resize(maxNumJumps+1);
		for(diIdx=0; diIdx<=maxNumJumps; diIdx++)	{
			partialTags[peakIdx][diIdx].resize(tagLen);
			for(lenIdx=0; lenIdx<tagLen; lenIdx++) partialTags[peakIdx][diIdx][lenIdx].clear();
		}
	}

	// Generate tags
	for(peakIdx=0; peakIdx<numPeaks; peakIdx++) {
		FindMassNeighbors(spec[peakIdx][0], spec, jumps, aaNeighs, peakTol, (int)peakIdx);
		unsigned int lastAAneigh=peakIdx;  // Index of the lowest-mass detected aa neighbor of peakIdx (used for diNeighs searches)
		
		for(aaIdx=0; aaIdx<numJumps; aaIdx++) {

			// No neighbors 1-aa away, look for neighbors 2 aa masses away
			if(aaNeighs[aaIdx].size()==0 and maxNumJumps>0) {
				FindMassNeighbors(spec[peakIdx][0]-jumps[aaIdx], spec, jumps, diNeighs, peakTol, (int)lastAAneigh);

				for(unsigned int aaDiIdx=0; aaDiIdx<numJumps; aaDiIdx++) {
					for(list<short>::iterator neigh = diNeighs[aaDiIdx].begin(); neigh!=diNeighs[aaDiIdx].end(); neigh++) {

						// Initialize tags of length 2 with no middle peak
						curTag.sequence[0]=(char)aaDiIdx;          curTag.sequence[1]=(char)aaIdx;
						curTag.score=spec[*neigh][1]+spec[peakIdx][1];
						curTag.flankingPrefix = spec[*neigh][0];   curTag.flankingSuffix = spec.parentMass-AAJumps::massMH-spec[peakIdx][0];
						partialTags[peakIdx][1][1].push_back(curTag);
						if(tagLen==2) tags.push_back(curTag);

						// Extend to tags of length 3+
						for(diIdx=1; diIdx<=maxNumJumps; diIdx++) {
							for(lenIdx=2; lenIdx<tagLen; lenIdx++) {
								if(partialTags[*neigh][diIdx-1][lenIdx-2].size()==0) continue; 
								for(list<Tag>::iterator tagIter=partialTags[*neigh][diIdx-1][lenIdx-2].begin(); tagIter!=partialTags[*neigh][diIdx-1][lenIdx-2].end(); tagIter++) {
									curTag = *tagIter;               
									curTag.sequence[lenIdx-1]=(char)aaDiIdx;  curTag.sequence[lenIdx]=(char)aaIdx; 
									curTag.score+=spec[peakIdx][1];           curTag.flankingSuffix = spec.parentMass-AAJumps::massMH-spec[peakIdx][0];
									partialTags[peakIdx][diIdx][lenIdx].push_back(curTag);
									if(lenIdx==tagLen-1) tags.push_back(curTag);
								}
							}
						}
					}
				}
				continue;
			}

			// Handle neighbors 1-aa away
			for(list<short>::iterator neigh = aaNeighs[aaIdx].begin(); neigh!=aaNeighs[aaIdx].end(); neigh++) {
				lastAAneigh = *neigh;

				// Initialize tags of length 1
				curTag.sequence[0]=(char)aaIdx;            curTag.score=spec[*neigh][1]+spec[peakIdx][1];
				curTag.flankingPrefix = spec[*neigh][0];   curTag.flankingSuffix = spec.parentMass-AAJumps::massMH-spec[peakIdx][0];
				partialTags[peakIdx][0][0].push_back(curTag);
				if(tagLen==1) tags.push_back(curTag);

				// Extend to tags of length 2+
				for(diIdx=0; diIdx<=maxNumJumps; diIdx++) {
					for(lenIdx=1; lenIdx<tagLen; lenIdx++) {
						if(partialTags[*neigh][diIdx][lenIdx-1].size()==0) continue;
						for(list<Tag>::iterator tagIter=partialTags[*neigh][diIdx][lenIdx-1].begin(); tagIter!=partialTags[*neigh][diIdx][lenIdx-1].end(); tagIter++) {
							curTag = *tagIter;               curTag.sequence[lenIdx]=(char)aaIdx;
							curTag.score+=spec[peakIdx][1];  curTag.flankingSuffix = spec.parentMass-AAJumps::massMH-spec[peakIdx][0];
							partialTags[peakIdx][diIdx][lenIdx].push_back(curTag);
							if(lenIdx==tagLen-1) tags.push_back(curTag);
						}
					}
				}
			}
		}
	}
	
	tags.sort(cmp_tag);
	return(tags.size());
}
