#include <fstream>
#include <math.h>
#include <algorithm>
#include <time.h>
#include <sstream>
#include <vector>
// #include "denovo_seq.h"
#include "peptide_spectrum_score.h"
#include "util.h"
#include "src/PepNovo/auxfun.h"
#include "src/PepNovo/FastaDB.h"
#include "src/PepNovo/PeakList.h"
#include "src/PepNovo/SpectraFile.h"
#include "src/PepNovo/SpectraAggregator.h"
#include "src/PepNovo/SingleSpectrumHeader.h"
#include "src/PepNovo/Spectrum.h"
#include "src/PepNovo/SpectraList.h"
#include "src/PepNovo/AnnotatedSpectrum.h"
#include "src/PepNovo/AllScoreModels.h"

// test modification for svn test

using namespace std;


void print_help(const char *message) 
{

	printf("***************************************************************************\n%s\n",message);
	printf("cycloquest_html : database search for linear/cyclic peptides\n");
	printf("Required arguments:\n");
	printf("-------------------\n");

	printf("<path to input mzxml/mgf file>\n");
	printf("<path to input fasta file>\n");
  	printf("\n");

	printf("Optional cycloquest_html arguments: \n");
	printf("----------------------------- \n"); 
	
	printf("-out <path to output html file>                                      (default : out.html)\n");
	printf("-linear                                                   search only for linear peptides\n");
	printf("-cyclic                                                   search only for cyclic peptides\n");
	printf("-num_top X                               number of top search hits returned (default: 20)\n");
	printf("-num_peaks X 		                             number of top peaks kept in spectra \n");
	printf("-parent_mass X                          Parent Mass in Da (Default : from mzxml/mgf file)\n");
  	printf("-charge                                                   (Default : from mzxml/mgf file)\n");
	printf("-add_decoy                                                          	  search decoy DB\n");
  	printf("-precursor_ion_tolerance               		                         Default : 0.5 Da\n");
  	printf("-product_ion_tolerance     	                                	 Default : 0.5 Da\n");
	printf("-scan_num X                                                   Default : analyze all scans\n");
	printf("-filter_parent_mass X Y         only analyze the species with parent mass between X and Y\n");
        printf("-num_cys X                                   only acept peptides with X cysteines\n");
  	printf("-reduced     	                        		                          	 \n");
  	printf("-IAM_reduced 	                                                 			 \n");
  	printf("-MAL_reduced    	                      	                             		 \n");
  	printf("-PFA_reduced  	                                                      			 \n");
  	printf("-NaBH4_reduced                          			                         \n");
  	printf("-CMet_reduced    	                      			                         \n");
  	printf("-NIPIA_reduced                          			                         \n");
	exit(1);
}


int main(int argc, char **argv)
{
  vector<int> correct_seq;
  bool add_decoy_flag = false;
  int cys_num;
  bool enforce_cys_num_flag = false;
  int charge;
  int use_charge_flag = false;
  int scan_num;
  bool use_scan_num_flag = false;
  float min_anal_pm, max_anal_pm;
  bool use_anal_pm_flag = false;
  int search_cyc_lin = 2; // 0 : just linear, 1: just cyclic, 2: both
  int num_top_hits = 20;
  int num_peaks = -1; // use 5 peak in 50Da filter
  float parent_mass;
  bool use_parent_mass_flag = false;
  float precursor_ion_tolerance = 0.5;
  float product_ion_tolerance = 0.5;
  seedRandom(112233);
  AllScoreModels model;
  bool use_large_db_model = true;
  char initial_model[] = "LTQ_TRYP";
  model.read_model(initial_model, true);
  model.read_rank_models(initial_model, true);
  Config *config = model.get_config();
  string PTM_string = "C-1"; // native
  int redox_type;

  	char input_file[256];
  	char fasta_file[256];
	char output_file[256] = "out.html";

	int i=0;
	i++;
	if(argc == i) {
		print_help("Missing input spectrum file!");
	} else {
		strcpy(input_file, argv[i]);
	}
	i++;
	if(argc == i) {
		print_help("Missing input fasta file!");
	} else {
		strcpy(fasta_file, argv[i]);
	}
  	i++;
	while (i<argc)
	{
		if (! strcmp(argv[i],"-add_decoy"))
		{
		      add_decoy_flag = true;
		      i++;
		}
		if (! strcmp(argv[i],"-linear"))
		{
		      search_cyc_lin = 0;
		      i++;
		}
		else if (! strcmp(argv[i],"-cyclic"))
		{
		      search_cyc_lin = 1;
		      i++;
		}
		else if (! strcmp(argv[i],"-out"))
		{
			if (++i == argc)
				print_help("Missing output file!");
			strcpy(output_file, argv[i]);
			i++;
		}
		else
		if (! strcmp(argv[i],"-num_top"))
		{
			if (++i == argc)
				print_help("Missing top hits number!");
			num_top_hits = atoi(argv[i]);	
		        i++;
		}
		else
		if (! strcmp(argv[i],"-num_peaks"))
		{
			if (++i == argc)
				print_help("Missing top peaks number!");
			num_peaks = atoi(argv[i]);
		        i++;
	        }
		else if (! strcmp(argv[i],"-parent_mass"))
		{
			if (++i == argc)
				print_help("Missing precursor ion mass!");
		        parent_mass = atof(argv[i]);
		        use_parent_mass_flag = true;
		        i++;
		}
		else if (! strcmp(argv[i],"-precursor_ion_tolerance"))
		{
			if (++i == argc)
				print_help("Missing parent ion tolerance!");
		        precursor_ion_tolerance = atof(argv[i]);
		        i++;
		}
		else if (! strcmp(argv[i],"-product_ion_tolerance"))
		{
			if (++i == argc)
				print_help("Missing product ion tolerance!");
		        product_ion_tolerance = atoi(argv[i]);
		        i++;
		}
		else if (! strcmp(argv[i],"-num_cys"))
		{
			if (++i == argc)
				print_help("Missing number of cysteines!");
		        cys_num = atoi(argv[i]);
		        enforce_cys_num_flag = true;
		        i++;
		}
		else if (! strcmp(argv[i],"-charge"))
		{
			if (++i == argc)
				print_help("Missing charge!");
		        charge = atoi(argv[i]);
		        use_charge_flag = true;
		        i++;
		}
		else if (! strcmp(argv[i],"-scan_num"))
		{
			if (++i == argc)
				print_help("Missing scan num!");
		        scan_num = atoi(argv[i]);
		        use_scan_num_flag = true;
		        i++;
		}
		else if (! strcmp(argv[i],"-filter_parent_mass"))
		{
			if (++i == argc)
				print_help("Missing min parent mass!");
		        min_anal_pm = atof(argv[i]);
			if (++i == argc)
				print_help("Missing max parent mass!");
		        max_anal_pm = atof(argv[i]);
		        use_anal_pm_flag = true;
		        i++;
		}
		else if (! strcmp(argv[i],"-reduced"))
		{
		      PTM_string = "";
		      i++;
		}
		else if (! strcmp(argv[i],"-MAL_reduced"))
		{
		      PTM_string = "C+97";
		      i++;
		}
		else if (! strcmp(argv[i],"-IAM_reduced"))
		{
		      PTM_string = "C+57";
		      i++;
		}
		else if (! strcmp(argv[i],"-NaBH4_reduced"))
		{
		      PTM_string = "C-32:M-46";
		      i++;
		}
		else if (! strcmp(argv[i],"-PFA_reduced"))
		{
		      PTM_string = "C+48";
		      i++;
		}
		else if (! strcmp(argv[i],"-CMet_reduced"))
		{
		      PTM_string = "C+58";
		      i++;
		}
		else if (! strcmp(argv[i],"-NIPIA_reduced"))
		{
		      PTM_string = "C+99";
		      i++;
		}
	        else {
		      stringstream error_str;
		      error_str << "invalid argument : ";
		      error_str << argv[i];
		      print_help(error_str.str().c_str());
	        }
	}

  // cout << "here" << endl;

  config->apply_selected_PTMs(PTM_string.c_str());
  PeptideRankScorer *db_score = (PeptideRankScorer *)model.get_rank_model_ptr(0);

  vector<float> acc_thresh;
  acc_thresh.clear();
  acc_thresh.push_back(product_ion_tolerance);
  acc_thresh.push_back(product_ion_tolerance/2);
  acc_thresh.push_back(product_ion_tolerance/3);

  Offset offset;
  offset.add_off(0,1);

  AnnotatedSpectrumCounterList spectrum_vector;
  ReadSpecModel read_spec_model;
  read_spec_model.setMinMsLevel(1);
  read_spec_model.setMinPeakCount(0);
  read_spec_model.filter_type = 1;
  read_spec_model.set_binning();
  read_spec_model.set_acc_thresh(acc_thresh);
  if(use_scan_num_flag) {
	  read_spec_model.set_start_scan(scan_num);
	  read_spec_model.set_stop_scan(scan_num+1);
  }
  if(use_anal_pm_flag) {
	  read_spec_model.set_min_max_anal_pm(min_anal_pm, max_anal_pm);
  }
  if(use_charge_flag)
  	read_spec_model.add_charge(charge);
  if(use_parent_mass_flag)
  	read_spec_model.fix_mass(parent_mass);
  read_spec_model.set_num_top_peaks(num_peaks);
  cout << use_scan_num_flag << endl;
  // cout << "here" << endl;

  cout << "reading spectra ... ";
  read_spec_model.read_spectra(config, input_file, spectrum_vector);
  cout << spectrum_vector.size() << " spectra." << endl;
  spectrum_vector.print_pm_charge();

  FastaDB fdb;
  PeptideCounterList pep_list;
  cout << "creating database from fasta file ... " << endl;
  fdb.create_db_from_fasta(fasta_file, config, false);
  if(add_decoy_flag) {
	  fdb.add_decoy();
  	  pep_list.decoy_flag = true;
  }
  fdb.set_config(config);
  
  

  int linear_scoring = 0;
  int cyclic_scoring = 1;

  vector<float> pm_threshold;
  pm_threshold.clear();
  pm_threshold.push_back(precursor_ion_tolerance);
  pm_threshold.push_back(precursor_ion_tolerance);
  pm_threshold.push_back(precursor_ion_tolerance);
  
  float cyclic_offset = 0;
  float linear_offset = -MASS_H2O;
  vector<float> final_offsets;
  vector<int> scoring_methods;
  if(search_cyc_lin == 0 || search_cyc_lin == 2) {
	  final_offsets.push_back(linear_offset);
	  scoring_methods.push_back(0);
  }

  if(search_cyc_lin == 1 || search_cyc_lin == 2) {
	  final_offsets.push_back(cyclic_offset);
	  scoring_methods.push_back(1);
  }

  pep_list.set_model(model);
  pep_list.set_num_top_hits(num_top_hits);

  int score_type = 0;

  // fdb.sort_by_pval_est();

  if(enforce_cys_num_flag)
	fdb.enforce_num_cys(cys_num);
  fdb.FilterDB(pep_list, spectrum_vector, pm_threshold, final_offsets, scoring_methods, acc_thresh, offset, score_type); 

  SearchResults sr;
   
  if(enforce_cys_num_flag)
	sr.enforce_cys(cys_num);
  sr.set_protein_names(fdb.get_protein_names());
  // sr.set_max_num_top_hits(num_top_hits);
  sr.parse_from_pep_list(pep_list, spectrum_vector);
  sr.print_html(output_file);

  cout<<"DONE"<<endl;

  return 0;

}



