#include "peptide_spectrum_score.h"

extern const float H2O_mass = 18.01056;
extern const float charge_mass = 1.007276;
extern const float neutron_mass = 1.008664;

score_t peptide_spectrum_score(AnnotatedSpectrumAdd& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type);
score_t peptide_spectrum_score(AnnotatedSpectrumAdd& input_spectrum, PeptideAdd& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type);

void PvalModel::set_pval_norm(AnnotatedSpectrumCounter& input_spectrum, PeptideCounter& pep, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type, bool pval_flag, bool norm_flag) {
  // cout << "thresh : " << thresh << endl;
  // cout << pval_flag << " " << norm_flag << " " << num_peptide << " ";
  if(pval_flag || norm_flag) {
	PeptideAdd sample_pep;
	score = peptide_spectrum_score(input_spectrum, pep, max_charge, acc_thresh, offset, do_cyclic, score_type);
	int temp_score;
	int num_over = 0;
	int sum_score = 0;
	int sum_square_score = 0;
	// cout << "main score : " << main_score << endl;
	for(int ind = 0; ind < num_peptide; ind++) {
	    //cout << "mass : " << pep.get_mass() << endl;
	    //cout << "len : " << pep.get_length() << endl;
	    //cout << "str : " << pep.getPeptideStr() << endl;
	    sample_pep = generate_random_peptide(pep);
	    // cout << sample_pep.get_mass() << endl;
	    // cout << sample_pep.get_length() << endl;
	    // cout << sample_pep.get_amino_acid_masses().size() << endl;
	    temp_score = peptide_spectrum_score(input_spectrum, sample_pep, max_charge, acc_thresh, offset, do_cyclic, score_type);
    	    // cout << max_charge << " " << acc_thresh.size() << " " << acc_thresh[0] << " " << do_cyclic << " " << score_type << endl;
	    // cout << temp_score << endl;
	    // cout << score << endl;
	    if(temp_score>score)
		num_over++;
	    sum_score += temp_score;
	    sum_square_score += temp_score*temp_score;
	}
	mean_val = float(sum_score)/num_peptide;
	float mean_square_val = float(sum_square_score)/num_peptide;
	std_val = sqrt(mean_square_val - mean_val*mean_val);
  // cout << mean_val << " " << std_val << endl;
  if(norm_flag)
  	pep.norm_score = float(score - mean_val)/std_val;
  if(pval_flag)
  	pep.pval = float(num_over)/num_peptide;		
  }
}

PeptideAdd PvalModel::generate_random_peptide(PeptideAdd& pep) {
	PeptideAdd sam_pep;
	bool flag = false;
	while(flag == false) {
		flag = true;
		sam_pep.generate_random_peptide(config, pep.get_length(), 1);
    // cout << "len : " << sam_pep.get_length() << endl;
    // cout << "original len : " << pep.get_length() << endl;
    // cout << "mass : " << sam_pep.get_mass() << endl;
		if(enforce_mass_flag && abs(sam_pep.get_mass() - pep.get_mass())>thresh)
			flag = false;
		if(enforce_cys_flag && (calc_num_aa(pep, Cys) != calc_num_aa(sam_pep, Cys)))
			flag = false;
	}
	return sam_pep;
}



score_t peptide_spectrum_score_intens_cyclic(AnnotatedSpectrumAdd& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset) {

    int n = seq.size();
    int I = 0;
    score_t final_score = 0;
    
    int spec_val_len = (n*(n-1)+1);
    float spec_val[max_charge][spec_val_len];
    float mat_val[n][n-1];
    float temp_val;
    float seq_rep[2*n];

    for(int ind = 0; ind<n; ind++) {
	seq_rep[ind] = seq[ind];
	seq_rep[ind+n] = seq[ind];
    }
  
    float parent_mass = 0.f;
    for(int i = 0; i < seq.size(); i++){
        parent_mass += seq[i];
     }

    int mass_vector_size = input_spectrum.getNumPeaks();
    float mass_vector[mass_vector_size];
    float intensity_vector[mass_vector_size];
    for(int ind = 0; ind<mass_vector_size; ind++) {
	mass_vector[ind] = input_spectrum.getPeakMass(ind);
	intensity_vector[ind] = input_spectrum.getPeakIntensity(ind);	
    }

    for(int charge_number = 1; charge_number <= max_charge; charge_number++){
	I = 0;
        for(int pos_1 = 0; pos_1<n; pos_1++) {
	    temp_val = seq_rep[pos_1];
       	    mat_val[pos_1][0] = temp_val;
	    spec_val[charge_number-1][I] = (temp_val+charge_mass*charge_number)/charge_number;
	    I++;
	}   
        for(int len_seq = 1; len_seq<n-1; len_seq++){
	    for(int pos_1 = 0; pos_1<n; pos_1++){
	        temp_val = mat_val[pos_1][len_seq-1] + seq_rep[pos_1+len_seq];
		mat_val[pos_1][len_seq] = temp_val;
		spec_val[charge_number-1][I] = (temp_val+charge_mass*charge_number)/charge_number;
		// cout << spec_val[charge_number-1][I] << endl;
		I++;
	    }
        }
        spec_val[charge_number-1][I] = (parent_mass + charge_mass*charge_number)/(charge_number);
        I++;
	final_score += intersect_threshold_intensity(spec_val[charge_number-1], spec_val_len, mass_vector, intensity_vector, mass_vector_size, acc_thresh[charge_number-1]);	
    }
return final_score;
}

void AnnotRes::construct_theoric_spec_repeat_cyclic(vector< vector<float> > seq, int max_charge, vector<float> acc_thresh, Offset offset) {

    int n = seq[0].size();
    int I = 0;
    score_t final_score = 0;
    int position_stop;

    float mat_val[seq.size()][n][n-1];
    float temp_val;
    vector<vector <float> > seq_rep;
    seq_rep.resize(seq.size());
    spec_val.resize(seq.size());

    for(int seq_ind = 0; seq_ind < seq.size();seq_ind++) {
	seq_rep[seq_ind].resize(2*n);
        for(int ind = 0; ind<n; ind++) {
      	    seq_rep[seq_ind][ind] = seq[seq_ind][ind];
    	    seq_rep[seq_ind][ind+n] = seq[seq_ind][ind];
        }
    }
  
    vector<float> parent_mass;
    parent_mass.resize(seq.size());
    for(int seq_ind = 0; seq_ind<seq.size(); seq_ind++) {
	parent_mass[seq_ind] = 0;
	for(int i = 0; i < seq[seq_ind].size(); i++){
            parent_mass[seq_ind] += seq[seq_ind][i];
	}
    }
     
     if(pep_set_flag) {
	pep_str_rep = pep_string;
     	for(int ind = 0; ind < pep_string.size();ind++) {
     	    pep_str_rep.push_back(pep_string[ind]);
	}
     }

    for(int charge_number = 1; charge_number <= max_charge; charge_number++){
	for(int offset_ind = 0; offset_ind < offset.val.size() ; offset_ind++) {
            for(int pos_1 = 0; pos_1<n; pos_1++) {
	        for(int seq_ind = 0; seq_ind < seq.size(); seq_ind++) {
	    	    temp_val = seq_rep[seq_ind][pos_1];
	            mat_val[seq_ind][pos_1][0] = temp_val;
		    spec_val[seq_ind].push_back((temp_val+charge_mass*charge_number+offset.val[offset_ind])/charge_number);
		}
		charge_num.push_back(charge_number);
		position.push_back(pos_1);
       	        length.push_back(1);
		position_stop = pos_1+1;
		if(position_stop>=n)
		    position_stop = position_stop - n;
		stop.push_back(position_stop);
		
		off.push_back(offset_ind);	
	    }
	    for(int len_seq = 1; len_seq<n-1; len_seq++){
  	        for(int pos_1 = 0; pos_1<n; pos_1++){
		    for(int seq_ind = 0; seq_ind <seq.size(); seq_ind++) {
		        temp_val = mat_val[seq_ind][pos_1][len_seq-1] + seq_rep[seq_ind][pos_1+len_seq];
		    	mat_val[seq_ind][pos_1][len_seq] = temp_val;
		    	spec_val[seq_ind].push_back((temp_val+charge_mass*charge_number+offset.val[offset_ind])/charge_number);
		    }
		    charge_num.push_back(charge_number);
		    position.push_back(pos_1);
		    length.push_back(len_seq+1);
		    off.push_back(offset_ind);	
		    position_stop = pos_1+len_seq+1;
		    if(position_stop>=n)
			position_stop = position_stop - n;
                    stop.push_back(position_stop);
		}
	    }
	    for(int seq_ind = 0; seq_ind < seq.size(); seq_ind++)
       	        spec_val[seq_ind].push_back((parent_mass[seq_ind]+charge_mass*charge_number+offset.val[offset_ind])/(charge_number));
	    charge_num.push_back(charge_number);
	    position.push_back(-1);
	    length.push_back(n);
	    stop.push_back(-1);
	    off.push_back(offset_ind);	
	}
    }

}
    
void AnnotRes::construct_theoric_spec_repeat_cyclic(vector<PeptideAdd> pep_vec, int max_charge, vector<float> acc_thresh, Offset offset) {
    vector< vector <float> > seq;
    seq.resize(pep_vec.size());
    for(int ind = 0; ind < pep_vec.size(); ind++)
	seq[ind] = pep_vec[ind].get_amino_acid_masses();
    construct_theoric_spec_repeat_cyclic(seq, max_charge, acc_thresh, offset);
}

void AnnotRes::peptide_spectrum_score_repeat_cyclic(AnnotatedSpectrumCounter& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset) {

    vector< vector<float> > seq_list;
    seq_list.push_back(seq);
    construct_theoric_spec_repeat_cyclic(seq_list, max_charge, acc_thresh, offset);
    
    // cout << "spec val size : " << spec_val[0].size() << endl;
    num_start.resize(seq.size());
    num_end.resize(seq.size());
    for(int I  = 0; I < spec_val[0].size(); I++) {
	for(int spec_ind = 0; spec_ind < input_spectrum.getNumPeaks(); spec_ind++) {
	    if(abs(spec_val[0][I] - input_spectrum.getPeakMass(spec_ind) )< acc_thresh[charge_num[I]-1]) {
	        spec_id_vec.push_back(spec_ind);
		spec_val_vec.push_back(spec_val[0][I]);
		charge_num_vec.push_back(charge_num[I]); 
		mass_error_vec.push_back(spec_val[0][I] - input_spectrum.getPeakMass(spec_ind));
		position_vec.push_back(position[I]);
		length_vec.push_back(length[I]);
		offset_vec.push_back(off[I]);
		if(position[I]>=0) {
			num_start[position[I]]++;
			num_end[stop[I]]++;
		}
		if(pep_set_flag == true) {
		    string pep_str;
		    if(position[I]>=0) {
			    for(int id = position[I]; id < position[I] + length[I]; id++) {		
			    	pep_str.push_back(pep_str_rep[id]);
			    }
		   } else {
		   	    pep_str = pep_string;	    
		   }
		   		
		   pep_str_vec.push_back(pep_str);
		}
	    }
	}	 
    }
}

void AnnotRes::peptide_spectrum_score_repeat_cyclic_mult(vector<AnnotatedSpectrumCounter*> spec_vec, vector<PeptideAdd> pep_vec, int max_charge, vector<float> acc_thresh, Offset offset) {

    set_peptide(pep_vec[0]);
    construct_theoric_spec_repeat_cyclic(pep_vec, max_charge, acc_thresh, offset);
    stat.resize(spec_vec.size()+1);
    for(int ind = 0; ind < stat.size(); ind++)
	stat[ind] = 0;
    for(int I  = 0; I < spec_val[0].size(); I++) {
	vector<int> match_index;
	vector<float> mass_temp_vec;
	vector<float> mass_error_temp_vec;
	match_index.resize(spec_vec.size());
	mass_temp_vec.resize(spec_vec.size());
	mass_error_temp_vec.resize(spec_vec.size());
	bool flag_match;
	int num_match;
	num_match = 0;
	flag_match = false;
	for(int spec_vec_ind = 0; spec_vec_ind < spec_vec.size(); spec_vec_ind++) {
	   bool flag_match_local = false;
	    match_index[spec_vec_ind] = -1;
	    mass_temp_vec[spec_vec_ind] = 0;
       	    for(int spec_ind = 0; spec_ind < spec_vec[spec_vec_ind]->getNumPeaks(); spec_ind++) {
	        if(abs(spec_val[spec_vec_ind][I] - spec_vec[spec_vec_ind]->getPeakMass(spec_ind) )< acc_thresh[charge_num[I]-1]) {
		    match_index[spec_vec_ind] = spec_ind;
		    mass_temp_vec[spec_vec_ind] = spec_vec[spec_vec_ind]->getPeakMass(spec_ind);
		    mass_error_temp_vec[spec_vec_ind] = mass_temp_vec[spec_vec_ind] - spec_val[spec_vec_ind][I];
		    flag_match = true;
		    flag_match_local = true;
		}
	    }
	    if(flag_match_local == true)
		num_match++;
        }
	stat[num_match]++;
	if(flag_match) {
       	    spec_id_mult_vec.push_back(match_index);
	    mass_error_mult_vec.push_back(mass_error_temp_vec);
	    mass_mult_vec.push_back(mass_temp_vec);
 	    spec_val_vec.push_back(spec_val[0][I]);
	    charge_num_vec.push_back(charge_num[I]); 
	    position_vec.push_back(position[I]);
	    length_vec.push_back(length[I]);
	    offset_vec.push_back(off[I]);
	    if(pep_set_flag == true) {
	        string pep_str;
	        for(int id = position[I]; id < position[I] + length[I]; id++) {
	            pep_str.push_back(pep_str_rep[id]);
	        }
	        pep_str_vec.push_back(pep_str);
	    }
	}
    }
}

void AnnotRes::peptide_spectrum_score_repeat_mult(vector<AnnotatedSpectrumCounter*> spec_vec, vector<PeptideAdd> pep_vec, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic) {
    if(do_cyclic == 1)
	peptide_spectrum_score_repeat_cyclic_mult(spec_vec, pep_vec, max_charge, acc_thresh, offset);
}

void AnnotRes::peptide_spectrum_score_mult(vector<AnnotatedSpectrumCounter*> spec_vec, vector<PeptideAdd> pep_vec, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type) {
    if(score_type == 0)
	peptide_spectrum_score_repeat_mult(spec_vec, pep_vec, max_charge, acc_thresh, offset, do_cyclic);
}


void AnnotRes::print(string file_name_string) {
    ofstream out_file;
    out_file.open(file_name_string.c_str(), fstream::in | fstream::out | fstream::app);
    out_file <<"Annotation : " << endl;
    for(int ind = 0; ind < spec_id_vec.size(); ind++) {
	    out_file << ind << " " << spec_id_vec[ind] << " " << spec_val_vec[ind] << " " << charge_num_vec[ind] << " " << mass_error_vec[ind] << " " << offset_vec[ind] << " " << position_vec[ind] << " " << length_vec[ind] << " " << pep_str_vec[ind] << endl;
    } 
    for(int ind = 0; ind<pep_string.size(); ind++) {
	out_file << ind << " " << pep_string[ind] << " : " << num_start[ind] << " " << num_end[ind] << endl;
    }
    out_file.close();
}

void AnnotRes::print_mult(string file_name_string) {
    ofstream out_file;
    out_file.open(file_name_string.c_str(), fstream::in | fstream::out | fstream::app);
    for(int ind = 0; ind < stat.size(); ind++ ) { 
	    out_file << "num " << ind << " : " << stat[ind] << endl;
    }
    for(int ind = 0; ind < spec_id_mult_vec.size(); ind++) {
	out_file << ind << " " << pep_str_vec[ind] << " "  << charge_num_vec[ind] << " "  << offset_vec[ind] << " " << position_vec[ind] << " " << length_vec[ind] << " ";
	for(int spec_ind = 0; spec_ind < mass_mult_vec[ind].size(); spec_ind++) {
	    out_file << mass_mult_vec[ind][spec_ind] << " ";
	}
	out_file << endl;
    }
    
}

void AnnotRes::peptide_spectrum_score_repeat(AnnotatedSpectrumCounter& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic) {
	if(do_cyclic == 1) {
		peptide_spectrum_score_repeat_cyclic(input_spectrum, seq, max_charge, acc_thresh, offset); }
	else {
		cout << "no support for linear yet" << endl;
	     }

}

void AnnotRes::peptide_spectrum_score(AnnotatedSpectrumCounter& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type) {
	if(score_type == 0) {
		peptide_spectrum_score_repeat(input_spectrum, seq, max_charge, acc_thresh, offset, do_cyclic); }
	else {
		cout << "no support for unique score yet" << endl;
	}

}

void AnnotRes::peptide_spectrum_score(AnnotatedSpectrumCounter& input_spectrum, PeptideAdd& pep, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type) {
	set_peptide(pep);
	peptide_spectrum_score(input_spectrum, pep.get_amino_acid_masses(), max_charge, acc_thresh, offset, do_cyclic, score_type);

}


score_t peptide_spectrum_score_repeat_cyclic(AnnotatedSpectrumAdd& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset){

    int n = seq.size();
    int I = 0;
    score_t final_score = 0;
    
    int spec_val_len = (n*(n-1)+1)*offset.val.size();
    float spec_val[max_charge][spec_val_len];
    float mat_val[n][n-1];
    float temp_val;
    float seq_rep[2*n];

    for(int ind = 0; ind<n; ind++) {
	seq_rep[ind] = seq[ind];
	seq_rep[ind+n] = seq[ind];
    }
  
    float parent_mass = 0.f;
    for(int i = 0; i < seq.size(); i++){
        parent_mass += seq[i];
     }

    /*int mass_vector_size = input_spectrum.getNumPeaks();
    float mass_vector[mass_vector_size];
    for(int ind = 0; ind<mass_vector_size; ind++)
	mass_vector[ind] = input_spectrum.getPeakMass(ind);*/

    for(int charge_number = 1; charge_number <= max_charge; charge_number++){
	I = 0;
        for(int pos_1 = 0; pos_1<n; pos_1++) {
	    temp_val = seq_rep[pos_1];
       	    mat_val[pos_1][0] = temp_val;
	    for(int offset_ind = 0; offset_ind < offset.val.size() ; offset_ind++) {
		    spec_val[charge_number-1][I] = (temp_val+charge_mass*charge_number+offset.val[offset_ind])/charge_number;
		    I++;
	    }
	}   
        for(int len_seq = 1; len_seq<n-1; len_seq++){
	    for(int pos_1 = 0; pos_1<n; pos_1++){
	        temp_val = mat_val[pos_1][len_seq-1] + seq_rep[pos_1+len_seq];
		mat_val[pos_1][len_seq] = temp_val;
	        for(int offset_ind = 0; offset_ind < offset.val.size() ; offset_ind++) {
			spec_val[charge_number-1][I] = (temp_val+charge_mass*charge_number+offset.val[offset_ind])/charge_number;
			I++;
		}
		// cout << spec_val[charge_number-1][I] << endl;
	    }
        }
	for(int offset_ind = 0; offset_ind < offset.val.size() ; offset_ind++) {
        	spec_val[charge_number-1][I] = (parent_mass+charge_mass*charge_number+offset.val[offset_ind])/(charge_number);
        	I++;
	}
	// for(int ind = 0; ind<input_spectrum.binned_spectrum.length; ind++)
	//	cout << input_spectrum.binned_spectrum.bin[charge_number-1][ind];
	final_score += intersect_threshold_binned(spec_val[charge_number-1], spec_val_len, input_spectrum.binned_spectrum, charge_number-1);
    }
return final_score;
}


score_t peptide_spectrum_score_repeat_linear(AnnotatedSpectrumAdd& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset){
	
        int n = seq.size();
      	score_t final_score = 0;
    	float H2O_mass = 18.01528;
      	float charge_mass = 1.007276;
      	float neutron_mass = 1.008664;
      	int spec_val_len = 2*n;
    	  float spec_val[max_charge][spec_val_len];
      	int I;
        float parent_mass = 0.f;
        for(int i = 0; i < seq.size(); i++){
            parent_mass += seq[i];
        }
        // parent_mass += H2O_mass;
        
        for(int charge_number= 1; charge_number <= max_charge; charge_number++){
            float v = 0.f;
      	    float w;
	          I = 0;
            for(int pos = 0 ; pos < n-1; pos++){
                v += seq[pos];
            		spec_val[charge_number-1][I] = (v+charge_mass*charge_number)/charge_number;
            		I++;
            		w = parent_mass + H2O_mass - v;
            		spec_val[charge_number-1][I] = (w+charge_mass*charge_number)/charge_number;
                I++;
            }
      	    spec_val[charge_number-1][I] = (parent_mass+max_charge*charge_number)/charge_number;
      	    I++;
	          spec_val[charge_number-1][I] = (parent_mass+ H2O_mass+max_charge*charge_number)/charge_number;
      	    I++;
	          final_score += intersect_threshold_binned(spec_val[charge_number-1], spec_val_len, input_spectrum.binned_spectrum, charge_number-1);	
        }
      	return final_score;
}


score_t peptide_spectrum_score_unique_linear(AnnotatedSpectrumAdd& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset){
        int n = seq.size();
    	score_t final_score = 0;
    	float H2O_mass = 18.01528;
    	float charge_mass = 1.007276;
    	float neutron_mass = 1.008664;
    	int spec_val_len = 2*n;
    	float spec_val[max_charge][spec_val_len];
    	int I;
        float parent_mass = 0.f;
        for(int i = 0; i < seq.size(); i++){
            parent_mass += seq[i];
        }
        // parent_mass += H2O_mass;
        
	int mass_vector_size = input_spectrum.getNumPeaks();
        float mass_vector[mass_vector_size];
        for(int ind = 0; ind<mass_vector_size; ind++)
	    mass_vector[ind] = input_spectrum.getPeakMass(ind);

        for(int charge_number= 1; charge_number <= max_charge; charge_number++){
            float v = 0.f;
	    float w;
	    I = 0;
            for(int pos = 0 ; pos < n-1; pos++){
                v += seq[pos];
		spec_val[charge_number-1][I] = (v+charge_mass*charge_number)/charge_number;
		I++;
		w = parent_mass + H2O_mass - v;
		spec_val[charge_number-1][I] = (w+charge_mass*charge_number)/charge_number;
                I++;
            }
	    spec_val[charge_number-1][I] = (parent_mass+max_charge*charge_number)/charge_number;
	    I++;
	    spec_val[charge_number-1][I] = (parent_mass+ H2O_mass+max_charge*charge_number)/charge_number;
	    I++;
	
	    final_score += intersect_threshold_unique(spec_val[charge_number-1], spec_val_len, mass_vector, mass_vector_size, acc_thresh[charge_number-1]);	
        }
    	return final_score;
}


score_t peptide_spectrum_score_unique_cyclic(AnnotatedSpectrumAdd& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset){
        
    float H2O_mass = 18.01528;
    float charge_mass = 1.007276;
    
    int n = seq.size();
    int I = 0;
    score_t final_score = 0;
    int spec_val_len = (n*(n-1)+1)*offset.val.size()*max_charge;
    float spec_val[spec_val_len];
    float thresh[spec_val_len];
    float mat_val[n][n-1];
    float temp_val;
    float seq_rep[2*n];

    for(int ind = 0; ind<n; ind++) {
	seq_rep[ind] = seq[ind];
	seq_rep[ind+n] = seq[ind];
    }
  
    float parent_mass = 0.f;
    for(int i = 0; i < seq.size(); i++){
        parent_mass += seq[i];
    }
    int mass_vector_size = input_spectrum.getNumPeaks();
    float mass_vector[mass_vector_size];
    for(int ind = 0; ind<mass_vector_size; ind++)
	mass_vector[ind] = input_spectrum.getPeakMass(ind);
    
    I = 0;

    for(int charge_number = 1; charge_number <= max_charge; charge_number++){
        for(int pos_1 = 0; pos_1<n; pos_1++) {
	    temp_val = seq_rep[pos_1];
       	    mat_val[pos_1][0] = temp_val;
	    for(int offset_ind = 0; offset_ind < offset.val.size(); offset_ind++) {
		    spec_val[I] = (temp_val+charge_mass*charge_number+offset.val[offset_ind])/charge_number;
		    thresh[I] = acc_thresh[charge_number - 1];
		    I++;
	    }
	}   
        for(int len_seq = 1; len_seq<n-1; len_seq++){
	    for(int pos_1 = 0; pos_1<n; pos_1++){
	        temp_val = mat_val[pos_1][len_seq-1] + seq_rep[pos_1+len_seq];
		mat_val[pos_1][len_seq] = temp_val;	
	        for(int offset_ind = 0; offset_ind < offset.val.size(); offset_ind++) {
			spec_val[I] = (temp_val+charge_mass*charge_number+offset.val[offset_ind])/charge_number;
			thresh[I] = acc_thresh[charge_number - 1];
			I++;
		}
	    }
        }
	for(int offset_ind = 0; offset_ind < offset.val.size(); offset_ind++) {
	        spec_val[I] = (parent_mass + charge_mass*charge_number+offset.val[offset_ind])/(charge_number);
		thresh[I] = acc_thresh[charge_number - 1];
        	I++;
	}
    }
final_score = intersect_threshold_unique_thresh(spec_val, spec_val_len, mass_vector, mass_vector_size, thresh);	
return final_score;
}

score_t peptide_spectrum_score_average_cyclic(AnnotatedSpectrumAdd& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset){
  score_t unique_score = peptide_spectrum_score_unique_cyclic(input_spectrum, seq, max_charge, acc_thresh, offset);    
  score_t repeat_score = peptide_spectrum_score_repeat_cyclic(input_spectrum, seq, max_charge, acc_thresh, offset);
  return unique_score + repeat_score;
}
        
score_t peptide_spectrum_score_average_linear(AnnotatedSpectrumAdd& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh, Offset offset){
  score_t unique_score = peptide_spectrum_score_unique_linear(input_spectrum, seq, max_charge, acc_thresh, offset);    
  score_t repeat_score = peptide_spectrum_score_repeat_linear(input_spectrum, seq, max_charge, acc_thresh, offset);
  return unique_score + repeat_score;
}

score_t peptide_spectrum_score_repeat(AnnotatedSpectrumAdd& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic){
    score_t final_score = 0;    
    if(do_cyclic == 1){
        vector<float> seq = peptide.seq;
	final_score = peptide_spectrum_score_repeat_cyclic(input_spectrum, seq, max_charge, acc_thresh, offset);
    }
    else if(do_cyclic == 0){
        vector<float> seq = peptide.seq;
	final_score = peptide_spectrum_score_repeat_linear(input_spectrum, seq, max_charge, acc_thresh, offset);
    } else {
	Non_Linear_Peptide nlp_lin, nlp_cyc;
	peptide.get_linear_nlp(nlp_lin);
	peptide.get_cyclic_nlp(nlp_cyc);
        int lin_score = peptide_spectrum_score_repeat_linear(input_spectrum, nlp_lin.seq, max_charge, acc_thresh, offset);
        int cyc_score = peptide_spectrum_score_repeat_cyclic(input_spectrum, nlp_cyc.seq, max_charge, acc_thresh, offset);
        final_score = lin_score + cyc_score;
    }
    return final_score;
}


score_t peptide_spectrum_score_average(AnnotatedSpectrumAdd& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic){
    score_t final_score = 0;    
    if(do_cyclic == 1){
        vector<float> seq = peptide.seq;
	final_score = peptide_spectrum_score_average_cyclic(input_spectrum, seq, max_charge, acc_thresh, offset);
    }
    else if(do_cyclic == 0){
        vector<float> seq = peptide.seq;
	final_score = peptide_spectrum_score_average_linear(input_spectrum, seq, max_charge, acc_thresh, offset);
    } else {
	}
    return final_score;
}

score_t peptide_spectrum_score_unique(AnnotatedSpectrumAdd& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic){
    score_t final_score = 0;
    if(do_cyclic == 1){
	vector<float> seq = peptide.seq;
        final_score = peptide_spectrum_score_unique_cyclic(input_spectrum, seq, max_charge, acc_thresh, offset);
    }
    else if(do_cyclic == 0){ 
	vector<float> seq = peptide.seq;
        final_score = peptide_spectrum_score_unique_linear(input_spectrum, seq, max_charge, acc_thresh, offset);
    } else{
	Non_Linear_Peptide nlp_lin, nlp_cyc;
	peptide.get_linear_nlp(nlp_lin);
	peptide.get_cyclic_nlp(nlp_cyc);
        int lin_score = peptide_spectrum_score_unique_linear(input_spectrum, nlp_lin.seq, max_charge, acc_thresh, offset);
        int cyc_score = peptide_spectrum_score_unique_cyclic(input_spectrum, nlp_cyc.seq, max_charge, acc_thresh, offset);
        final_score = lin_score + cyc_score;
	// cout << "siz lin : " << nlp_lin.seq.size() << " siz cyc : " << nlp_cyc.seq.size() << " spec siz : "  << input_spectrum.getNumPeaks() << endl;
    }
    return final_score;
}


score_t peptide_spectrum_score_intens(AnnotatedSpectrumAdd& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic){
    score_t final_score = 0;
    if(do_cyclic == 1){
	vector<float> seq = peptide.seq;
        final_score = peptide_spectrum_score_intens_cyclic(input_spectrum, seq, max_charge, acc_thresh, offset);
    }
    else if(do_cyclic == 0){ 
	final_score = 0;
    }
    return final_score;
}

void convert_int(Non_Linear_Peptide& peptide, Non_Linear_Peptide& peptide_int) {
  float scale_factor = 0.9995;
  peptide_int = peptide;
  peptide_int.seq.clear();
  for(int ind = 0; ind < peptide.seq.size(); ind++) {
	peptide_int.seq.push_back((int)peptide.seq[ind]/scale_factor);
  }
}


//void convert_int(AnnotatedSpectrumAdd& input_spectrum, AnnotatedSpectrumAdd& input_spectrum_int) {
//  input_spectrum_int = input_spectrum;
//}

score_t peptide_spectrum_score_int(AnnotatedSpectrumAdd& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type){
  // AnnotatedSpectrumAdd input_spectrum_int;
  // convert_int(input_spectrum, input_spectrum_int);
  Non_Linear_Peptide peptide_int;
  convert_int(peptide, peptide_int);
  return peptide_spectrum_score(input_spectrum, peptide_int, max_charge, acc_thresh, offset, do_cyclic, score_type);
}

score_t peptide_spectrum_score_native(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type) {
    score_t final_score = 0;
    score_t score_1 = peptide_spectrum_score(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type);
    PeptideAdd pep;
    pep.parseFromString(input_spectrum.config_list[0], peptide.pep.getPeptideStr());
    Non_Linear_Peptide np;
    np.seq = pep.get_amino_acid_masses();
    np.pep = pep;
    score_t score_2 = peptide_spectrum_score(*(input_spectrum.aux_spec_list[0]), np, max_charge, acc_thresh, offset, do_cyclic, score_type);
    return score_1 + score_2;
}


score_t peptide_spectrum_score(AnnotatedSpectrumAdd& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type){
    score_t final_score = 0;
    if(score_type == 1){
        final_score = peptide_spectrum_score_unique(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic);
    } else if (score_type == 0){
        final_score = peptide_spectrum_score_repeat(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic);
    } else if (score_type == 2) {
        final_score = peptide_spectrum_score_average(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic);
    } else if (score_type == 3) {
        final_score = peptide_spectrum_score_intens(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic);
    } else if (score_type == 6 || score_type == 7) {
        final_score = peptide_spectrum_score_int(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type - 6);
    }
    return final_score;
}

score_t peptide_spectrum_score(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type){
    score_t final_score = 0;
    if(score_type == 1){
        final_score = peptide_spectrum_score_unique(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic);
    } else if (score_type == 0){
        final_score = peptide_spectrum_score_repeat(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic);
    } else if (score_type == 2) {
        final_score = peptide_spectrum_score_average(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic);
    } else if (score_type == 3) {
        final_score = peptide_spectrum_score_intens(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic);
    } else if (score_type == 4 || score_type == 5) {
        final_score = peptide_spectrum_score_native(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type - 4);
    } else if (score_type == 6 || score_type == 7) {
        final_score = peptide_spectrum_score_int(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type - 6);
    }
    return final_score;
}

score_t peptide_spectrum_score(AnnotatedSpectrumCounter& input_spectrum, PeptideAdd& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type){
    score_t final_score = 0;
    Non_Linear_Peptide np;
    np.pep = peptide;
    np.seq = peptide.get_amino_acid_masses();
    final_score = peptide_spectrum_score(input_spectrum, np, max_charge, acc_thresh, offset, do_cyclic, score_type);
    return final_score;
}
 
score_t peptide_spectrum_score(AnnotatedSpectrumAdd& input_spectrum, PeptideAdd& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type){
    score_t final_score = 0;
    Non_Linear_Peptide np;
    np.pep = peptide;
    np.seq = peptide.get_amino_acid_masses();
    final_score = peptide_spectrum_score(input_spectrum, np, max_charge, acc_thresh, offset, do_cyclic, score_type);
    return final_score;
}

float peptide_spectrum_score_pvalue_mod_branch(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type, int num_rand) {
    float scale_factor = 0.9995;
    peptide.score = peptide_spectrum_score_mod(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type).score;
    // cout << main_score << endl;
    int num_high = 0;
    for(int ind = 0; ind < num_rand; ind++) {
	float mass_real_lin = peptide.get_mass_real_lin();
	float mass_real_cyc = peptide.get_mass_real_cyc();
	int mass_int_lin = int(mass_real_lin*scale_factor);
	int mass_int_cyc = int(mass_real_cyc*scale_factor);
        Non_Linear_Peptide nlp_lin, nlp_cyc, nlp;
	nlp_lin.generate_random_nlp(peptide.get_length_lin(), mass_int_lin);
	nlp_cyc.generate_random_nlp(peptide.get_length_cyc(), mass_int_cyc);
	nlp_lin.scale_by(1/scale_factor);
	nlp_cyc.scale_by(1/scale_factor);
	nlp.seq_lin = nlp_lin.seq;
	nlp.seq_cyc = nlp_cyc.seq;
	nlp.seq_lin_first = peptide.seq_lin_first;
	nlp.structure = peptide.structure;
    	nlp.score = peptide_spectrum_score_mod(input_spectrum, nlp, max_charge, acc_thresh, offset, do_cyclic, score_type).score;
	if(nlp.score >= peptide.score)
		num_high++;
	// nlp.print();
	// cout << score << endl;
    }
    return (float(num_high)/num_rand);
}


float peptide_spectrum_score_pvalue_branch(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type, int num_rand) {
    float scale_factor = 0.9995;
    peptide.score = peptide_spectrum_score(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type);
    // cout << main_score << endl;
    int num_high = 0;
    for(int ind = 0; ind < num_rand; ind++) {
	float mass_real_lin = peptide.get_mass_real_lin();
	float mass_real_cyc = peptide.get_mass_real_cyc();
	int mass_int_lin = int(mass_real_lin*scale_factor);
	int mass_int_cyc = int(mass_real_cyc*scale_factor);
        Non_Linear_Peptide nlp_lin, nlp_cyc, nlp;
	nlp_lin.generate_random_nlp(peptide.get_length_lin(), mass_int_lin);
	nlp_cyc.generate_random_nlp(peptide.get_length_cyc(), mass_int_cyc);
	nlp_lin.scale_by(1/scale_factor);
	nlp_cyc.scale_by(1/scale_factor);
	nlp.seq_lin = nlp_lin.seq;
	nlp.seq_cyc = nlp_cyc.seq;
	nlp.seq_lin_first = peptide.seq_lin_first;
	nlp.structure = peptide.structure;
    	nlp.score = peptide_spectrum_score(input_spectrum, nlp, max_charge, acc_thresh, offset, do_cyclic, score_type);
	if(nlp.score >= peptide.score)
		num_high++;
	// nlp.print();
	// cout << score << endl;
    }
    return (float(num_high)/num_rand);
}


float peptide_spectrum_score_pvalue_mod_linear_cyclic(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type, int num_rand) {
    float scale_factor = 0.9995;
    peptide.score = peptide_spectrum_score_mod(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type).score;
    // cout << peptide.score << endl;
    int num_high = 0;
    for(int ind = 0; ind < num_rand; ind++) {
	float mass_real = peptide.get_mass_real();
	int mass_int = int(mass_real*scale_factor);
        Non_Linear_Peptide nlp;
	// cout << mass_int << endl;
	// cout << peptide.get_length() << endl;
	nlp.generate_random_nlp(peptide.get_length(), mass_int);
	nlp.scale_by(1/scale_factor);
	nlp.structure = peptide.structure;
    	nlp.score = peptide_spectrum_score_mod(input_spectrum, nlp, max_charge, acc_thresh, offset, do_cyclic, score_type).score;
	if(nlp.score >= peptide.score)
		num_high++;
	// nlp.print();
	// cout << nlp.score << endl;
	// cout << "nlp score : " << nlp.score << endl;
    }
    return (float(num_high)/num_rand);
}


float peptide_spectrum_score_pvalue_linear_cyclic(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type, int num_rand) {
    float scale_factor = 0.9995;
    peptide.score = peptide_spectrum_score(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type);
    // cout << peptide.score << endl;
    int num_high = 0;
    for(int ind = 0; ind < num_rand; ind++) {
	float mass_real = peptide.get_mass_real();
	int mass_int = int(mass_real*scale_factor);
        Non_Linear_Peptide nlp;
	// cout << mass_int << endl;
	// cout << peptide.get_length() << endl;
	nlp.generate_random_nlp(peptide.get_length(), mass_int);
	nlp.scale_by(1/scale_factor);
	nlp.structure = peptide.structure;
    	nlp.score = peptide_spectrum_score(input_spectrum, nlp, max_charge, acc_thresh, offset, do_cyclic, score_type);
	if(nlp.score >= peptide.score)
		num_high++;
	// nlp.print();
    }
    return (float(num_high)/num_rand);
}


score_pos peptide_spectrum_score_mod_branch(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type) {
	score_pos res;
	res.diff = input_spectrum.get_org_pm() - peptide.get_mass();
	res.score = 0;
	res.pos = -1;
	res.lin_mod = true;
	Non_Linear_Peptide peptide_mod;
	for(int ind = 0; ind < peptide.seq_lin.size(); ind++) {
		peptide_mod = peptide;
		peptide_mod.seq_lin[ind] += res.diff;
		int score = peptide_spectrum_score(input_spectrum, peptide_mod, max_charge, acc_thresh, offset, do_cyclic, score_type);
		if(score>res.score) {
			res.score = score;
			res.pos = ind;
			res.lin_mod = true;
		}
	}

	for(int ind = 0; ind < peptide.seq_cyc.size(); ind++) {
		peptide_mod = peptide;
		peptide_mod.seq_cyc[ind] += res.diff;
		int score = peptide_spectrum_score(input_spectrum, peptide_mod, max_charge, acc_thresh, offset, do_cyclic, score_type);
		if(score>res.score) {
			res.score = score;
			res.pos = ind;
			res.lin_mod = false;
		}
	}

	return res;
}


score_pos peptide_spectrum_score_mod_linear_cyclic(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type) {
	score_pos res;
	res.diff = input_spectrum.get_org_pm() - peptide.get_mass();
	res.score = 0;
	res.pos = -1;
	Non_Linear_Peptide peptide_mod;
	for(int ind = 0; ind < peptide.seq.size(); ind++) {
		peptide_mod = peptide;
		peptide_mod.seq[ind] += res.diff;
		int score = peptide_spectrum_score(input_spectrum, peptide_mod, max_charge, acc_thresh, offset, do_cyclic, score_type);
		if(score>res.score) {
			res.score = score;
			res.pos = ind;
		}
	}
	return res;
}


score_pos peptide_spectrum_score_mod(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type){
    if(do_cyclic == 2)
	return peptide_spectrum_score_mod_branch(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type);
    else 
	return peptide_spectrum_score_mod_linear_cyclic(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type);
}

float peptide_spectrum_score_pvalue_mod(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type, int num_rand) {
    if(do_cyclic == 2)
	return peptide_spectrum_score_pvalue_mod_branch(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type, num_rand);
    else
	return peptide_spectrum_score_pvalue_mod_linear_cyclic(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type, num_rand);
}


float peptide_spectrum_score_pvalue(AnnotatedSpectrumCounter& input_spectrum, Non_Linear_Peptide& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type, int num_rand) {
    if(do_cyclic == 2)
	return peptide_spectrum_score_pvalue_branch(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type, num_rand);
    else
	return peptide_spectrum_score_pvalue_linear_cyclic(input_spectrum, peptide, max_charge, acc_thresh, offset, do_cyclic, score_type, num_rand);
}

float peptide_spectrum_score_pvalue(AnnotatedSpectrumCounter& input_spectrum, PeptideAdd& peptide, int max_charge, vector<float> acc_thresh, Offset offset, int do_cyclic, int score_type, int num_rand) {
return 0.0;
}

/*int peptide_spectrum_score_repeat_cyclic_old(Spectrum& input_spectrum, vector<float> seq, int max_charge, vector<float> acc_thresh){
        
    float H2O_mass = 18.01528;
    float charge_mass = 1.007276;
    
    int n = seq.size();
    int I = 0;
    int final_score = 0;
    int spec_val_len = (n*(n-1)+1);
    float spec_val[max_charge][spec_val_len];
    float mat_val[n][n-1];
    float temp_val;
    float seq_rep[2*n];

    for(int ind = 0; ind<n; ind++) {
	seq_rep[ind] = seq[ind];
	seq_rep[ind+n] = seq[ind];
    }
  
    float parent_mass = 0.f;
    int mass_vector_size = input_spectrum.getNumPeaks();
    float mass_vector[mass_vector_size];
    for(int ind = 0; ind<mass_vector_size; ind++)
	mass_vector[ind] = input_spectrum.getPeakMass(ind);

    for(int charge_number = 1; charge_number <= max_charge; charge_number++){
	I = 0;
        for(int pos_1 = 0; pos_1<n; pos_1++) {
	    temp_val = seq_rep[pos_1];
       	    mat_val[pos_1][0] = temp_val;
	    spec_val[charge_number-1][I] = (temp_val+charge_mass*charge_number)/charge_number;
	    I++;
	}   
        for(int len_seq = 1; len_seq<n-1; len_seq++){
	    for(int pos_1 = 0; pos_1<n; pos_1++){
	        temp_val = mat_val[pos_1][len_seq-1] + seq_rep[pos_1+len_seq];
		mat_val[pos_1][len_seq] = temp_val;
		spec_val[charge_number-1][I] = (temp_val+charge_mass*charge_number)/charge_number;
		I++;
	    }
        }
        spec_val[charge_number-1][I] = (parent_mass + charge_mass*charge_number)/(charge_number);
        I++;
	final_score += intersect_threshold(spec_val[charge_number-1], spec_val_len, mass_vector, mass_vector_size, acc_thresh[charge_number-1]);	
    }
return final_score;
}*/
