#include "AnnotatedSpectrum_add.h"
#include "Isotopes.h"
#include "QuickClustering.h"
#include "FileManagement.h"
#include "auxfun.h"
#include "../../util.h"

void AnnotatedSpectrumAdd::merge(AnnotatedSpectrumAdd spec_1, AnnotatedSpectrumAdd spec_2) {
	vector_peak_flag = true;
	peaks_vector.clear();
	for(int ind = 0; ind < spec_1.getNumPeaks(); ind++)
		peaks_vector.push_back(spec_1.getPeak(ind));
	for(int ind = 0; ind < spec_2.getNumPeaks(); ind++)
		peaks_vector.push_back(spec_2.getPeak(ind));
	sort(peaks_vector.begin(), peaks_vector.end(), ComparePeakByMass);
}

int AnnotatedSpectrumAdd::getNumPeaks() const {
	if(vector_peak_flag)
		return peaks_vector.size();
	else
		return Spectrum::getNumPeaks();
}

Peak const& AnnotatedSpectrumAdd::getPeak(int index) const {
	if(vector_peak_flag)
		return peaks_vector[index];
	else
		return Spectrum::getPeak(index);
}

float AnnotatedSpectrumAdd::getPeakMass(int index) const {
	if(vector_peak_flag)
		return peaks_vector[index].mass;
	else
		return Spectrum::getPeakMass(index);
}

float AnnotatedSpectrumAdd::getPeakIntensity(int index) const {
	if(vector_peak_flag)
		return peaks_vector[index].intensity;
	else
		return Spectrum::getPeakIntensity(index);
}

void AnnotatedSpectrumAdd::generate_random_spectrum(float mass, int len) {
	vector_peak_flag = true;
	peaks_vector.resize(len);
	for(int ind = 0; ind < len; ind++) {
		peaks_vector[ind].mass = myRandom()*mass;
		peaks_vector[ind].intensity = 1;
	}
	sort(peaks_vector.begin(), peaks_vector.end(), ComparePeakByMass);
}


void AnnotatedSpectrumAdd::print() {
	cout << "mz : " << get_m_over_z() << " charge : " << getCharge() << endl;
	cout << "spectrum size : " << getNumPeaks() << endl;
	for(int ind = 0; ind < getNumPeaks(); ind++)
		cout << getPeakMass(ind) << " " << getPeakIntensity(ind) << endl;
}

void AnnotatedSpectrumAdd::print(string out_file) {
	ofstream out;
	out.open(out_file.c_str());
	out << "spectrum size : " << getNumPeaks() << endl;
	for(int ind = 0; ind < getNumPeaks(); ind++)
		out << getPeakMass(ind) << " " << getPeakIntensity(ind) << endl;
	out.close();
}

AnnotatedSpectrumAdd::AnnotatedSpectrumAdd() {
    vector_peak_flag = false;
    peaks_vector.clear();
    spectrum_length = -1; 
    filter_step = 50.0;
    filter_type = 0; 
    filter_peak_num = 5;
    scale_factor = 0.9995;
}

void AnnotatedSpectrumAdd::de_isotope(vector<float> de_isotope_vec, vector<float> de_isotope_thresh) {
  // cout << "charge : " << getCharge() << endl;
  // cout << de_isotope_vec[0] << " " << de_isotope_thresh[0] << endl;
  vector<bool> mark_for_removal;
  mark_for_removal.resize(getNumPeaks());
  for(int ind = 0; ind<getNumPeaks(); ind++) {
    mark_for_removal[ind] = false;
  }
  for(int ind = 0; ind<getNumPeaks(); ind++) {
    int next_ind = ind+1;
    while(next_ind<getNumPeaks() && getPeakMass(next_ind) < (getPeakMass(ind)+de_isotope_vec[0]+de_isotope_thresh[0])) {
      for(int ch = 1; ch<=getCharge(); ch++) {
        if(abs(getPeakMass(next_ind) - getPeakMass(ind) - de_isotope_vec[ch-1]) < de_isotope_thresh[ch-1]) {
          mark_for_removal[next_ind] = true;
        }
      }
      next_ind++;
    }
  }
  
  vector<Peak> final_peaks;
  for(int ind = 0; ind < getNumPeaks(); ind++) {
    if(mark_for_removal[ind] == false) {
      final_peaks.push_back(getPeak(ind));
    }
  }
  numPeaks_ = final_peaks.size();
  for(int ind = 0; ind < numPeaks_; ind++)
    peaks_[ind] = final_peaks[ind];
}

void AnnotatedSpectrumAdd::operator= (const AnnotatedSpectrumAdd& other){
  cout << "here! " << endl;
  assert(0);  
}

void AnnotatedSpectrumAdd::initialize_integer_spectrum(){
  int int_mass;
  integer_spectrum.clear();
  for(int ind = 0; ind<getNumPeaks(); ind++) {
    int_mass = round((getPeakMass(ind)-MASS_PROTON)*scale_factor);
    integer_spectrum.push_back(int_mass);
  }
}


bool exist_within_threshold_tmp(Spectrum& S, float mass, int first, int last, float threshold) {
if(first == last)
	return false;
int middle_element = (first+last)/2;
if(abs(S.getPeakMass(middle_element) - mass) < threshold)
	return true;
if(S.getPeakMass(middle_element) < mass) {
	return exist_within_threshold_tmp(S, mass, middle_element+1, last, threshold);}
else {
	return exist_within_threshold_tmp(S, mass, first, middle_element, threshold);}
}


bool exist_within_threshold(Spectrum& S, float mass, float threshold) {
	return exist_within_threshold_tmp(S, mass, 0, S.getNumPeaks(), threshold);
}


bool ComparePeakByMass(Peak peak_1, Peak peak_2) {
	return (peak_1.mass < peak_2.mass);
}

bool ComparePeakByIntensity(Peak peak_1, Peak peak_2) {
	return (peak_1.intensity > peak_2.intensity);
}

void AnnotatedSpectrumAdd::initialize_binned_spectrum(float bw, vector<float> threshold_vector) {
	float mass;
	binned_spectrum.thresh_vec = threshold_vector;
	binned_spectrum.max_thresh = *max_element(binned_spectrum.thresh_vec.begin(), binned_spectrum.thresh_vec.end());
	binned_spectrum.bin_width = bw;
    	binned_spectrum.offset = get_min_peak_mass() - binned_spectrum.max_thresh;
    	binned_spectrum.length = (get_max_peak_mass() - get_min_peak_mass() + 2*binned_spectrum.max_thresh)/binned_spectrum.bin_width;
	binned_spectrum.bin.resize(binned_spectrum.thresh_vec.size());
	for(int thresh_index = 0; thresh_index<binned_spectrum.thresh_vec.size();thresh_index++) {
		binned_spectrum.bin[thresh_index].resize(binned_spectrum.length);
	    	for(int ind = 0; ind<binned_spectrum.length; ind++) {
      			mass = binned_spectrum.offset + ind*binned_spectrum.bin_width;
 			binned_spectrum.bin[thresh_index][ind] = exist_within_threshold(*this, mass, binned_spectrum.thresh_vec[thresh_index]);
	      	}
	}
}


void AnnotatedSpectrumAdd::do_spectral_filter() {

	vector<Peak> temp_peak_list(peaks_, peaks_+numPeaks_);
	vector<Peak> window_peak_list;
	vector<Peak> final_peak_list;
	final_peak_list.clear();
	// cout << "tmp peak list size: " << temp_peak_list.size() << endl;
	sort(temp_peak_list.begin(), temp_peak_list.end(), ComparePeakByMass);
	// cout << temp_peak_list[0].mass << " " << temp_peak_list[1].mass << endl;
	int window_num = (get_max_peak_mass() - get_min_peak_mass())/filter_step;
	float adjusted_filter_step = (get_max_peak_mass() - get_min_peak_mass())/window_num;
	int peak_ind = 0;
	float window_stop = get_min_peak_mass();
  // cout << get_max_peak_mass() << " " << get_min_peak_mass() << " " << window_num << " " << filter_step << endl;
	// cout << numPeaks_ << " " << spectrum_length << " " << filter_peak_num << " " << filter_step << " " << window_num << " " << get_max_peak_mass() << " " << adjusted_filter_step << " " << temp_peak_list.size() << " "  << endl;
	for(int window_index = 0; window_index < window_num; window_index++) {
		window_stop += adjusted_filter_step;
		window_peak_list.clear();
		while(peak_ind<numPeaks_ && temp_peak_list[peak_ind].mass<window_stop) {
			window_peak_list.push_back(temp_peak_list[peak_ind]);
			peak_ind++;
		}
		sort(window_peak_list.begin(), window_peak_list.end(), ComparePeakByIntensity);
		for(int ind = 0; (ind < filter_peak_num) && (ind < window_peak_list.size()); ind++)
			final_peak_list.push_back(window_peak_list[ind]);
	}
	
	sort(final_peak_list.begin(), final_peak_list.end(), ComparePeakByIntensity);
	numPeaks_ = final_peak_list.size();
	// cout << "np" << numPeaks_ << endl;
	if(!(spectrum_length == -1) && spectrum_length < final_peak_list.size())
		numPeaks_ = spectrum_length;
	final_peak_list.resize(numPeaks_);
	// cout << "np" << numPeaks_ << endl;
	sort(final_peak_list.begin(), final_peak_list.end(), ComparePeakByMass);
	for(int ind = 0; ind < numPeaks_; ind++)
		peaks_[ind] = final_peak_list[ind];
}


bool AnnotatedSpectrumAdd::readSpectrum(const SpectraAggregator& sa,	
							const SingleSpectrumHeader* header, 
							bool indFilterSpectrum)
{
  // cout << "Here" << endl;
	 config_ = sa.getConfig();

	 const int numPeaksRead = readPeaksToLocalAllocation(sa, header);
	 // if (numPeaksRead<5)
	 //	 return false;

	 copyHeaderInformation();
   
	 // cout << filter_type << endl;
	 bool doFilterSpectrum = indFilterSpectrum && (filter_type == 0);

	 initializePeakList(config_, doFilterSpectrum);

	 initializeSpectrum(indFilterSpectrum);
   
	 if(indFilterSpectrum && (filter_type == 1))
		do_spectral_filter();
	 

   if (! sanityCheck())
		 return false;


	 initializeSpectrum(indFilterSpectrum);


	 return (numPeaks_>0);
  // return 0;
}


void AnnotatedSpectrumAdd::initializeSpectrum(bool indFilterSpectrum)
{

	if(indFilterSpectrum) {

	this->computeLogIntensities(logIntensities_); 
	this->createIndexArray(indexArray_);
	this->calculatePeakRanks(ranks_);
	this->calculateLogLocalRanks(config_->get_local_window_size(), logLocalRanks_);
	this->calculateIsotopicLevels(config_->getTolerance(), isotopicLevels_);
	this->selectStrongPeakIndexes(config_->get_number_of_strong_peaks_per_local_window(),
								  logLocalRanks_, isotopicLevels_, strongPeakIndexes_);
	this->calculateLogRandomProbabilities(logIntensities_, logRandomProbabilities_);

	if (numPeaks_ >0)
	{
		minimalPeakMass_ = peaks_[0].mass -1; // margin 1 Dalton
		maximalPeakMass_ = peaks_[numPeaks_-1].mass +1;
	
		if (charge_>=2)
		{
			maximalPeakMassToConsider_ =  originalPmWith19_ > maximalPeakMass_ ? 
						originalPmWith19_ : maximalPeakMass_;
		}
		else
			maximalPeakMassToConsider_ = maximalPeakMass_;
	}
}
	maximalPeakMassToConsider_ += 10.0; // margin of error

	sizeIndex_ = config_->calc_size_idx(charge_, originalPmWith19_);
}


