package org.pdb.beans;

/**
 *  Data items in the ATOM_SITE category record details about
 *  the atom sites in a macromolecular crystal structure, such as
 *  the positional coordinates, atomic displacement parameters,
 *  magnetic moments and directions.
 *  
 *  The data items for describing anisotropic atomic
 *  displacement factors are only used if the corresponding items
 *  are not given in the ATOM_SITE_ANISOTROP category.
 *  
 * <pre>
 *  Example 1 - based on PDB entry 5HVP and laboratory records for the
 *  structure corresponding to PDB entry 5HVP.
 *  <PDBx:atom_siteCategory>
 *  <PDBx:atom_site id="1">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>N</PDBx:type_symbol>
 *  <PDBx:label_atom_id>N</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.369</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>30.691</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>11.795</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.93</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="2">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CA</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.970</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>31.965</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>12.332</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.75</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="3">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.569</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.010</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>13.808</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.83</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="4">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>O</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>24.735</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>31.190</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>14.167</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.53</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="5">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CB</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.379</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.146</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>11.540</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.66</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="6">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.584</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.034</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>10.030</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>18.86</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="7">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>23.933</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.309</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>11.872</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.12</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="8">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>N</PDBx:type_symbol>
 *  <PDBx:label_atom_id>N</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>26.095</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.930</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>14.590</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>18.97</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="9">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CA</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.734</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.995</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.032</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>19.80</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="10">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>24.695</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>34.106</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.113</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.92</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="11">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>O</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>24.869</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>35.118</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>15.421</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>21.84</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="12">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CB</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>26.911</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.346</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.018</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.51</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="13">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>OG1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id>3</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>27.946</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.921</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.183</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.50</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.29</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="14">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>OG1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id>4</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>27.769</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.142</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.103</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.50</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.59</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="15">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id>3</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>27.418</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.181</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.878</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.50</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.47</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="16">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id>4</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>26.489</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.778</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>18.426</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.50</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.00</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="17">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>N</PDBx:type_symbol>
 *  <PDBx:label_atom_id>N</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>23.664</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.855</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.884</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>22.08</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="18">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CA</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>22.623</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>34.850</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.093</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>23.44</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="19">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>22.657</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>35.113</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>18.610</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>25.77</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="20">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>O</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>23.123</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>34.250</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>19.406</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>26.28</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="21">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CB</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>21.236</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>34.463</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.492</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>22.67</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="22">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>20.478</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.469</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.371</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>22.14</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="23">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>21.357</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.986</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>15.016</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>21.75</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="101">
 *  <PDBx:group_PDB>HETATM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>APS</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>C</PDBx:label_asym_id>
 *  <PDBx:label_seq_id xsi:nil="true" />
 *  <PDBx:label_alt_id>1</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>4.171</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>29.012</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>7.116</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.58</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.27</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>1</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>300</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="102">
 *  <PDBx:group_PDB>HETATM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>APS</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>C</PDBx:label_asym_id>
 *  <PDBx:label_seq_id xsi:nil="true" />
 *  <PDBx:label_alt_id>1</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>4.949</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>27.758</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>6.793</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.58</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>16.95</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>1</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>300</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="103">
 *  <PDBx:group_PDB>HETATM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>O3</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>APS</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>C</PDBx:label_asym_id>
 *  <PDBx:label_seq_id xsi:nil="true" />
 *  <PDBx:label_alt_id>1</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>4.800</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>26.678</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>7.393</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.58</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>16.85</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>1</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>300</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="104">
 *  <PDBx:group_PDB>HETATM</PDBx:group_PDB>
 *  <PDBx:type_symbol>N</PDBx:type_symbol>
 *  <PDBx:label_atom_id>N4</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>APS</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>C</PDBx:label_asym_id>
 *  <PDBx:label_seq_id xsi:nil="true" />
 *  <PDBx:label_alt_id>1</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>5.930</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>27.841</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>5.869</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.58</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>16.43</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>1</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>300</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  </PDBx:atom_siteCategory>
 * </pre>
 */
public class Atom_site 
{
		protected java.lang.Float B_equiv_geom_mean;


	/**
	 *  Equivalent isotropic atomic displacement parameter, B~eq~,
	 *  in angstroms squared, calculated as the geometric mean of
	 *  the anisotropic atomic displacement parameters.
	 *  
	 *  B~eq~ = (B~i~ B~j~ B~k~)^1/3^
	 *  
	 *  B~n~ = the principal components of the orthogonalized B^ij^
	 *  
	 *  The IUCr Commission on Nomenclature recommends against the use
	 *  of B for reporting atomic displacement parameters. U, being
	 *  directly proportional to B, is preferred.
	 */
	public java.lang.Float getB_equiv_geom_mean() 
	{
		return B_equiv_geom_mean;
	}
 
	public void setB_equiv_geom_mean(java.lang.Float B_equiv_geom_mean)
	{
		this.B_equiv_geom_mean = B_equiv_geom_mean;
	}

 	protected java.lang.Float B_equiv_geom_mean_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute B_equiv_geom_mean in category atom_site.
	 */
	public java.lang.Float getB_equiv_geom_mean_esd() 
	{
		return B_equiv_geom_mean_esd;
	}
 
	public void setB_equiv_geom_mean_esd(java.lang.Float B_equiv_geom_mean_esd)
	{
		this.B_equiv_geom_mean_esd = B_equiv_geom_mean_esd;
	}

 	protected java.lang.Float B_iso_or_equiv;


	/**
	 *  Isotropic atomic displacement parameter, or equivalent isotropic
	 *  atomic displacement parameter, B~eq~, calculated from the
	 *  anisotropic displacement parameters.
	 *  
	 *  B~eq~ = (1/3) sum~i~[sum~j~(B^ij^ A~i~ A~j~ a*~i~ a*~j~)]
	 *  
	 *  A     = the real space cell lengths
	 *  a*    = the reciprocal space cell lengths
	 *  B^ij^ = 8 pi^2^ U^ij^
	 *  
	 *  Ref: Fischer, R. X. & Tillmanns, E. (1988). Acta Cryst. C44,
	 *  775-776.
	 *  
	 *  The IUCr Commission on Nomenclature recommends against the use
	 *  of B for reporting atomic displacement parameters. U, being
	 *  directly proportional to B, is preferred.
	 */
	public java.lang.Float getB_iso_or_equiv() 
	{
		return B_iso_or_equiv;
	}
 
	public void setB_iso_or_equiv(java.lang.Float B_iso_or_equiv)
	{
		this.B_iso_or_equiv = B_iso_or_equiv;
	}

 	protected java.lang.Float B_iso_or_equiv_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute B_iso_or_equiv in category atom_site.
	 */
	public java.lang.Float getB_iso_or_equiv_esd() 
	{
		return B_iso_or_equiv_esd;
	}
 
	public void setB_iso_or_equiv_esd(java.lang.Float B_iso_or_equiv_esd)
	{
		this.B_iso_or_equiv_esd = B_iso_or_equiv_esd;
	}

 	protected java.lang.Float Cartn_x;


	/**
	 *  The x atom-site coordinate in angstroms specified according to
	 *  a set of orthogonal Cartesian axes related to the cell axes as
	 *  specified by the description given in
	 *  attribute Cartn_transform_axes in category atom_sites.
	 */
	public java.lang.Float getCartn_x() 
	{
		return Cartn_x;
	}
 
	public void setCartn_x(java.lang.Float Cartn_x)
	{
		this.Cartn_x = Cartn_x;
	}

 	protected java.lang.Float Cartn_x_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute Cartn_x in category atom_site.
	 */
	public java.lang.Float getCartn_x_esd() 
	{
		return Cartn_x_esd;
	}
 
	public void setCartn_x_esd(java.lang.Float Cartn_x_esd)
	{
		this.Cartn_x_esd = Cartn_x_esd;
	}

 	protected java.lang.Float Cartn_y;


	/**
	 *  The y atom-site coordinate in angstroms specified according to
	 *  a set of orthogonal Cartesian axes related to the cell axes as
	 *  specified by the description given in
	 *  attribute Cartn_transform_axes in category atom_sites.
	 */
	public java.lang.Float getCartn_y() 
	{
		return Cartn_y;
	}
 
	public void setCartn_y(java.lang.Float Cartn_y)
	{
		this.Cartn_y = Cartn_y;
	}

 	protected java.lang.Float Cartn_y_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute Cartn_y in category atom_site.
	 */
	public java.lang.Float getCartn_y_esd() 
	{
		return Cartn_y_esd;
	}
 
	public void setCartn_y_esd(java.lang.Float Cartn_y_esd)
	{
		this.Cartn_y_esd = Cartn_y_esd;
	}

 	protected java.lang.Float Cartn_z;


	/**
	 *  The z atom-site coordinate in angstroms specified according to
	 *  a set of orthogonal Cartesian axes related to the cell axes as
	 *  specified by the description given in
	 *  attribute Cartn_transform_axes in category atom_sites.
	 */
	public java.lang.Float getCartn_z() 
	{
		return Cartn_z;
	}
 
	public void setCartn_z(java.lang.Float Cartn_z)
	{
		this.Cartn_z = Cartn_z;
	}

 	protected java.lang.Float Cartn_z_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute Cartn_z in category atom_site.
	 */
	public java.lang.Float getCartn_z_esd() 
	{
		return Cartn_z_esd;
	}
 
	public void setCartn_z_esd(java.lang.Float Cartn_z_esd)
	{
		this.Cartn_z_esd = Cartn_z_esd;
	}

 	protected java.lang.Float U_equiv_geom_mean;


	/**
	 *  Equivalent isotropic atomic displacement parameter, U~eq~,
	 *  in angstroms squared, calculated as the geometric mean of
	 *  the anisotropic atomic displacement parameters.
	 *  
	 *  U~eq~ = (U~i~ U~j~ U~k~)^1/3^
	 *  
	 *  U~n~ = the principal components of the orthogonalized U^ij^
	 */
	public java.lang.Float getU_equiv_geom_mean() 
	{
		return U_equiv_geom_mean;
	}
 
	public void setU_equiv_geom_mean(java.lang.Float U_equiv_geom_mean)
	{
		this.U_equiv_geom_mean = U_equiv_geom_mean;
	}

 	protected java.lang.Float U_equiv_geom_mean_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute U_equiv_geom_mean in category atom_site.
	 */
	public java.lang.Float getU_equiv_geom_mean_esd() 
	{
		return U_equiv_geom_mean_esd;
	}
 
	public void setU_equiv_geom_mean_esd(java.lang.Float U_equiv_geom_mean_esd)
	{
		this.U_equiv_geom_mean_esd = U_equiv_geom_mean_esd;
	}

 	protected java.lang.Float U_iso_or_equiv;


	/**
	 *  Isotropic atomic displacement parameter, or equivalent isotropic
	 *  atomic  displacement parameter, U~eq~, calculated from
	 *  anisotropic atomic displacement  parameters.
	 *  
	 *  U~eq~ = (1/3) sum~i~[sum~j~(U^ij^ A~i~ A~j~ a*~i~ a*~j~)]
	 *  
	 *  A  = the real space cell lengths
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  Ref: Fischer, R. X. & Tillmanns, E. (1988). Acta Cryst. C44,
	 *  775-776.
	 */
	public java.lang.Float getU_iso_or_equiv() 
	{
		return U_iso_or_equiv;
	}
 
	public void setU_iso_or_equiv(java.lang.Float U_iso_or_equiv)
	{
		this.U_iso_or_equiv = U_iso_or_equiv;
	}

 	protected java.lang.Float U_iso_or_equiv_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute U_iso_or_equiv in category atom_site.
	 */
	public java.lang.Float getU_iso_or_equiv_esd() 
	{
		return U_iso_or_equiv_esd;
	}
 
	public void setU_iso_or_equiv_esd(java.lang.Float U_iso_or_equiv_esd)
	{
		this.U_iso_or_equiv_esd = U_iso_or_equiv_esd;
	}

 	protected java.lang.String Wyckoff_symbol;


	/**
	 *  The Wyckoff symbol (letter) as listed in the space-group tables
	 *  of International Tables for Crystallography, Vol. A (2002).
	 */
	public java.lang.String getWyckoff_symbol() 
	{
		return Wyckoff_symbol;
	}
 
	public void setWyckoff_symbol(java.lang.String Wyckoff_symbol)
	{
		this.Wyckoff_symbol = Wyckoff_symbol;
	}

 	protected java.lang.String adp_type;


	/**
	 *  A standard code used to describe the type of atomic displacement
	 *  parameters used for the site.
	 */
	public java.lang.String getAdp_type() 
	{
		return adp_type;
	}
 
	public void setAdp_type(java.lang.String adp_type)
	{
		this.adp_type = adp_type;
	}

 	protected java.lang.Float aniso_B11;


	/**
	 *  The [1][1] element of the anisotropic atomic displacement
	 *  matrix B, which appears in the structure-factor term as:
	 *  
	 *  T = exp{-1/4 sum~i~[sum~j~(B^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 *  
	 *  The IUCr Commission on Nomenclature recommends against the use
	 *  of B for reporting atomic displacement parameters. U, being
	 *  directly proportional to B, is preferred.
	 */
	public java.lang.Float getAniso_B11() 
	{
		return aniso_B11;
	}
 
	public void setAniso_B11(java.lang.Float aniso_B11)
	{
		this.aniso_B11 = aniso_B11;
	}

 	protected java.lang.Float aniso_B11_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_B[1][1] in category atom_site.
	 */
	public java.lang.Float getAniso_B11_esd() 
	{
		return aniso_B11_esd;
	}
 
	public void setAniso_B11_esd(java.lang.Float aniso_B11_esd)
	{
		this.aniso_B11_esd = aniso_B11_esd;
	}

 	protected java.lang.Float aniso_B12;


	/**
	 *  The [1][2] element of the anisotropic atomic displacement
	 *  matrix B, which appears in the structure-factor term as:
	 *  
	 *  T = exp{-1/4 sum~i~[sum~j~(B^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 *  
	 *  The IUCr Commission on Nomenclature recommends against the use
	 *  of B for reporting atomic displacement parameters. U, being
	 *  directly proportional to B, is preferred.
	 */
	public java.lang.Float getAniso_B12() 
	{
		return aniso_B12;
	}
 
	public void setAniso_B12(java.lang.Float aniso_B12)
	{
		this.aniso_B12 = aniso_B12;
	}

 	protected java.lang.Float aniso_B12_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_B[1][2] in category atom_site.
	 */
	public java.lang.Float getAniso_B12_esd() 
	{
		return aniso_B12_esd;
	}
 
	public void setAniso_B12_esd(java.lang.Float aniso_B12_esd)
	{
		this.aniso_B12_esd = aniso_B12_esd;
	}

 	protected java.lang.Float aniso_B13;


	/**
	 *  The [1][3] element of the anisotropic atomic displacement
	 *  matrix B, which appears in the structure-factor term as:
	 *  
	 *  T = exp{-1/4 sum~i~[sum~j~(B^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 *  
	 *  The IUCr Commission on Nomenclature recommends against the use
	 *  of B for reporting atomic displacement parameters. U, being
	 *  directly proportional to B, is preferred.
	 */
	public java.lang.Float getAniso_B13() 
	{
		return aniso_B13;
	}
 
	public void setAniso_B13(java.lang.Float aniso_B13)
	{
		this.aniso_B13 = aniso_B13;
	}

 	protected java.lang.Float aniso_B13_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_B[1][3] in category atom_site.
	 */
	public java.lang.Float getAniso_B13_esd() 
	{
		return aniso_B13_esd;
	}
 
	public void setAniso_B13_esd(java.lang.Float aniso_B13_esd)
	{
		this.aniso_B13_esd = aniso_B13_esd;
	}

 	protected java.lang.Float aniso_B22;


	/**
	 *  The [2][2] element of the anisotropic atomic displacement
	 *  matrix B, which appears in the structure-factor term as:
	 *  
	 *  T = exp{-1/4 sum~i~[sum~j~(B^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 *  
	 *  The IUCr Commission on Nomenclature recommends against the use
	 *  of B for reporting atomic displacement parameters. U, being
	 *  directly proportional to B, is preferred.
	 */
	public java.lang.Float getAniso_B22() 
	{
		return aniso_B22;
	}
 
	public void setAniso_B22(java.lang.Float aniso_B22)
	{
		this.aniso_B22 = aniso_B22;
	}

 	protected java.lang.Float aniso_B22_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_B[2][2] in category atom_site.
	 */
	public java.lang.Float getAniso_B22_esd() 
	{
		return aniso_B22_esd;
	}
 
	public void setAniso_B22_esd(java.lang.Float aniso_B22_esd)
	{
		this.aniso_B22_esd = aniso_B22_esd;
	}

 	protected java.lang.Float aniso_B23;


	/**
	 *  The [2][3] element of the anisotropic atomic displacement
	 *  matrix B, which appears in the structure-factor term as:
	 *  
	 *  T = exp{-1/4 sum~i~[sum~j~(B^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 *  
	 *  The IUCr Commission on Nomenclature recommends against the use
	 *  of B for reporting atomic displacement parameters. U, being
	 *  directly proportional to B, is preferred.
	 */
	public java.lang.Float getAniso_B23() 
	{
		return aniso_B23;
	}
 
	public void setAniso_B23(java.lang.Float aniso_B23)
	{
		this.aniso_B23 = aniso_B23;
	}

 	protected java.lang.Float aniso_B23_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_B[2][3] in category atom_site.
	 */
	public java.lang.Float getAniso_B23_esd() 
	{
		return aniso_B23_esd;
	}
 
	public void setAniso_B23_esd(java.lang.Float aniso_B23_esd)
	{
		this.aniso_B23_esd = aniso_B23_esd;
	}

 	protected java.lang.Float aniso_B33;


	/**
	 *  The [3][3] element of the anisotropic atomic displacement
	 *  matrix B, which appears in the structure-factor term as:
	 *  
	 *  T = exp{-1/4 sum~i~[sum~j~(B^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 *  
	 *  The IUCr Commission on Nomenclature recommends against the use
	 *  of B for reporting atomic displacement parameters. U, being
	 *  directly proportional to B, is preferred.
	 */
	public java.lang.Float getAniso_B33() 
	{
		return aniso_B33;
	}
 
	public void setAniso_B33(java.lang.Float aniso_B33)
	{
		this.aniso_B33 = aniso_B33;
	}

 	protected java.lang.Float aniso_B33_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_B[3][3] in category atom_site.
	 */
	public java.lang.Float getAniso_B33_esd() 
	{
		return aniso_B33_esd;
	}
 
	public void setAniso_B33_esd(java.lang.Float aniso_B33_esd)
	{
		this.aniso_B33_esd = aniso_B33_esd;
	}

 	protected java.lang.Float aniso_U11;


	/**
	 *  The [1][1] element of the standard anisotropic atomic
	 *  displacement matrix U, which appears in the structure-factor
	 *  term as:
	 *  
	 *  T = exp{-2 pi^2^ sum~i~[sum~j~(U^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 */
	public java.lang.Float getAniso_U11() 
	{
		return aniso_U11;
	}
 
	public void setAniso_U11(java.lang.Float aniso_U11)
	{
		this.aniso_U11 = aniso_U11;
	}

 	protected java.lang.Float aniso_U11_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_U[1][1] in category atom_site.
	 */
	public java.lang.Float getAniso_U11_esd() 
	{
		return aniso_U11_esd;
	}
 
	public void setAniso_U11_esd(java.lang.Float aniso_U11_esd)
	{
		this.aniso_U11_esd = aniso_U11_esd;
	}

 	protected java.lang.Float aniso_U12;


	/**
	 *  The [1][2] element of the standard anisotropic atomic
	 *  displacement matrix U, which appears in the structure-factor
	 *  term as:
	 *  
	 *  T = exp{-2 pi^2^ sum~i~[sum~j~(U^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 */
	public java.lang.Float getAniso_U12() 
	{
		return aniso_U12;
	}
 
	public void setAniso_U12(java.lang.Float aniso_U12)
	{
		this.aniso_U12 = aniso_U12;
	}

 	protected java.lang.Float aniso_U12_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_U[1][2] in category atom_site.
	 */
	public java.lang.Float getAniso_U12_esd() 
	{
		return aniso_U12_esd;
	}
 
	public void setAniso_U12_esd(java.lang.Float aniso_U12_esd)
	{
		this.aniso_U12_esd = aniso_U12_esd;
	}

 	protected java.lang.Float aniso_U13;


	/**
	 *  The [1][3] element of the standard anisotropic atomic
	 *  displacement matrix U, which appears in the structure-factor
	 *  term as:
	 *  
	 *  T = exp{-2 pi^2^ sum~i~[sum~j~(U^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 */
	public java.lang.Float getAniso_U13() 
	{
		return aniso_U13;
	}
 
	public void setAniso_U13(java.lang.Float aniso_U13)
	{
		this.aniso_U13 = aniso_U13;
	}

 	protected java.lang.Float aniso_U13_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_U[1][3] in category atom_site.
	 */
	public java.lang.Float getAniso_U13_esd() 
	{
		return aniso_U13_esd;
	}
 
	public void setAniso_U13_esd(java.lang.Float aniso_U13_esd)
	{
		this.aniso_U13_esd = aniso_U13_esd;
	}

 	protected java.lang.Float aniso_U22;


	/**
	 *  The [2][2] element of the standard anisotropic atomic
	 *  displacement matrix U, which appears in the structure-factor
	 *  term as:
	 *  
	 *  T = exp{-2 pi^2^ sum~i~[sum~j~(U^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 */
	public java.lang.Float getAniso_U22() 
	{
		return aniso_U22;
	}
 
	public void setAniso_U22(java.lang.Float aniso_U22)
	{
		this.aniso_U22 = aniso_U22;
	}

 	protected java.lang.Float aniso_U22_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_U[2][2] in category atom_site.
	 */
	public java.lang.Float getAniso_U22_esd() 
	{
		return aniso_U22_esd;
	}
 
	public void setAniso_U22_esd(java.lang.Float aniso_U22_esd)
	{
		this.aniso_U22_esd = aniso_U22_esd;
	}

 	protected java.lang.Float aniso_U23;


	/**
	 *  The [2][3] element of the standard anisotropic atomic
	 *  displacement matrix U, which appears in the structure-factor
	 *  term as:
	 *  
	 *  T = exp{-2 pi^2^ sum~i~[sum~j~(U^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 */
	public java.lang.Float getAniso_U23() 
	{
		return aniso_U23;
	}
 
	public void setAniso_U23(java.lang.Float aniso_U23)
	{
		this.aniso_U23 = aniso_U23;
	}

 	protected java.lang.Float aniso_U23_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_U[2][3] in category atom_site.
	 */
	public java.lang.Float getAniso_U23_esd() 
	{
		return aniso_U23_esd;
	}
 
	public void setAniso_U23_esd(java.lang.Float aniso_U23_esd)
	{
		this.aniso_U23_esd = aniso_U23_esd;
	}

 	protected java.lang.Float aniso_U33;


	/**
	 *  The [3][3] element of the standard anisotropic atomic
	 *  displacement matrix U, which appears in the structure-factor
	 *  term as:
	 *  
	 *  T = exp{-2 pi^2^ sum~i~[sum~j~(U^ij^ h~i~ h~j~ a*~i~ a*~j~)]}
	 *  
	 *  h  = the Miller indices
	 *  a* = the reciprocal space cell lengths
	 *  
	 *  These matrix elements may appear with atomic coordinates
	 *  in the ATOM_SITE category, or they may appear in the separate
	 *  ATOM_SITE_ANISOTROP category, but they may not appear in both
	 *  places. Similarly, anisotropic displacements may appear as
	 *  either B's or U's, but not as both.
	 *  
	 *  The unique elements of the real symmetric matrix are
	 *  entered by row.
	 */
	public java.lang.Float getAniso_U33() 
	{
		return aniso_U33;
	}
 
	public void setAniso_U33(java.lang.Float aniso_U33)
	{
		this.aniso_U33 = aniso_U33;
	}

 	protected java.lang.Float aniso_U33_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute aniso_U[3][3] in category atom_site.
	 */
	public java.lang.Float getAniso_U33_esd() 
	{
		return aniso_U33_esd;
	}
 
	public void setAniso_U33_esd(java.lang.Float aniso_U33_esd)
	{
		this.aniso_U33_esd = aniso_U33_esd;
	}

 	protected java.lang.Float aniso_ratio;


	/**
	 *  Ratio of the maximum to minimum principal axes of
	 *  displacement (thermal) ellipsoids.
	 */
	public java.lang.Float getAniso_ratio() 
	{
		return aniso_ratio;
	}
 
	public void setAniso_ratio(java.lang.Float aniso_ratio)
	{
		this.aniso_ratio = aniso_ratio;
	}

 	protected java.lang.Integer attached_hydrogens;


	/**
	 *  The number of hydrogen atoms attached to the atom at this site
	 *  excluding any hydrogen atoms for which coordinates (measured or
	 *  calculated) are given.
	 *  water oxygen
	 *  2
	 *  hydroxyl oxygen
	 *  1
	 *  ammonium nitrogen
	 *  4
	 */
	public java.lang.Integer getAttached_hydrogens() 
	{
		return attached_hydrogens;
	}
 
	public void setAttached_hydrogens(java.lang.Integer attached_hydrogens)
	{
		this.attached_hydrogens = attached_hydrogens;
	}

 	protected java.lang.String auth_asym_id;


	/**
	 *  An alternative identifier for  attribute label_asym_id in category atom_site that
	 *  may be provided by an author in order to match the identification
	 *  used in the publication that describes the structure.
	 */
	public java.lang.String getAuth_asym_id() 
	{
		return auth_asym_id;
	}
 
	public void setAuth_asym_id(java.lang.String auth_asym_id)
	{
		this.auth_asym_id = auth_asym_id;
	}

 	protected java.lang.String auth_atom_id;


	/**
	 *  An alternative identifier for  attribute label_atom_id in category atom_site that
	 *  may be provided by an author in order to match the identification
	 *  used in the publication that describes the structure.
	 */
	public java.lang.String getAuth_atom_id() 
	{
		return auth_atom_id;
	}
 
	public void setAuth_atom_id(java.lang.String auth_atom_id)
	{
		this.auth_atom_id = auth_atom_id;
	}

 	protected java.lang.String auth_comp_id;


	/**
	 *  An alternative identifier for  attribute label_comp_id in category atom_site that
	 *  may be provided by an author in order to match the identification
	 *  used in the publication that describes the structure.
	 */
	public java.lang.String getAuth_comp_id() 
	{
		return auth_comp_id;
	}
 
	public void setAuth_comp_id(java.lang.String auth_comp_id)
	{
		this.auth_comp_id = auth_comp_id;
	}

 	protected java.lang.String auth_seq_id;


	/**
	 *  An alternative identifier for  attribute label_seq_id in category atom_site that
	 *  may be provided by an author in order to match the identification
	 *  used in the publication that describes the structure.
	 *  
	 *  Note that this is not necessarily a number, that the values do
	 *  not have to be positive, and that the value does not have to
	 *  correspond to the value of  attribute label_seq_id in category atom_site.  The value
	 *  of  attribute label_seq_id in category atom_site is required to be a sequential list
	 *  of positive integers.
	 *  
	 *  The author may assign values to  attribute auth_seq_id in category atom_site in any
	 *  desired way. For instance, the values may be used to relate
	 *  this structure to a numbering scheme in a homologous structure,
	 *  including sequence gaps or insertion codes. Alternatively, a
	 *  scheme may be used for a truncated polymer that maintains the
	 *  numbering scheme of the full length polymer. In all cases, the
	 *  scheme used here must match the scheme used in the publication
	 *  that describes the structure.
	 */
	public java.lang.String getAuth_seq_id() 
	{
		return auth_seq_id;
	}
 
	public void setAuth_seq_id(java.lang.String auth_seq_id)
	{
		this.auth_seq_id = auth_seq_id;
	}

 	protected java.lang.String calc_attached_atom;


	/**
	 *  The  attribute id in category atom_site of the atom site to which the
	 *  'geometry-calculated' atom site is attached.
	 */
	public java.lang.String getCalc_attached_atom() 
	{
		return calc_attached_atom;
	}
 
	public void setCalc_attached_atom(java.lang.String calc_attached_atom)
	{
		this.calc_attached_atom = calc_attached_atom;
	}

 	protected java.lang.String calc_flag;


	/**
	 *  A standard code to signal whether the site coordinates have been
	 *  determined from the intensities or calculated from the geometry
	 *  of surrounding sites, or have been assigned dummy values. The
	 *  abbreviation 'c' may be used in place of 'calc'.
	 */
	public java.lang.String getCalc_flag() 
	{
		return calc_flag;
	}
 
	public void setCalc_flag(java.lang.String calc_flag)
	{
		this.calc_flag = calc_flag;
	}

 	protected java.lang.Integer chemical_conn_number;


	/**
	 *  This data item is a pointer to  attribute number in category chemical_conn_atom in the
	 *  CHEMICAL_CONN_ATOM category.
	 */
	public java.lang.Integer getChemical_conn_number() 
	{
		return chemical_conn_number;
	}
 
	public void setChemical_conn_number(java.lang.Integer chemical_conn_number)
	{
		this.chemical_conn_number = chemical_conn_number;
	}

 	protected java.lang.String constraints;


	/**
	 *  A description of the constraints applied to parameters at this
	 *  site during refinement. See also  attribute refinement_flags
	 *  in category atom_site and  attribute ls_number_constraints in category refine.
	 *  pop=1.0-pop(Zn3)
	 */
	public java.lang.String getConstraints() 
	{
		return constraints;
	}
 
	public void setConstraints(java.lang.String constraints)
	{
		this.constraints = constraints;
	}

 	protected java.lang.String details;


	/**
	 *  A description of special aspects of this site. See also
	 *  attribute refinement_flags in category atom_site.
	 *  Ag/Si disordered
	 */
	public java.lang.String getDetails() 
	{
		return details;
	}
 
	public void setDetails(java.lang.String details)
	{
		this.details = details;
	}

 	protected java.lang.String disorder_assembly;


	/**
	 *  A code which identifies a cluster of atoms that show long-range
	 *  positional disorder but are locally ordered. Within each such
	 *  cluster of atoms,  attribute disorder_group in category atom_site is used to identify
	 *  the sites that are simultaneously occupied. This field is only
	 *  needed if there is more than one cluster of disordered atoms
	 *  showing independent local order.
	 *  
	 *  *** This data item would not in general be used in a
	 *  macromolecular data block. ***
	 */
	public java.lang.String getDisorder_assembly() 
	{
		return disorder_assembly;
	}
 
	public void setDisorder_assembly(java.lang.String disorder_assembly)
	{
		this.disorder_assembly = disorder_assembly;
	}

 	protected java.lang.String disorder_group;


	/**
	 *  A code which identifies a group of positionally disordered atom
	 *  sites that are locally simultaneously occupied. Atoms that are
	 *  positionally disordered over two or more sites (e.g. the hydrogen
	 *  atoms of a methyl group that exists in two orientations) can
	 *  be assigned to two or more groups. Sites belonging to the same
	 *  group are simultaneously occupied, but those belonging to
	 *  different groups are not. A minus prefix (e.g. '-1') is used to
	 *  indicate sites disordered about a special position.
	 *  
	 *  *** This data item would not in general be used in a
	 *  macromolecular data block. ***
	 */
	public java.lang.String getDisorder_group() 
	{
		return disorder_group;
	}
 
	public void setDisorder_group(java.lang.String disorder_group)
	{
		this.disorder_group = disorder_group;
	}

 	protected java.lang.String footnote_id;


	/**
	 *  The value of  attribute footnote_id in category atom_site must match an ID
	 *  specified by  attribute id in category atom_sites_footnote in the
	 *  ATOM_SITES_FOOTNOTE list.
	 */
	public java.lang.String getFootnote_id() 
	{
		return footnote_id;
	}
 
	public void setFootnote_id(java.lang.String footnote_id)
	{
		this.footnote_id = footnote_id;
	}

 	protected java.lang.Float fract_x;


	/**
	 *  The x coordinate of the atom-site position specified as a
	 *  fraction of  attribute length_a in category cell.
	 */
	public java.lang.Float getFract_x() 
	{
		return fract_x;
	}
 
	public void setFract_x(java.lang.Float fract_x)
	{
		this.fract_x = fract_x;
	}

 	protected java.lang.Float fract_x_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute fract_x in category atom_site.
	 */
	public java.lang.Float getFract_x_esd() 
	{
		return fract_x_esd;
	}
 
	public void setFract_x_esd(java.lang.Float fract_x_esd)
	{
		this.fract_x_esd = fract_x_esd;
	}

 	protected java.lang.Float fract_y;


	/**
	 *  The y coordinate of the atom-site position specified as a
	 *  fraction of  attribute length_b in category cell.
	 */
	public java.lang.Float getFract_y() 
	{
		return fract_y;
	}
 
	public void setFract_y(java.lang.Float fract_y)
	{
		this.fract_y = fract_y;
	}

 	protected java.lang.Float fract_y_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute fract_y in category atom_site.
	 */
	public java.lang.Float getFract_y_esd() 
	{
		return fract_y_esd;
	}
 
	public void setFract_y_esd(java.lang.Float fract_y_esd)
	{
		this.fract_y_esd = fract_y_esd;
	}

 	protected java.lang.Float fract_z;


	/**
	 *  The z coordinate of the atom-site position specified as a
	 *  fraction of  attribute length_c in category cell.
	 */
	public java.lang.Float getFract_z() 
	{
		return fract_z;
	}
 
	public void setFract_z(java.lang.Float fract_z)
	{
		this.fract_z = fract_z;
	}

 	protected java.lang.Float fract_z_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute fract_z in category atom_site.
	 */
	public java.lang.Float getFract_z_esd() 
	{
		return fract_z_esd;
	}
 
	public void setFract_z_esd(java.lang.Float fract_z_esd)
	{
		this.fract_z_esd = fract_z_esd;
	}

 	protected java.lang.String group_PDB;


	/**
	 *  The group of atoms to which the atom site belongs. This data
	 *  item is provided for compatibility with the original Protein
	 *  Data Bank format, and only for that purpose.
	 */
	public java.lang.String getGroup_PDB() 
	{
		return group_PDB;
	}
 
	public void setGroup_PDB(java.lang.String group_PDB)
	{
		this.group_PDB = group_PDB;
	}

 	protected java.lang.String id_;


	/**
	 *  The value of  attribute id in category atom_site must uniquely identify a record in the
	 *  ATOM_SITE list.
	 *  
	 *  Note that this item need not be a number; it can be any unique
	 *  identifier.
	 *  
	 *  This data item was introduced to provide compatibility between
	 *  small-molecule and macromolecular CIFs. In a small-molecule
	 *  CIF, _atom_site_label is the identifier for the atom. In a
	 *  macromolecular CIF, the atom identifier is the aggregate of
	 *  _atom_site.label_alt_id, _atom_site.label_asym_id,
	 *  _atom_site.label_atom_id, _atom_site.label_comp_id and
	 *  attribute label_seq_id in category atom_site.  For the two types of files to be
	 *  compatible, a formal identifier for the category had to be
	 *  introduced that was independent of the different modes of
	 *  identifying the atoms. For compatibility with older CIFs,
	 *  _atom_site_label is aliased to  attribute id in category atom_site.
	 *  5
	 *  C12
	 *  Ca3g28
	 *  Fe3+17
	 *  H*251
	 *  boron2a
	 *  C_a_phe_83_a_0
	 *  Zn_Zn_301_A_0
	 */
	public java.lang.String getId_() 
	{
		return id_;
	}
 
	public void setId_(java.lang.String id_)
	{
		this.id_ = id_;
	}

 	protected java.lang.String label_alt_id;


	/**
	 *  A component of the identifier for this atom site.
	 *  For further details, see the definition of the ATOM_SITE_ALT
	 *  category.
	 *  
	 *  This data item is a pointer to  attribute id in category atom_sites_alt in the
	 *  ATOM_SITES_ALT category.
	 */
	public java.lang.String getLabel_alt_id() 
	{
		return label_alt_id;
	}
 
	public void setLabel_alt_id(java.lang.String label_alt_id)
	{
		this.label_alt_id = label_alt_id;
	}

 	protected java.lang.String label_asym_id;


	/**
	 *  A component of the identifier for this atom site.
	 *  For further details, see the definition of the STRUCT_ASYM
	 *  category.
	 *  
	 *  This data item is a pointer to  attribute id in category struct_asym in the
	 *  STRUCT_ASYM category.
	 */
	public java.lang.String getLabel_asym_id() 
	{
		return label_asym_id;
	}
 
	public void setLabel_asym_id(java.lang.String label_asym_id)
	{
		this.label_asym_id = label_asym_id;
	}

 	protected java.lang.String label_atom_id;


	/**
	 *  A component of the identifier for this atom site.
	 *  
	 *  This data item is a pointer to  attribute atom_id in category chem_comp_atom in the
	 *  CHEM_COMP_ATOM category.
	 */
	public java.lang.String getLabel_atom_id() 
	{
		return label_atom_id;
	}
 
	public void setLabel_atom_id(java.lang.String label_atom_id)
	{
		this.label_atom_id = label_atom_id;
	}

 	protected java.lang.String label_comp_id;


	/**
	 *  A component of the identifier for this atom site.
	 *  
	 *  This data item is a pointer to  attribute id in category chem_comp in the CHEM_COMP
	 *  category.
	 */
	public java.lang.String getLabel_comp_id() 
	{
		return label_comp_id;
	}
 
	public void setLabel_comp_id(java.lang.String label_comp_id)
	{
		this.label_comp_id = label_comp_id;
	}

 	protected java.lang.String label_entity_id;


	/**
	 *  This data item is a pointer to  attribute id in category entity in the ENTITY category.
	 */
	public java.lang.String getLabel_entity_id() 
	{
		return label_entity_id;
	}
 
	public void setLabel_entity_id(java.lang.String label_entity_id)
	{
		this.label_entity_id = label_entity_id;
	}

 	protected java.lang.Integer label_seq_id;


	/**
	 *  This data item is a pointer to  attribute num in category entity_poly_seq in the
	 *  ENTITY_POLY_SEQ category.
	 */
	public java.lang.Integer getLabel_seq_id() 
	{
		return label_seq_id;
	}
 
	public void setLabel_seq_id(java.lang.Integer label_seq_id)
	{
		this.label_seq_id = label_seq_id;
	}

 	protected java.lang.Float occupancy;


	/**
	 *  The fraction of the atom type present at this site.
	 *  The sum of the occupancies of all the atom types at this site
	 *  may not significantly exceed 1.0 unless it is a dummy site.
	 */
	public java.lang.Float getOccupancy() 
	{
		return occupancy;
	}
 
	public void setOccupancy(java.lang.Float occupancy)
	{
		this.occupancy = occupancy;
	}

 	protected java.lang.Float occupancy_esd;


	/**
	 *  The standard uncertainty (estimated standard deviation)
	 *  of  attribute occupancy in category atom_site.
	 */
	public java.lang.Float getOccupancy_esd() 
	{
		return occupancy_esd;
	}
 
	public void setOccupancy_esd(java.lang.Float occupancy_esd)
	{
		this.occupancy_esd = occupancy_esd;
	}

 	protected java.lang.String pdbx_PDB_atom_name;


	/**
	 *  PDB atom name.
	 */
	public java.lang.String getPdbx_PDB_atom_name() 
	{
		return pdbx_PDB_atom_name;
	}
 
	public void setPdbx_PDB_atom_name(java.lang.String pdbx_PDB_atom_name)
	{
		this.pdbx_PDB_atom_name = pdbx_PDB_atom_name;
	}

 	protected java.lang.String pdbx_PDB_ins_code;


	/**
	 *  PDB insertion code.
	 */
	public java.lang.String getPdbx_PDB_ins_code() 
	{
		return pdbx_PDB_ins_code;
	}
 
	public void setPdbx_PDB_ins_code(java.lang.String pdbx_PDB_ins_code)
	{
		this.pdbx_PDB_ins_code = pdbx_PDB_ins_code;
	}

 	protected java.lang.Integer pdbx_PDB_model_num;


	/**
	 *  PDB model number.
	 */
	public java.lang.Integer getPdbx_PDB_model_num() 
	{
		return pdbx_PDB_model_num;
	}
 
	public void setPdbx_PDB_model_num(java.lang.Integer pdbx_PDB_model_num)
	{
		this.pdbx_PDB_model_num = pdbx_PDB_model_num;
	}

 	protected java.lang.String pdbx_PDB_residue_name;


	/**
	 *  PDB residue name.
	 */
	public java.lang.String getPdbx_PDB_residue_name() 
	{
		return pdbx_PDB_residue_name;
	}
 
	public void setPdbx_PDB_residue_name(java.lang.String pdbx_PDB_residue_name)
	{
		this.pdbx_PDB_residue_name = pdbx_PDB_residue_name;
	}

 	protected java.lang.String pdbx_PDB_residue_no;


	/**
	 *  PDB residue number.
	 */
	public java.lang.String getPdbx_PDB_residue_no() 
	{
		return pdbx_PDB_residue_no;
	}
 
	public void setPdbx_PDB_residue_no(java.lang.String pdbx_PDB_residue_no)
	{
		this.pdbx_PDB_residue_no = pdbx_PDB_residue_no;
	}

 	protected java.lang.String pdbx_PDB_strand_id;


	/**
	 *  PDB strand id.
	 */
	public java.lang.String getPdbx_PDB_strand_id() 
	{
		return pdbx_PDB_strand_id;
	}
 
	public void setPdbx_PDB_strand_id(java.lang.String pdbx_PDB_strand_id)
	{
		this.pdbx_PDB_strand_id = pdbx_PDB_strand_id;
	}

 	protected java.lang.String pdbx_auth_alt_id;


	/**
	 *  Author's alternate location identifier.
	 */
	public java.lang.String getPdbx_auth_alt_id() 
	{
		return pdbx_auth_alt_id;
	}
 
	public void setPdbx_auth_alt_id(java.lang.String pdbx_auth_alt_id)
	{
		this.pdbx_auth_alt_id = pdbx_auth_alt_id;
	}

 	protected java.lang.String pdbx_auth_atom_name;


	/**
	 *  Author's atom name.
	 */
	public java.lang.String getPdbx_auth_atom_name() 
	{
		return pdbx_auth_atom_name;
	}
 
	public void setPdbx_auth_atom_name(java.lang.String pdbx_auth_atom_name)
	{
		this.pdbx_auth_atom_name = pdbx_auth_atom_name;
	}

 	protected java.lang.String refinement_flags;


	/**
	 *  A concatenated series of single-letter codes which indicate the
	 *  refinement restraints or constraints applied to this site.  This
	 *  item should not be used.  It has been replaced by
	 *  attribute refinement_flags_posn in category atom_site *_adp and *_occupancy. It is
	 *  retained in this dictionary only to provide compatibility with
	 *  old CIFs.
	 */
	public java.lang.String getRefinement_flags() 
	{
		return refinement_flags;
	}
 
	public void setRefinement_flags(java.lang.String refinement_flags)
	{
		this.refinement_flags = refinement_flags;
	}

 	protected java.lang.String refinement_flags_adp;


	/**
	 *  A code which indicates the refinement restraints or constraints
	 *  applied to the atomic displacement parameters of this site.
	 */
	public java.lang.String getRefinement_flags_adp() 
	{
		return refinement_flags_adp;
	}
 
	public void setRefinement_flags_adp(java.lang.String refinement_flags_adp)
	{
		this.refinement_flags_adp = refinement_flags_adp;
	}

 	protected java.lang.String refinement_flags_occupancy;


	/**
	 *  A code which indicates that refinement restraints or
	 *  constraints were applied to the occupancy of this site.
	 */
	public java.lang.String getRefinement_flags_occupancy() 
	{
		return refinement_flags_occupancy;
	}
 
	public void setRefinement_flags_occupancy(java.lang.String refinement_flags_occupancy)
	{
		this.refinement_flags_occupancy = refinement_flags_occupancy;
	}

 	protected java.lang.String refinement_flags_posn;


	/**
	 *  A code which indicates the refinement restraints or constraints
	 *  applied to the positional coordinates of this site.
	 */
	public java.lang.String getRefinement_flags_posn() 
	{
		return refinement_flags_posn;
	}
 
	public void setRefinement_flags_posn(java.lang.String refinement_flags_posn)
	{
		this.refinement_flags_posn = refinement_flags_posn;
	}

 	protected java.lang.String restraints;


	/**
	 *  A description of restraints applied to specific parameters at
	 *  this site during refinement. See also  attribute refinement_flags
	 *  in category atom_site and  attribute ls_number_restraints in category refine.
	 *  restrained to planar ring
	 */
	public java.lang.String getRestraints() 
	{
		return restraints;
	}
 
	public void setRestraints(java.lang.String restraints)
	{
		this.restraints = restraints;
	}

 	protected java.lang.Integer symmetry_multiplicity;


	/**
	 *  The multiplicity of a site due to the space-group symmetry as is
	 *  given in International Tables for Crystallography Vol. A (2002).
	 */
	public java.lang.Integer getSymmetry_multiplicity() 
	{
		return symmetry_multiplicity;
	}
 
	public void setSymmetry_multiplicity(java.lang.Integer symmetry_multiplicity)
	{
		this.symmetry_multiplicity = symmetry_multiplicity;
	}

 	protected java.lang.String thermal_displace_type;


	/**
	 *  A standard code used to describe the type of atomic displacement
	 *  parameters used for the site.
	 */
	public java.lang.String getThermal_displace_type() 
	{
		return thermal_displace_type;
	}
 
	public void setThermal_displace_type(java.lang.String thermal_displace_type)
	{
		this.thermal_displace_type = thermal_displace_type;
	}

 	protected java.lang.String type_symbol;


	/**
	 *  This data item is a pointer to  attribute symbol in category atom_type in the
	 *  ATOM_TYPE category.
	 */
	public java.lang.String getType_symbol() 
	{
		return type_symbol;
	}
 
	public void setType_symbol(java.lang.String type_symbol)
	{
		this.type_symbol = type_symbol;
	}

 }
