/***************************************************************************
* Title:          BasicDataStructs.h
* Author:         Ari Frank
* Copyright (c) 2009 The Regents of the University of California
* All Rights Reserved
* See file LICENSE for details.
***************************************************************************/

/******************************************************************************
Copyright 2008, The Regents of the University of California
All Rights Reserved

Permission to use, copy, modify and distribute any part of this
program for educational, research and non-profit purposes, without fee,
and without a written agreement is hereby granted, provided that the
above copyright notice, this paragraph and the following three paragraphs
appear in all copies.

Those desiring to incorporate this work into commercial
products or use for commercial purposes should contact the Technology
Transfer & Intellectual Property Services, University of California,
San Diego, 9500 Gilman Drive, Mail Code 0910, La Jolla, CA 92093-0910,
Ph: (858) 534-5815, FAX: (858) 534-7345, E-MAIL:invent@ucsd.edu.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE, EVEN
IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.

THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.  THE UNIVERSITY OF CALIFORNIA MAKES NO
REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER IMPLIED OR
EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR THAT THE USE OF
THE SOFTWARE WILL NOT INFRINGE ANY PATENT, TRADEMARK OR OTHER RIGHTS.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*******************************************************************************/



#ifndef __BASICDATASTRUCTS_H__
#define __BASICDATASTRUCTS_H__

#include "includes.h"

class Config;

class Peptide {
public:
	Peptide() : mass(0), n_term(N_TERM), c_term(C_TERM) {};

	void clear() { mass=0; n_term=N_TERM; c_term=C_TERM; amino_acids.clear(); gaps.clear(); }

	bool operator== (const Peptide& other) const
	{
		if (amino_acids.size() != other.amino_acids.size())
			return false;
		unsigned int i;
		for (i=0; i<amino_acids.size(); i++)
			if (amino_acids[i] != other.amino_acids[i])
				return false;
		return true;
	}

	mass_t get_mass() const { return mass; } // mass of aas + terminals, doesn't include 19
	int get_n_term() const { return n_term; }
	int get_c_term() const { return c_term; }
	void set_n_term(int aa) { n_term = aa; }
	void set_c_term(int aa) { c_term = aa; }


	int get_length() const { return amino_acids.size(); } // gaps count as amino acids
	int get_num_aas() const { return amino_acids.size() - gaps.size(); }
	int get_num_gaps() const { return gaps.size(); }

	const vector<int>& get_amino_acids() const { return amino_acids; }

	/*********************************************************************
	Returns the global edit distance between two peptides.
	**********************************************************************/
	float peptide_edit_distance(Config *config, Peptide& other) const;

	// changes the amino acids I->L
	// an Q->K if not at terminal and tolerance > 0.1
	void convert_ILQK(const Config *config);

	// changes all the amino acids to their original form (without PTMs)
	void convert_to_org(const Config *config);

	void convert_IL();

	void reverse();

	string as_string(const Config* config) const;

	void parse_from_string(const Config* config, const string& str);

	void set_peptide(vector<int>& aas, vector<mass_t>& gaps, mass_t mass, 
					 int n_term_aa = N_TERM, int c_term_aa = C_TERM);

	void set_peptide_aas(const vector<int>& aas) { amino_acids = aas; }

	void calc_mass(const Config *config);

	void calc_expected_breakage_masses(Config *config, vector<mass_t>& break_masses) const;

private:

	mass_t mass; // mass of aas + terminals, doesn't include 19
	int n_term; // N_TERM or a modification (not the aa N-terminal to the peptide)
	int c_term; // C_TERM or a modification (not the aa C-terminal to the peptide)

	vector<int>    amino_acids;
	vector<mass_t> gaps;
};



struct CutPeak {
	CutPeak() : monoMass(-1), tolerance(-1), score(0), clusterIdx(-1),
				originalPeakIdx(-1), fragmentInterpertation(-1) {};

	bool operator < (const CutPeak& other) const
	{
		return (monoMass<other.monoMass);
	}

	mass_t	monoMass;
	mass_t	tolerance; // how much +- mass to tolerate with this peak
	score_t score;
	int		clusterIdx;
	int		originalPeakIdx;
	int		fragmentInterpertation; // the idx of the fragment type used to create this mass
};






	


#endif




