/***************************************************************************
* Title:          MonoAlignment.cpp
* Author:         Ari Frank
* Copyright (c) 2009 The Regents of the University of California
* All Rights Reserved
* See file LICENSE for details.
***************************************************************************/

/******************************************************************************
Copyright 2008, The Regents of the University of California
All Rights Reserved

Permission to use, copy, modify and distribute any part of this
program for educational, research and non-profit purposes, without fee,
and without a written agreement is hereby granted, provided that the
above copyright notice, this paragraph and the following three paragraphs
appear in all copies.

Those desiring to incorporate this work into commercial
products or use for commercial purposes should contact the Technology
Transfer & Intellectual Property Services, University of California,
San Diego, 9500 Gilman Drive, Mail Code 0910, La Jolla, CA 92093-0910,
Ph: (858) 534-5815, FAX: (858) 534-7345, E-MAIL:invent@ucsd.edu.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE, EVEN
IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.

THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.  THE UNIVERSITY OF CALIFORNIA MAKES NO
REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER IMPLIED OR
EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR THAT THE USE OF
THE SOFTWARE WILL NOT INFRINGE ANY PATENT, TRADEMARK OR OTHER RIGHTS.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*******************************************************************************/




#include "MonoAlignment.h"




// adds a new pointer to the list of pointers if its score is greater
// than any of the existing (pointer with least score is removed)
// does not use heap becasue the list is ususally short
void DpCell::addPointer(const PathPointer& newPointer, int maxNumPointers)
{
	int i;
	int minIdx =0;

	if (newPointer.previousCell == pointers[0].previousCell)
		return;

	for (i=1; i<maxNumPointers; i++)
	{
		if (pointers[i]<pointers[minIdx])
			minIdx=i;

		// no duplicates allowed
		if (newPointer.previousCell == pointers[i].previousCell)
			return;
	}

	if (pointers[minIdx]<newPointer)
		pointers[minIdx]=newPointer;
}

// sort the pointers so they appear in descending score
// use simple bubble sort
void DpCell::sortPointers(int maxNumPointers)
{
	if (maxNumPointers == 1)
		return;

	const int lastIdx = maxNumPointers-1;
	int i;
	for (i=0; i<lastIdx; i++)
	{
		int j;
		for (j=0; j<lastIdx; j++)
			if (pointers[j]<pointers[j+1])
			{
				static PathPointer t; //swap
				t=pointers[j+1];
				pointers[j+1]=pointers[j];
				pointers[j]=t;
			}	
	}
}


/**********************************************************************

  Build table of codiagonal pairs of peaks and prefix masses
  lists all the shift pairs in the pairs vecote
***********************************************************************/
void CoDiags::calcCodiags(const vector<CutPeak>& cutPeaks, 
						  const vector<mass_t>& prefixMasses,
						  const vector<PtmCombo>& ptmCombos)
{
	vector<PairShift> pairs;
	pairs.clear();
	pairs.reserve(cutPeaks.size()*prefixMasses.size());
	unsigned int peakIdx;
	for (peakIdx=0; peakIdx<cutPeaks.size(); peakIdx++)
	{
		unsigned int aaIdx;
		for (aaIdx=0; aaIdx<prefixMasses.size(); aaIdx++)
		{
			PairShift ps;
			ps.idxs.peakIdx=peakIdx;
			ps.idxs.aaIdx=aaIdx;
			ps.massShift = cutPeaks[peakIdx].monoMass - prefixMasses[aaIdx];
			pairs.push_back(ps);

	//		if ((peakIdx==141 && aaIdx==87) || (peakIdx==125 && aaIdx==71) || 
	//			 (peakIdx==129 && aaIdx==73) || (peakIdx==129 && aaIdx==73) )
	//		{
	//			cout << peakIdx << "," << aaIdx << "=" << ps.massShift << endl;
	//		}
		}
	}
	


	sort(pairs.begin(),pairs.end());

	prevCodiag.resize(cutPeaks.size());
	allCodiags.resize(cutPeaks.size());
	ptmCodiags.resize(cutPeaks.size());


	for (peakIdx=0; peakIdx<cutPeaks.size(); peakIdx++)
	{
		prevCodiag[peakIdx].resize(prefixMasses.size());
		allCodiags[peakIdx].resize(prefixMasses.size());
		ptmCodiags[peakIdx].resize(prefixMasses.size());
	}
	
	unsigned int p;
	for (p=0; p<pairs.size(); p++)
	{
		const PairShift& currPair =  pairs[p];
		const mass_t& tolerance = cutPeaks[currPair.idxs.peakIdx].tolerance;
		const mass_t minShiftVal = currPair.massShift - tolerance;
		const mass_t maxShiftVal = currPair.massShift + tolerance;

		IdxPair closestPair;
		closestPair.peakIdx=-1;
		closestPair.aaIdx=-1;
		int piv=p-1;
		int bestPiv=-1;
		while (piv>=0 && pairs[piv].massShift>=minShiftVal)
		{
			if (pairs[piv].idxs.peakIdx < currPair.idxs.peakIdx)
				allCodiags[currPair.idxs.peakIdx][currPair.idxs.aaIdx].push_back(pairs[piv].idxs);

			if (pairs[piv].idxs.peakIdx < currPair.idxs.peakIdx &&
				pairs[piv].idxs.aaIdx   < currPair.idxs.aaIdx  && 
				pairs[piv].idxs.peakIdx > closestPair.peakIdx)
			{
				
				closestPair = pairs[piv].idxs;
				bestPiv=piv;
			}
			piv--;
		}

		piv = p+1;
		while (piv<static_cast<int>(pairs.size()) && pairs[piv].massShift<=maxShiftVal)
		{
			if (pairs[piv].idxs.peakIdx < currPair.idxs.peakIdx)
				allCodiags[currPair.idxs.peakIdx][currPair.idxs.aaIdx].push_back(pairs[piv].idxs);

			if (pairs[piv].idxs.peakIdx < currPair.idxs.peakIdx &&
				pairs[piv].idxs.aaIdx   < currPair.idxs.aaIdx   &&
				pairs[piv].idxs.peakIdx > closestPair.peakIdx)
			{
				
				closestPair = pairs[piv].idxs;
				bestPiv=piv;
			}
			piv++;
		}

		prevCodiag[currPair.idxs.peakIdx][currPair.idxs.aaIdx]=closestPair;
	}



	// calc the coDiags for ptms
	int lastIdxInspected = 1;
	for (p=0; p<pairs.size(); p++)
	{
		const PairShift& currPair =  pairs[p];

		vector<IdxPair>& currentPairPtmCodiags = ptmCodiags[currPair.idxs.peakIdx][currPair.idxs.aaIdx];
		
		currentPairPtmCodiags.clear();
		currentPairPtmCodiags.resize(ptmCombos.size());
		
		int comboIdx;
		const int startComboIdx=1; // ignore first dummy
		const int lastComboIdx =ptmCombos.size()-2; // ignore last dummy

		for (comboIdx=startComboIdx; comboIdx<=lastComboIdx; comboIdx++)
		{
			const PtmCombo& combo = ptmCombos[comboIdx];
			const mass_t& tolerance = cutPeaks[currPair.idxs.peakIdx].tolerance;
			const mass_t minShiftVal = currPair.massShift - combo.totalMassJump - tolerance;
			const mass_t maxShiftVal = currPair.massShift - combo.totalMassJump + tolerance;

			IdxPair closestPair;
			closestPair.peakIdx=-1;
			closestPair.aaIdx=-1;

			bool verb=false;
			if (0 && currPair.idxs.peakIdx == 19 && currPair.idxs.aaIdx == 16)
			{
				verb=true;
			}

			if (verb)
			{
				cout << comboIdx << ">> " << currPair.idxs << " : " << setprecision(3) << currPair.massShift << " " << minShiftVal << " - " << maxShiftVal << endl;
			}

			int piv=lastIdxInspected-1;

			int bestPiv=-1;


			while (piv<(int)pairs.size() && pairs[piv].massShift<minShiftVal)
				piv++;
			
			if (piv == pairs.size())
				piv--;

			while (piv>0 && pairs[piv].massShift>=minShiftVal)
				piv--;

			piv++;
			while (piv<static_cast<int>(pairs.size()) && pairs[piv].massShift<= maxShiftVal)
			{
				if (verb)
					cout << pairs[piv].idxs << " " << pairs[piv].massShift << endl;

				if (pairs[piv].idxs.peakIdx < currPair.idxs.peakIdx &&
					pairs[piv].idxs.aaIdx   < currPair.idxs.aaIdx &&
					pairs[piv].idxs.peakIdx > closestPair.peakIdx)
				{
					
					PairShift pp = pairs[piv];
					closestPair = pairs[piv].idxs;
					bestPiv=piv;
				}
				piv++;
			}

			currentPairPtmCodiags[comboIdx]=closestPair;
			if (verb && closestPair.peakIdx>=0)
			{
				cout << "ptmCD: " << currPair.idxs << " with  " << closestPair << "  combo: " << combo << endl << endl;
			}

			if (verb && currPair.idxs.peakIdx==141 && currPair.idxs.aaIdx==87 && closestPair.peakIdx>=0)
			{	
				cout << currPair.idxs << " " << currPair.massShift << endl;
				cout << closestPair   << " " << cutPeaks[closestPair.peakIdx].monoMass-prefixMasses[closestPair.aaIdx] << endl;
				cout << "Mass shift allowed " << (maxShiftVal - minShiftVal)/2 <<endl;
			}

			if (bestPiv>=0)
				lastIdxInspected = bestPiv-1;
		}
	}
	
}


/********************************************************************************
	This function is given a list of prefix peak masses (after converting them 
	from the original masses of their c,z fragments to their prefix cut masses)
*********************************************************************************/
void MonoAlignment::makeQuickDpTable(Config *config, 
									 int maxNumGeneralJumps, 
									 int maxNumSpecificJumps,
									 int maxNumForms,
									 const ProteinSequence& ps, 
									 const vector<CutPeak>& cutPeaks,
									 const vector<PTM>& allPtms,
									 bool verbose)
{
	const vector<int>& aminoAcids = ps.getAminoAcids();
	const int numPeaks = cutPeaks.size();
	const vector<mass_t>& aa2mass = config->get_aa2mass();

	vector<mass_t> prefixAminoAcidMasses;
	int generalIdx, // number of general shifts
		s; // number of specific shifts

	ps.calcNonModifiedCutMasses(prefixAminoAcidMasses);
	initAaCounts();

	const int numAminoAcids   = prefixAminoAcidMasses.size();
	const int numConsecutivePtms = (maxNumSpecificJumps<=2 ? maxNumSpecificJumps : 2);
	initAlignmentPtms(allPtms, numConsecutivePtms);

	coDiags.calcCodiags(cutPeaks,prefixAminoAcidMasses, ptmManager.getPtmCombos());

	int p,q;
	if (0 && verbose)
	{
		for (p=0; p<9; p++)
			for (q=0; q<9; q++)
			{
				IdxPair diag = coDiags.getPtmComboCodiag(5,p,q);
			
				vector<IdxPair> diags;
				if (diag.peakIdx>=0)
					diags.push_back(diag);

				if (diags.size() == 0)
					continue;

				cout << "[" << p << "," << q << "] ";
				unsigned int t;
				for (t=0; t<diags.size(); t++)
					cout << " (" << diags[t].peakIdx << "," << diags[t].aaIdx << ")";
				cout << endl;

			}
		cout << endl;
	}


	D.clear();
	D.resize(maxNumGeneralJumps+1);

	for (generalIdx=0; generalIdx<=maxNumGeneralJumps; generalIdx++)
	{
		D[generalIdx].resize(maxNumSpecificJumps+1);
		for (s=0; s<=maxNumSpecificJumps; s++)
		{
			int peakIdx;
			D[generalIdx][s].resize(numPeaks);
			
			for (peakIdx=0; peakIdx<numPeaks; peakIdx++)
			{
				D[generalIdx][s][peakIdx].resize(numAminoAcids);

				int aaIdx;
				for (aaIdx=0; aaIdx<numAminoAcids; aaIdx++)
					D[generalIdx][s][peakIdx][aaIdx].init(maxNumForms);
			}
		}
	}

	M=D;
	
	const DpCellLocation zeroLoc = DpCellLocation(0,0,0,0);
	
	D[0][0][0][0].init(maxNumForms);
	D[0][0][0][0].pointers[0].score=0;
	M[0][0][0][0].init(maxNumForms);

	int peakIdx;
	for (peakIdx=1; peakIdx<numPeaks; peakIdx++)
	{
		M[0][0][peakIdx][0].pointers[0].score =0;
		M[0][0][peakIdx][0].pointers[0].previousCell = zeroLoc;
	}

	int aaIdx;
	for (aaIdx=1; aaIdx<numAminoAcids; aaIdx++)
	{
		M[0][0][0][aaIdx].pointers[0].score=0;
		M[0][0][0][aaIdx].pointers[0].previousCell = zeroLoc;
	} 
	
	// fill table
	for (generalIdx=0; generalIdx<=maxNumGeneralJumps; generalIdx++) 
	{
		int s;
		for (s=0; s<=maxNumSpecificJumps; s++)
		{
			vector< vector<DpCell> >& Dap = D[generalIdx][s];
			vector< vector<DpCell> >& Map = M[generalIdx][s];

			int peakIdx;
			for (peakIdx=1; peakIdx<numPeaks; peakIdx++)
			{
				const mass_t peakMass = cutPeaks[peakIdx].monoMass;
				const mass_t massTolerance = cutPeaks[peakIdx].tolerance;

				int aaIdx;
				for (aaIdx=1; aaIdx<numAminoAcids; aaIdx++)
				{
					const mass_t aaMass = prefixAminoAcidMasses[aaIdx];

					DpCell& currentDCell = Dap[peakIdx][aaIdx];
					
					// add pointers from paths from the diagonal
					IdxPair diagLocation = coDiags.getCodiag(peakIdx,aaIdx);

			
					if (diagLocation.peakIdx>=0)
					{
						const DpCell& prevDiagCell = Dap[diagLocation.peakIdx][diagLocation.aaIdx];
						
						int formIdx;
						for (formIdx=0; 
							 formIdx<maxNumForms && prevDiagCell.pointers[formIdx].score>=0; 
							 formIdx++)
						{
							PathPointer pp;
							pp.massShift = 0;
							pp.score = prevDiagCell.pointers[formIdx].score;
							pp.previousCell = DpCellLocation(generalIdx,s,
								diagLocation.peakIdx, diagLocation.aaIdx, formIdx);

							// check that the pointer being added does not equal the previous
							// pointers of one of the pointers already added. If this is the case
							// then all we are doing is skipping a peak on the same diagonal
							// which is not what we want (we want new added peaks)
							int i;
							for (i=0; i<maxNumForms; i++)
								if (currentDCell.pointers[i].score>=0 &&
									doesPathContainCellLocation(currentDCell.pointers[i].previousCell, pp.previousCell))
										break;
							if (i<maxNumForms)
								continue;

							currentDCell.addPointer(pp,maxNumForms);
						}
					}

					// add pointers from PTM diags
					if (s>0)
					{	
						const vector<PtmCombo>& ptmCombos = ptmManager.getPtmCombos();
						
						int comboIdx;
						int lastIdx = ptmCombos.size()-2;
						for (comboIdx=1; comboIdx<= lastIdx; comboIdx++)
						{
							const PtmCombo& combo = ptmCombos[comboIdx];
							const int sources = s- combo.numPtmsInCombo;

							if (sources<0)
								continue; // the combo requires to many ptms to be satisfied on this s
							
							const IdxPair ptmDiagLocation = coDiags.getPtmComboCodiag(comboIdx, peakIdx, aaIdx);

							if (ptmDiagLocation.peakIdx<0)
								continue; // no cell on the ptm diagonal was found

							const int previousAaIdx = ptmDiagLocation.aaIdx;
							if (! checkIfComboSatisfiable(comboIdx,previousAaIdx,aaIdx))
								continue; // the aas from the diag location to the currect aaIdx
										  // cannot satisfy this ptm combination

							const DpCell& ptmDiagCell = D[generalIdx][sources][ptmDiagLocation.peakIdx][ptmDiagLocation.aaIdx];
							int formIdx;
							for (formIdx=0; formIdx<maxNumForms; formIdx++)
							{
								PathPointer pp;

								pp.score = ptmDiagCell.pointers[formIdx].score;
								if (pp.score<0)
									break;

								pp.comboIdx = comboIdx;
								pp.previousCell = DpCellLocation(generalIdx,sources,
									ptmDiagLocation.peakIdx, ptmDiagLocation.aaIdx, formIdx);

								// check that the pointer being added does not equal the previous
								// pointers of one of the pointers already added (if this is the case)
								// then all we are doing is skipping a peak on the same diagonal
								// which is not what we want (we want new added peaks)
								int i;
								for (i=0; i<maxNumForms; i++)
									if (currentDCell.pointers[i].score>=0 &&
										doesPathContainCellLocation(currentDCell.pointers[i].previousCell, pp.previousCell))
											break;
									
								//	if (currentDCell.pointers[i].score>=0 &&
								//		currentDCell.pointers[i].previousCell == pp.previousCell)
								//			break;

								if (i<maxNumForms)
									continue;

								pp.massShift = (peakMass - cutPeaks[ptmDiagLocation.peakIdx].monoMass)
										     - (aaMass - prefixAminoAcidMasses[ptmDiagLocation.aaIdx]);
				
								currentDCell.addPointer(pp,maxNumForms);
							}
						}
					}


					// add pointers from previous A level diag
					// use an arbitrary jump
					if (generalIdx>0)
					{
						const DpCell& prevMCell = M[generalIdx-1][s][peakIdx-1][aaIdx-1];
						int formIdx;
						for (formIdx=0; formIdx<maxNumForms; formIdx++)
						{
							PathPointer pp;
							const PathPointer& prevPointer = prevMCell.pointers[formIdx];

							pp.score = prevPointer.score;
							if (pp.score<0)
								break;

							pp.previousCell = prevPointer.previousCell;

							// check that the pointer being added does not equal the previous
							// pointers of one of the pointers already added (if this is the case)
							// then all we are doing is skipping a peak on the same diagonal
							// which is not waht we want (we want new added peaks)
							int i;
							for (i=0; i<maxNumForms; i++)
								if (currentDCell.pointers[i].score>=0 &&
									doesPathContainCellLocation(currentDCell.pointers[i].previousCell, pp.previousCell))
										break;
								
							//	if (currentDCell.pointers[i].score>=0 &&
							//		currentDCell.pointers[i].previousCell == pp.previousCell)
							//			break;

							if (i<maxNumForms)
								continue;

							pp.massShift = (peakMass - cutPeaks[prevPointer.previousCell.peakIdx].monoMass)
										 - (aaMass - prefixAminoAcidMasses[prevPointer.previousCell.aaIdx]);

							currentDCell.addPointer(pp,maxNumForms);
						}
					}
					
					currentDCell.sortPointers(maxNumForms);

					// add current matched peak's score
					int formIdx;
					for (formIdx=0; formIdx<maxNumForms; formIdx++)
						if (currentDCell.pointers[formIdx].score>=0)
							currentDCell.pointers[formIdx].score += cutPeaks[peakIdx].score;


					// update M					
					DpCell& currentMCell = Map[peakIdx][aaIdx];
					currentMCell.init(maxNumForms);

					// Use the maximum attainable from curret D Cell
					// and consider all possiblites from M[peakIdx-1][aaIdx] and M[peakIdx][aaIdx-1]
					for (formIdx=0; formIdx<maxNumForms; formIdx++)
						if (currentDCell.pointers[formIdx].score>=0)
						{
							// check that the pointer being added does not equal the previous
							// pointers of one of the pointers already added (if this is the case)
							// then all we are doing is skipping a peak on the same diagonal
							// which is not waht we want (we want new added peaks)
							const DpCellLocation& prevPointer = currentDCell.pointers[formIdx].previousCell;
							int i;
							for (i=0; i<maxNumForms; i++)
								if (currentMCell.pointers[i].score>=0 &&
									doesPathContainCellLocation(currentMCell.pointers[i].previousCell,prevPointer))
										break;
							if (i<maxNumForms)
								continue;

							currentMCell.addPointer(currentDCell.pointers[formIdx],maxNumForms);
						}

					// look also at the two adjacent cells:
					// peakIdx-1,aaIdx and peakIdx,aaIdx-1
					int r;
					for (r=0; r<2; r++)
					{
						const DpCell& prevMCell = (r == 0 ? Map[peakIdx-1][aaIdx] :
															Map[peakIdx][aaIdx-1]);
						int formIdx;
						for (formIdx=0; formIdx<maxNumForms; formIdx++)
							if (prevMCell.pointers[formIdx].score>=0)
							{
								// check that the pointer being added does not equal the previous
								// pointers of one of the pointers already added (if this is the case)
								// then all we are doing is skipping a peak on the same diagonal
								// which is not waht we want (we want new added peaks)
								const DpCellLocation& prevPointer = prevMCell.pointers[formIdx].previousCell;
								int i;
								for (i=0; i<maxNumForms; i++)
									if (currentMCell.pointers[i].score>=0 &&
										doesPathContainCellLocation(currentMCell.pointers[i].previousCell,prevPointer))
											break;
								if (i<maxNumForms)
									continue;

								currentMCell.addPointer(prevMCell.pointers[formIdx],maxNumForms);
							}
					}
					currentMCell.sortPointers(maxNumForms);

		
					// allow the first level M to be co-diagonal with everyone
					if (generalIdx ==0 && s==0 && currentMCell.pointers[0].score == NEG_INF)
					{
						currentMCell.pointers[0].score    = 0;
						currentMCell.pointers[0].previousCell = DpCellLocation(0,0,0,0,0);
					}

					// update M's mass shifts
					for (formIdx=0; formIdx<maxNumForms; formIdx++)
						if (currentMCell.pointers[formIdx].score>=0)
						{
							PathPointer& pointer = currentMCell.pointers[formIdx];
							pointer.massShift = (peakMass - cutPeaks[pointer.previousCell.peakIdx].monoMass)
											   - (aaMass - prefixAminoAcidMasses[pointer.previousCell.aaIdx]);
						}
				} 
			}
		

			if (verbose)
			{
				int i_start_idx = 0;
				int j_start_idx = 0;
				int n=7;
				int i,j;
				cout << "D[" <<generalIdx << "][" << s << "]:" << endl << "\t";
				for (j = j_start_idx; j<j_start_idx+n; j++)
					cout << j << "\t";
				cout << setprecision(2) << endl;
				for (i=i_start_idx; i<i_start_idx+n; i++)
				{
					int j;

					int formIdx;
					for (formIdx =0; formIdx<maxNumForms; formIdx++)
					{
						bool had_val=false;
						if (formIdx == 0)
						{
							cout << i << "\t";
						}
						else
							cout << "\t";

						for (j=j_start_idx; j<j_start_idx+n; j++)
						{
							if (D[generalIdx][s][i][j].pointers[formIdx].score<-100)
							{
								cout << setprecision(0);
							}
							else
								cout << setprecision(2);
							
							if (D[generalIdx][s][i][j].pointers[formIdx].score<-100)
							{
								if (formIdx == 0)
								{
									cout << "-\t";
								}
								else
									cout << "\t";
							}
							else
								cout << D[generalIdx][s][i][j].pointers[formIdx].score << "\t";
						}
						cout<< endl;

						cout << "\t";
						for (j=j_start_idx; j<j_start_idx+n; j++)
						{
							if (D[generalIdx][s][i][j].pointers[formIdx].score>-100)
							{
								cout << D[generalIdx][s][i][j].pointers[formIdx].previousCell.specificIdx << 
										D[generalIdx][s][i][j].pointers[formIdx].previousCell.aaIdx << 
										D[generalIdx][s][i][j].pointers[formIdx].previousCell.peakIdx << "\t";
								had_val=true;
							}
							else
								cout << "\t";
						}
						cout<< endl;
						if (! had_val)
							break;
					}
				}

				cout << "M[" <<generalIdx << "][" << s << "]:" << endl << "\t";
				for (j = j_start_idx; j<j_start_idx+n; j++)
					cout << j << "\t";
				cout << setprecision(2) << endl;
				for (i=i_start_idx; i<i_start_idx+n; i++)
				{
					int j;

					int formIdx;
					for (formIdx =0; formIdx<maxNumForms; formIdx++)
					{
						bool had_val=false;
						if (formIdx == 0)
						{
							cout << i << "\t";
						}
						else
							cout << "\t";
				
						for (j=j_start_idx; j<j_start_idx+n; j++)
						{
							if (M[generalIdx][s][i][j].pointers[formIdx].score<-100)
							{
								cout << setprecision(0);
							}
							else
								cout << setprecision(2);
							
							if (M[generalIdx][s][i][j].pointers[formIdx].score>=0)
							{
								cout << M[generalIdx][s][i][j].pointers[formIdx].score << "\t";
							}
							else
							{
								if (formIdx == 0)
								{
									cout << "-\t";
								}
								else
									cout << "\t";
							}
						}
						cout<< endl;

						cout << "\t";
						for (j=j_start_idx; j<j_start_idx+n; j++)
						{
							if (M[generalIdx][s][i][j].pointers[formIdx].score>=0)
							{
								cout << M[generalIdx][s][i][j].pointers[formIdx].previousCell.specificIdx <<  
										M[generalIdx][s][i][j].pointers[formIdx].previousCell.aaIdx <<  
										M[generalIdx][s][i][j].pointers[formIdx].previousCell.peakIdx << "\t";
								had_val=true;
							}
							else
								cout << "\t";
						}
						cout<< endl;
						if (! had_val)
							break;
					}
				}
				cout << endl; 
			}
		}
	}


	if (verbose)
	{
		cout << "Best scores:" << endl;
		for (generalIdx = 0; generalIdx<= maxNumGeneralJumps; generalIdx++)
		{
			int s;
			for (s=0; s<= maxNumSpecificJumps; s++)
			{
				cout << generalIdx <<"," <<s << "  " << D[generalIdx][s][numPeaks-1][numAminoAcids-1].pointers[0].score << endl;
			}
		}
	}

/*	if (verbose)
	{
		int g,s,i,j;
		for (g=0; g<D.size(); g++)
			for (s=0; s<D[g].size(); s++)
				for (i=0; i<D[g][s].size(); i++)
					for (j=0; j<D[g][s][i].size(); j++)
						if (D[g][s][i][j].pointers[0].score>0.0)
							cout << g << "\t" << s << "\t" << i << "\t" << j << "\t == " << D[g][s][i][j].pointers[0].score << endl;
	}*/
}



/*************************************************************************
Prints a table that holds the score for each combination of a number general 
PTMs, specific PTMs and number of forms.
**************************************************************************/
void MonoAlignment::printScoreTable() const
{
	if (D.size()<=0)
	{
		cout << "Error: Array was not initialized!" << endl;
		exit(1);
	}

	const unsigned int maxGeneralPTMs    = D.size();
	const unsigned int maxSpecificPTMs   = D[0].size();
	const unsigned int lastPeakIdx = D[0][0].size()-1;
	const unsigned int lastAAIdx	= D[0][0][0].size()-1;
	const unsigned int numForms		   = D[0][0][0][0].pointers.size();

	if (maxGeneralPTMs>0)
		cout << "\t\t#General PTMs (any mass)" << endl;
	cout << "#Specific";
	
	unsigned int generalIdx;
	for (generalIdx=0; generalIdx<maxGeneralPTMs; generalIdx++)
		cout << "\t" << generalIdx;
	cout << endl;
	cout << "  PTMs" << "\t";
	if (numForms>1)
		cout << " form";

	unsigned int s;
	for (s=0; s<maxSpecificPTMs; s++)
	{
		cout << endl;
		cout << s << "\t";
		if (numForms>1)
			cout << "   1";
		
		for (generalIdx=0; generalIdx<maxGeneralPTMs; generalIdx++)
		{
			float score = D[generalIdx][s][lastPeakIdx][lastAAIdx].pointers[0].score;
			cout << "\t";
			if (score>=0)
			{
				cout << score;
			}
			else 
				cout << " -";
		}
		cout << endl;

		if (numForms>1)
		{
			unsigned int f;
			for (f=1; f<numForms; f++)
			{
				cout << "\t   " << f+1;
				for (generalIdx=0; generalIdx<maxGeneralPTMs; generalIdx++)
				{
					float score = D[generalIdx][s][lastPeakIdx][lastAAIdx].pointers[f].score;
					cout << "\t";
					if (score>=0)
					{
						cout << score;
					}
					else 
						cout << " -";
				}
				cout << endl;
			}
		}
	}	
}



/*************************************************************************
Traces a path that ends at pathEndLoc to see if any of the path's cells are 
the cell given by location. returns true if this happens.
**************************************************************************/
bool MonoAlignment::doesPathContainCellLocation(const DpCellLocation& pathEndLoc,
								 const DpCellLocation& location) const
{
	if (pathEndLoc == location)
		return true;

	DpCellLocation currentLoc = getPreviousLocation(pathEndLoc);

	while (currentLoc.peakIdx>location.peakIdx)
		currentLoc = getPreviousLocation(currentLoc);

	if (currentLoc == location)
		return true;

	return false;
}

/********************************************************************
Traces back the alignment from the cell (generalIdx,specificIdx,aaIdx,formIdx)
to the cell (0,0,0,0,0). Stores it in SimpleAlignmnet& align.
*********************************************************************/
void MonoAlignment::parsePathFromDpTable(
					const vector<CutPeak>& cutPeaks,
					SimpleAlignment& align, 
					int generalIdx, 
					int specificIdx, 
					int peakIdx, 
					int aaIdx,
					int formIdx) const
{
	align.score = D[generalIdx][specificIdx][peakIdx][aaIdx].pointers[formIdx].score;
	align.totalMonoMass = cutPeaks[peakIdx].monoMass;
	align.alignmentBlocks.clear();

	align.pSequence = (ProteinSequence *)&sequence;

	while (peakIdx>0 || aaIdx>0)
	{
		const PathPointer& currPointer = D[generalIdx][specificIdx][peakIdx][aaIdx].pointers[formIdx];
		DpCellLocation previousLocation = currPointer.previousCell;

		AlignmentBlock block;

		block.score       =  currPointer.score;
		block.blockDelta  =  currPointer.massShift;
		block.comboIdx	  =  currPointer.comboIdx;

		block.setEnd(generalIdx,specificIdx,peakIdx,aaIdx,formIdx);
		
		generalIdx     = previousLocation.generalIdx;
		specificIdx    = previousLocation.specificIdx;
		peakIdx        = previousLocation.peakIdx;
		aaIdx          = previousLocation.aaIdx;
		formIdx        = previousLocation.formIdx;
	
		block.setStart(generalIdx,specificIdx,peakIdx,aaIdx,formIdx);

		align.alignmentBlocks.push_back(block);
	}

	reverse(align.alignmentBlocks.begin(),align.alignmentBlocks.end());
}




/*****************************************************************************

******************************************************************************/
score_t MonoAlignment::getBestAlignmentInTable(
				const vector<CutPeak>& cutPeaks,
				SimpleAlignment& align, 
				int generalIdx, 
				int specificIdx, 
				int formIdx,
				int peakIdx, 
				int aaIdx) const
{
	if (generalIdx>=(int)D.size())
	{
		cout << "Error: too many arbitrary jumps: "<< generalIdx << endl;
		end(1);
	}

	if (specificIdx>=(int)D[generalIdx].size())
	{
		cout << "Error: too many PTM jumps: "<< specificIdx << endl;
		end(1);
	}

	if (peakIdx<0)
		peakIdx=cutPeaks.size()-1;

	if (aaIdx<0)
		aaIdx=sequence.getAminoAcids().size();

	if (formIdx>=static_cast<int>(D[0][0][0][0].pointers.size()))
	{
		cout << "Error: requesting formIdx that exceeds table dimensions!" << endl;
		cout << "# path pointers in table: " << D[0][0][0][0].pointers.size() << endl;
		cout << "Path level requested : " << formIdx << endl;
		end(1);
	}

	parsePathFromDpTable(cutPeaks,align,generalIdx,specificIdx,peakIdx,aaIdx,formIdx);

	return align.score;
}






void SimpleAlignment::print(const vector<CutPeak>& cutPeaks, const ProteinSequence& ps,
							ostream& os) const
{
	const vector<int>& aminoAcids = ps.getAminoAcids();
	const Config * config = ps.getConfig();
	vector<mass_t> cutMasses;

	ps.calcNonModifiedCutMasses(cutMasses);
	unsigned int i;
	mass_t totalShift = 0;

	if (alignmentBlocks.size()==0)
	{
		cout << "No alignment found..." << endl;
		return;
	}

	if (alignmentBlocks.size() ==1)
	{
		const AlignmentBlock& block = alignmentBlocks[0];
		cout << "Single block found!" << endl;
		cout << "aas:   " << block.aaIdxStart << " - " << block.aaIdxEnd << endl;
		cout << "peaks: " << block.peakIdxStart << " - " << block.peakIdxEnd << endl;
		return;
	}

	for (i=0; i < alignmentBlocks.size(); i++)
	{
		const AlignmentBlock& block = alignmentBlocks[i];
		os << i << " [" << block.generalIdxStart << " " << block.specificIdxStart << " - " <<
				block.aaIdxEnd << " " << block.specificIdxEnd << "] \tp:"<< block.peakIdxStart << "," << block.peakIdxEnd << "\ta:";
		os << block.aaIdxStart << "," << block.aaIdxEnd <<  "\t(";

		os << setprecision(2) << fixed << cutPeaks[block.peakIdxStart].monoMass << "," << cutPeaks[block.peakIdxEnd].monoMass <<")\t("
			<< cutMasses[block.aaIdxStart] + totalShift << ",";

		totalShift += block.blockDelta;

		os << cutMasses[block.aaIdxEnd] + totalShift << ") del: " << setprecision(2) << block.blockDelta << " (" << 
			totalShift << ")\t" << block.score << endl;
	}

	for (i=0; i < alignmentBlocks.size(); i++)
		cout << alignmentBlocks[i].aaIdxStart << " ";
	cout <<  alignmentBlocks[i-1].aaIdxEnd << endl;

	for (i=0; i < alignmentBlocks.size(); i++)
		cout << alignmentBlocks[i].peakIdxStart << " ";
	cout <<  alignmentBlocks[i-1].peakIdxEnd << endl;

}

/****************************************************************************************
Outputs the alignment by listing the PTMs and then giving an annotated peak list
including the ppm error.
*****************************************************************************************/
void SimpleAlignment::printNice(const vector<CutPeak>& cutPeaks, 
								const ProteinSequence& ps, 
							    const MonoSpectrum& ms, 
								const FragmentSet& fs,
								const AlignmentPtmManager& ptmManager,
								ostream& os) const
{
	const Config * config = ps.getConfig();
	const vector<string>& aa2label = config->get_aa2label();
	const vector<int>& aminoAcids = ps.getAminoAcids();
	const vector<MonoPeak>& peaks = ms.getMonoPeaks();
	vector<mass_t> cutMasses;

	ps.calcNonModifiedCutMasses(cutMasses);
	unsigned int i;
	double totalShift = 0.0;
	double proteinMassDelta = 0.0;
	
	int blockCounter=0;
	for (i=0; i < alignmentBlocks.size(); i++)
	{
		const AlignmentBlock& block = alignmentBlocks[i];
		if (block.blockDelta != 0)
		{
			if (blockCounter == 0)
			{
				os << "Mass shifts:" << endl << endl;
				os << "\tDelta\tPosition \tShift type" << endl;
				os << "\t-----\t-------- \t----------" << endl;
			}
			
			const double delta = (block.comboIdx >= 0 ? ptmManager.getCombo(block.comboIdx).totalMassJump :
			block.blockDelta);
			os << ++blockCounter << "\t" << fixed << setprecision(3) << delta << "\t";
			proteinMassDelta += delta;
			
			ostringstream oss;
			if (block.aaIdxEnd-block.aaIdxStart>1)
			{
				oss << aa2label[aminoAcids[block.aaIdxStart]] << block.aaIdxStart+1 << "-" <<
					aa2label[aminoAcids[block.aaIdxEnd-1]] << block.aaIdxEnd << "\t";
			}
			else
			{
				oss << aa2label[aminoAcids[block.aaIdxStart]] << block.aaIdxStart+1 << "\t";
			}
			string pos = oss.str();
			os << pos;
			if (pos.length()<9)
				os << "\t";

			if (block.comboIdx<0)
			{
				os << "General shift." <<endl;
			}
			else
				os << "Specific: " << ptmManager.getCombo(block.comboIdx).name << endl;
		}
	}

	if (blockCounter==0)
	{
		os << "No mass shifts detected." << endl;
	}

	const double theoreticalProteinMass = ps.calcMass()+ MASS_H2O + proteinMassDelta;
	os << endl;
	os << "Matched peaks:" << endl << endl;
	os << "\t  AA Cut     Mono Mass   Fragment   Theo Mono    Obs Mono     PPM  " << endl;
	os << "\t----------- ------------ -------- ------------ ------------ -------" << endl;
	bool allSpecific = true;
	for (i=0; i < alignmentBlocks.size()-1; i++)
	{
		const AlignmentBlock& block = alignmentBlocks[i];
		
		if (block.blockDelta !=0)
		{
			if (block.comboIdx>=0)
			{
				totalShift += ptmManager.getCombo(block.comboIdx).totalMassJump;
			}
			else
			{
				totalShift += block.blockDelta;
				allSpecific = false;
			}
		}
	
	
		const mass_t shiftedCutMass = cutMasses[block.aaIdxEnd] + totalShift;

		// print cut label and cut mass
		ostringstream cutLabel;
		cutLabel << config->get_aa2label()[aminoAcids[block.aaIdxEnd-1]] << block.aaIdxEnd
			<< (totalShift>=0.0 ? " +" : " -") << fixed << setprecision(1) << totalShift ;
		
		os << i+1 << "\t" <<  setw(11) << left << cutLabel.str() << " ";
		os << fixed << setw(11) << right << setprecision(4) << shiftedCutMass << "  ";

		// print fragment label
		ostringstream fragLabel;
		int frag = cutPeaks[block.peakIdxEnd].fragmentInterpertation;
		if (frag>=0)
		{
			fragLabel << fs.getFragment(frag).getLabel();
		}
		else
			fragLabel << "?";

		if (fs.getFragment(frag).getDirection() == PREFIX)
		{
			fragLabel << " " << block.aaIdxEnd;
		}
		else
			fragLabel << " " << aminoAcids.size() - block.aaIdxEnd;

		os << "  " << setw(6) << left << fragLabel.str() << " ";

		// print theoretical fragment mass, observed and PPM error
		const int orgSpecturmPeakIdx = cutPeaks[block.peakIdxEnd].originalPeakIdx;
		const double theoMass = fs.getFragment(frag).calcExpFragMass(shiftedCutMass, theoreticalProteinMass);
		const double obsMass  = peaks[orgSpecturmPeakIdx].monoMass;
		const double ppm_error = 1000000 * ((obsMass - theoMass) / theoMass);

	
		os << setw(11) << right << setprecision(4) << fixed << theoMass << "  ";
		os << setw(11) << right << setprecision(4) << fixed << obsMass << "  ";
		os << setw(7)  << right << setprecision(2) << ppm_error << endl;
	}
	/*
	os << endl;
	os << "Matched peaks:" << endl << endl;
	os << "\tMonoiso. Mass\ttype\tppm\tposition" << endl;
	os << "\t-------------\t----\t---\t--------" << endl;
	bool allSpecific = true;
	for (i=0; i < alignmentBlocks.size()-1; i++)
	{
		const AlignmentBlock& block = alignmentBlocks[i];
		
		if (block.blockDelta !=0)
		{
			if (block.comboIdx>=0)
			{
				totalShift += ptmManager.getCombo(block.comboIdx).totalMassJump;
			}
			else
			{
				totalShift += block.blockDelta;
				allSpecific = false;
			}
		}
	
		const mass_t mass = cutPeaks[block.peakIdxEnd].monoMass;

		
		os << i+1 << "\t" << fixed << setprecision(3) << mass << "\t";
		if (mass<1000.0)
			os << "\t";

		int orgPeakIdx = cutPeaks[block.peakIdxEnd].originalPeakIdx;

		int frag = cutPeaks[block.peakIdxEnd].fragmentInterpertation;
		if (frag>=0)
		{
			os << fs.getFragment(frag).getLabel();
		}
		else
			os << "_";

		mass_t exp_mass = cutMasses[block.aaIdxEnd] + totalShift;

		double ppm_error = 1000000 * ((cutPeaks[block.peakIdxEnd].monoMass - exp_mass) / exp_mass);

		os << "\t" <<  ppm_error << "\t" << config->get_aa2label()[aminoAcids[block.aaIdxEnd-1]] << block.aaIdxEnd << 
			"+" << setprecision(1) << totalShift << endl;
	}
	*/
	if (allSpecific)
	{
	//	mass_t exactMass = cutMasses[cutMasses.size()-1] + totalShift + MASS_H2O;
		mass_t ppm_error = 1000000 * (theoreticalProteinMass - ms.getParentMonoMass())/theoreticalProteinMass;
		os << endl << "With these specific PTMs, the exact monoisotopic mass should be " << fixed << setprecision(3) << theoreticalProteinMass << endl;
		os << "(ppm error " << setprecision(3) << ppm_error << " compared to mass given in spectrum " << 
			ms.getParentMonoMass() << ")" << endl;
	}
}


void SimpleAlignment::printAlignedMasses(const vector<CutPeak>& cutPeaks, const ProteinSequence& ps) const
{
	const vector<int>& aminoAcids = ps.getAminoAcids();
	const Config * config = ps.getConfig();
	vector<mass_t> cutMasses;

	ps.calcNonModifiedCutMasses(cutMasses);

	unsigned int i;
	for (i=0; i < alignmentBlocks.size()-1; i++)
	{
		const AlignmentBlock& block = alignmentBlocks[i];

		cout << -cutPeaks[block.peakIdxEnd].monoMass << "\t";
		cout << cutMasses[block.aaIdxEnd] << endl;
	}

}


void MonoAlignment::printAllCoDiagPairs(int maxPeakIdx, int maxAaIdx,  const ProteinSequence& ps, 
										const vector<CutPeak>& cutPeaks) const
{
	vector< vector<bool> > indicators;
	vector<mass_t> prefixAminoAcidMasses;
	ps.calcNonModifiedCutMasses(prefixAminoAcidMasses);

	int i;
	indicators.resize(maxPeakIdx+1);
	for (i=0; i<=maxPeakIdx; i++)
		indicators[i].resize(maxAaIdx+1,false);

	for (i=0; i<=maxPeakIdx; i++)
	{
		int j;
		for (j=0; j<=maxAaIdx; j++)
		{
			IdxPair pair = coDiags.getCodiag(i,j);
			if (pair.aaIdx>=0)
			{
				indicators[i][j]=true;
				indicators[pair.peakIdx][pair.aaIdx]=true;
			}
		}
	}

	for (i=0; i<=maxPeakIdx; i++)
	{
		int j;
		for (j=0; j<=maxAaIdx; j++)
		{
			if (indicators[i][j])
			{
				cout << -cutPeaks[i].monoMass << "\t" << prefixAminoAcidMasses[j] << "\t" <<
					i << "\t" << j << endl;
			}
		}
	}

	for (i=0; i<=maxPeakIdx || i<= maxAaIdx; i++)
	{
		if (i<=maxPeakIdx)
		{
			cout << -cutPeaks[i].monoMass << "\t0\t";
		}
		else
			cout << "\t\t0\t";

		if (i<=maxAaIdx)
		{
			cout << prefixAminoAcidMasses[i] << endl;
		}
		else
			cout << endl;
	}
	
}


ostream& operator << (ostream& os, const IdxPair& idxPair)
{
	os << idxPair.peakIdx << "," << idxPair.aaIdx;
	return os;
}


