/***************************************************************************
* Title:          MonoAlignmentPTMs.cpp
* Author:         Ari Frank
* Copyright (c) 2009 The Regents of the University of California
* All Rights Reserved
* See file LICENSE for details.
***************************************************************************/

/******************************************************************************
Copyright 2008, The Regents of the University of California
All Rights Reserved

Permission to use, copy, modify and distribute any part of this
program for educational, research and non-profit purposes, without fee,
and without a written agreement is hereby granted, provided that the
above copyright notice, this paragraph and the following three paragraphs
appear in all copies.

Those desiring to incorporate this work into commercial
products or use for commercial purposes should contact the Technology
Transfer & Intellectual Property Services, University of California,
San Diego, 9500 Gilman Drive, Mail Code 0910, La Jolla, CA 92093-0910,
Ph: (858) 534-5815, FAX: (858) 534-7345, E-MAIL:invent@ucsd.edu.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE, EVEN
IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.

THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.  THE UNIVERSITY OF CALIFORNIA MAKES NO
REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER IMPLIED OR
EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR THAT THE USE OF
THE SOFTWARE WILL NOT INFRINGE ANY PATENT, TRADEMARK OR OTHER RIGHTS.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*******************************************************************************/




#include "MonoAlignment.h"



void AaCombo::bubbleSortAaCombo()
{
	int i;
	const int lastIdx = numAasInCombo-1;
	for (i=0; i<numAasInCombo; i++)
	{
		bool swapped=false;
		int j;
		for (j=0; j<lastIdx; j++)
		{
			if (aminoAcids[j]>aminoAcids[j+1])
			{
				int t;
				t=aminoAcids[j];
				aminoAcids[j]=aminoAcids[j+1];
				aminoAcids[j+1]=t;
				t=counts[j];
				counts[j]=counts[j+1];
				counts[j+1]=t;
				swapped=true;
			}
		}
		if (! swapped)
			return;
	}
}


void PtmCombo::addUnsortedAaCombo(const vector<int>& unsortedAas)
{
	AaCombo aaCombo(unsortedAas);

	size_t i;
	for (i=0; i<satisfyingAaCombos.size(); i++)
		if (satisfyingAaCombos[i]==aaCombo)
			return;

	satisfyingAaCombos.push_back(aaCombo);

}

int PtmCombo::getSatisfieableAaComboIdx(const vector<int>& availableAas) const
{
	size_t i;
	for (i=0; i<satisfyingAaCombos.size(); i++)
	{
		const AaCombo& aaCombo = satisfyingAaCombos[i];
		int j;
		for (j=0; j<aaCombo.numAasInCombo; j++)
			if (aaCombo.counts[j]>availableAas[aaCombo.aminoAcids[j]])
				break;

		if (j==aaCombo.numAasInCombo)
			return static_cast<int>(i);
	}

	return -1;
}

void MonoAlignment::initAlignmentPtms(const vector<PTM>& allPtms, 
									  int maxNumPtmsInCombo)
{
	ptmManager.setPtms(allPtms); 
	
	ptmManager.generatePtmCombos(maxNumPtmsInCombo);

	
//	ptmManager.printAllPtms();
	
//	ptmManager.printAllPtmCombos();	
}





/*****************************************************************
Generates all data for combnations of PTMs
******************************************************************/
void AlignmentPtmManager::generatePtmCombos(int maxNumPtmsInCombo)
{
	ptmCombos.clear();

	if (maxNumPtmsInCombo<1)
		return;

	if (maxNumPtmsInCombo> MAX_NUM_PTMS_IN_COMBO-1)
	{
		cout << "Error: must increase MAX_NUM_PTMS_IN_COMBO to at least " << maxNumPtmsInCombo+1 <<endl;
		end(1);
	}

	const int numPtms = allPtms.size();
	int ptmIdx;
	for (ptmIdx = 0; ptmIdx<numPtms ; ptmIdx++)
	{
		if (allPtms[ptmIdx].type == PTM_FIXED)
			continue;

		PtmCombo combo;
		combo.numPtmsInCombo=1;
		combo.ptmIdxs[0] = ptmIdx;
		combo.totalMassJump = allPtms[ptmIdx].delta;
		combo.name = allPtms[ptmIdx].name;

	//	cout << ptmIdx << ">> :" << combo << endl;

		ptmCombos.push_back(combo);
	}

	int startComboIdx = 0;
	int numPtmsInCombo;
	for (numPtmsInCombo=2; numPtmsInCombo<=maxNumPtmsInCombo; numPtmsInCombo++)
	{
		const int currentNumCombos = ptmCombos.size();
		const int lastPtmIdxPos = numPtmsInCombo -2;
		int c;
		for (c=startComboIdx; c<currentNumCombos; c++)
		{
			const PtmCombo existingCombo = ptmCombos[c];
			const int startPtmIdx = existingCombo.ptmIdxs[lastPtmIdxPos];

			int ptmIdx;
			for (ptmIdx = startPtmIdx; ptmIdx<numPtms; ptmIdx++)
			{
				if (allPtms[ptmIdx].type == PTM_FIXED)
					continue;

				PtmCombo newCombo = existingCombo;
				newCombo.ptmIdxs[numPtmsInCombo-1]=ptmIdx;
				newCombo.totalMassJump += allPtms[ptmIdx].delta;
				newCombo.numPtmsInCombo++;
				newCombo.name = newCombo.name + " + " + allPtms[ptmIdx].name;

				int i;
				int numTerminalPtms = 0;
				for (i=0; i<numPtmsInCombo; i++)
				{
					const int region = allPtms[newCombo.ptmIdxs[i]].region;
					if (region == PTM_N_TERMINAL || region == PTM_C_TERMINAL)
						numTerminalPtms++;
				}

				if (numTerminalPtms>1)
					continue;

				ptmCombos.push_back(newCombo);
			}
		}
		startComboIdx = currentNumCombos;
	}

	// create lists of satisfying amino acids for each combo
	size_t comboIdx;
	for (comboIdx=0; comboIdx<ptmCombos.size(); comboIdx++)
	{
		PtmCombo& combo = ptmCombos[comboIdx];
		vector<int> aaSizes;
		vector< vector<int> > aaLists,idxs;

		aaSizes.resize(combo.numPtmsInCombo);
		aaLists.resize(combo.numPtmsInCombo);
		int i;
		for (i=0; i<combo.numPtmsInCombo; i++)
		{
			const PTM& ptm = allPtms[combo.ptmIdxs[i]];
			aaSizes[i]=ptm.applicableAAs.size();
			aaLists[i]=ptm.applicableAAs;
		}
	
		combo.satisfyingAaCombos.clear();
		generateAllCrossProducts(aaSizes,idxs);

		vector<int> unsortedComboAas;
		unsortedComboAas.resize(combo.numPtmsInCombo);
		size_t k;
		for (k=0; k<idxs.size(); k++)
		{
			const vector<int>& aaIdxs = idxs[k];
			size_t j;
			for (j=0; j<aaIdxs.size(); j++)
				unsortedComboAas[j]=aaLists[j][aaIdxs[j]];
			combo.addUnsortedAaCombo(unsortedComboAas);
		}
	}

	// add dummy PTM combos
	PtmCombo dummyBot,dummyTop;
	dummyBot.totalMassJump = -1E9;
	dummyTop.totalMassJump =  1E9;

	ptmCombos.push_back(dummyBot);
	ptmCombos.push_back(dummyTop);
	sort(ptmCombos.begin(),ptmCombos.end());

//	cout << "Created " << ptmCombos.size() << " ptm combos\n";

//	this->printAllPtmCombos();
//	end(0);
}






bool operator< (mass_t mass_a, const PtmCombo& b)
{
	return (mass_a<b.totalMassJump);
}

int	AlignmentPtmManager::getNumPtmCombosInRange(mass_t minMass, mass_t maxMass, int* firstPtmIdx) const
{
	*firstPtmIdx=-1;

	if (minMass < -5E8)
		minMass = -5E8;
	if (maxMass >  5E8)
		maxMass =  5E8;

	vector<PtmCombo>::const_iterator location = lower_bound(ptmCombos.begin(),
		ptmCombos.end(),minMass);

	while ((*location).totalMassJump < maxMass)
		++location;

	while ((*location).totalMassJump > maxMass)
		--location;
	
	if ((*location).totalMassJump < minMass)
		return 0;

	int topIdx = location - ptmCombos.begin();

	while ((*location).totalMassJump >= minMass)
		--location;

	++location;

	int botIdx = location - ptmCombos.begin();

	*firstPtmIdx = botIdx; // 

	return (topIdx-botIdx+1);
}


void MonoAlignment::initAaCounts()
{
	const vector<int>& aminoAcids = sequence.getAminoAcids();
	
	if (aminoAcids.size()==0)
	{
		cout << "Error: no sequence set for MonoAlignment!" << endl;
		end(1);
	}

	aaCounts.resize(aminoAcids.size()+1);
	aaCounts[0].resize(NUM_REGULAR_AAS);
	unsigned int i;
	for (i=0; i<NUM_REGULAR_AAS; i++)
		aaCounts[0][i]=0;

	aaCounts[0][N_TERM]=1;

	for (i=1; i<=aminoAcids.size(); i++)
	{
		aaCounts[i]=aaCounts[i-1];
		aaCounts[i][aminoAcids[i-1]]++;
//		cout << i << " >> " << aminoAcids[i-1] << endl;
	}

	aaCounts[aminoAcids.size()][C_TERM]=1;

}





/********************************************************************
Checks if the amino acids between start idx and end idx are enough
to statisfy the constraints given by the ptms in the combo.
*********************************************************************/
bool MonoAlignment::checkIfComboSatisfiable(int comboIdx, 
											int startAaIdx, 
											int endAaIdx) const
{
	vector<int> availableAaCounts;
	availableAaCounts = aaCounts[endAaIdx];
	
	if (startAaIdx>0)
	{
		const vector<int>& countsBefore = aaCounts[startAaIdx];
		unsigned int i;

		for (i=0; i<countsBefore.size(); i++)
			availableAaCounts[i] -= countsBefore[i];
	}

	const PtmCombo& currentCombo = ptmManager.getCombo(comboIdx);
	return (currentCombo.getSatisfieableAaComboIdx(availableAaCounts)>=0);
}





ostream& operator << (ostream& os, const PtmCombo& ptm)
{
	os << ptm.numPtmsInCombo << ": ";
	int i;
	for (i=0; i<ptm.numPtmsInCombo; i++)
		os << ptm.ptmIdxs[i] << " ";
	os << " mass: " << setprecision(3) << fixed << ptm.totalMassJump << " (aa combos " <<
		ptm.satisfyingAaCombos.size() << ")";

	return os;
}

void AlignmentPtmManager::printAllPtmCombos() const
{
	unsigned int i;
	for (i=0; i<this->ptmCombos.size(); i++)
		cout << i << " " << ptmCombos[i] << endl;
}

void AlignmentPtmManager::printAllPtms()      const
{
	unsigned int i;
	for (i=0; i< allPtms.size(); i++)
		cout << i << " " << allPtms[i] << " " << endl;
}
