/*
 * Decompiled with CFR 0.152.
 */
package maizeUtils;

import basicUtils.InspectAnnotation;
import basicUtils.Utils;
import errorUtils.ErrorThrower;
import java.io.BufferedReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Random;
import ms2dbUtils.MS2DB;
import proteogenomicUtils.GenomicLocator;
import trieUtils.TrieDB;

public class DecoyPeptideGenerator {
    public static String usageInfo = "maizeUtils.DecoyPeptideGenerator version 2012.02.14\nDecoyPeptideGenerator is used to generate decoy peptides using the genome of the species of interest.  This should be\nused to evaluate the specificity of event identification.[Required]\n -g [Dir] Directory containing trie or FASTA files of genome.\n OR \n -q [DIR] Directory containing genomic DBs (trie's and ms2db's)\n -p [File] Trie file containing the known proteome (so we don't accidentally choose a known peptide)\n -n [Dir] Directory containing novel peptides (so we don't accidentally choose a novel peptide)\n -o [File] Inspect-like output file of decoy peptides\n -s [Num] Number of peptides to generate\n[Optional]\n -i [File] Recompute localFDRs for a file already containing decoy peptides\n -b File specified with -i is a location file, not a Inspect-like file\n\n -c Construct decoy peptides from novel peptides (keep the novel peptide lFDR, length, and location count and choose a random genomic location)\n -l Write the output as locations instead of Inspect-like (this is used when you don't need all the locations for the peptide)\n\n -m [NUM] Max locations a peptide can appear in the genomic DB\n";
    public static double[] sizeDist = new double[]{1.58605854538606E-4, 3.96514636346514E-5, 2.27995915899246E-4, 8.42593602236343E-4, 0.0146611286789124, 0.0648004044449291, 0.0977012063957811, 0.103718316002339, 0.0979192894457717, 0.086707838103074, 0.0748024861467699, 0.0631152172404564, 0.0515667284568642, 0.0469770715411533, 0.039750592293738, 0.0342886031780648, 0.030432498339595, 0.0271513397238276, 0.0238305296444255, 0.0211738815809039, 0.0165643989333756, 0.0143736555675611, 0.0132832403176082, 0.0116773560404048, 0.0107257209131732, 0.00912974950187849, 0.00789064126329563, 0.0066713587565301, 0.00596754527701504, 0.00489695575887945, 0.00399488496119113, 0.00323159428622409, 0.00297385977259886, 0.00177440299765065, 0.00182396732719397, 0.00134814976357815, 9.81373724957623E-4, 7.33552077241051E-4, 5.65033356793783E-4, 3.96514636346514E-4, 2.87473111351223E-4, 2.67647379533897E-4, 1.88344452264594E-4, 2.0817018408192E-4, 1.78431586355931E-4};
    public static double[] localFDRDist = new double[]{0.0265245091666, 0.0278247302041, 0.0286048628267, 0.0259394096996, 0.0264594981147, 0.0265245091666, 0.0249642439215, 0.0207385255493, 0.0180730724223, 0.0228838902613, 0.0265245091666, 0.0213236250163, 0.0221037576388, 0.0258093875959, 0.0215836692238, 0.0189182160967, 0.0169678845404, 0.0104667793525, 0.0148875308802, 0.0120270445976, 0.0140423872058, 0.0, 0.0131972435314, 0.0215186581719, 0.0169678845404, 0.0101417240931, 0.00929658041867, 0.0115719672344, 0.00955662462619, 0.00884150305552, 0.00994669093746, 0.0, 0.0124171109089, 0.00572097256534, 0.00929658041867, 0.00942660252243, 0.00773631517358, 0.00942660252243, 0.00689117149915, 0.0, 0.00682616044728, 0.00864646989988, 0.00650110518788, 0.00949161357431, 0.00624106098037, 0.0, 0.0067611493954, 0.00689117149915, 0.00682616044728, 0.00773631517358, 0.0076713041217, 0.0, 0.00741125991419, 0.00780132622546, 0.00357560785334, 0.00500585099467, 0.0, 0.00416070732024, 0.00312053049018, 0.006436094136, 0.00507086204655, 0.0, 0.00208035366012, 0.00520088415031, 0.00585099466909, 0.00344558574958, 0.0, 0.00448576257964, 0.00234039786764, 0.00507086204655, 0.0, 0.00325055259394, 0.00318554154206, 0.00351059680146, 0.0, 0.00130022103758, 0.0012352099857, 0.00364061890521, 0.0, 0.00500585099467, 0.00208035366012, 0.00409569626837, 0.0, 0.0030555194383, 0.00396567416461, 0.00240540891952, 0.0, 0.00396567416461, 0.00390066311273, 0.0, 0.00273046417891, 0.00364061890521, 0.00357560785334, 0.0, 0.00325055259394, 0.00390066311273, 0.0, 0.00195033155636, 0.00169028734885, 0.00409569626837, 0.0, 0.00279547523079, 0.00403068521649, 0.0, 0.00409569626837, 0.00253543102327, 0.0, 0.00325055259394, 0.00201534260824, 0.0, 0.00188532050449, 0.00182030945261, 0.0, 0.00188532050449, 0.00325055259394, 0.0, 0.00260044207515, 0.00201534260824, 0.0, 9.75165778182E-4, 0.00104017683006, 0.0, 0.00234039786764, 0.0, 0.00201534260824, 0.00195033155636, 0.0, 0.00149525419321, 8.45143674425E-4, 0.0, 0.0012352099857, 0.00403068521649, 0.0, 9.75165778182E-4, 0.0, 4.55077363152E-4, 0.00201534260824, 0.0, 0.0024704199714, 0.0, 7.15121570667E-4, 0.00130022103758, 0.0, 0.00104017683006, 0.00130022103758, 0.0, 6.50110518788E-4, 0.0, 6.50110518788E-4, 0.0, 0.00318554154206, 0.00149525419321, 0.0, 0.00227538681576, 0.0, 0.00169028734885, 7.15121570667E-4, 0.0, 0.00117019893382, 0.0, 0.00162527629697, 0.0, 0.00195033155636, 5.20088415031E-4, 0.0, 5.85099466909E-4, 0.0, 7.80132622546E-4, 0.0, 0.00143024314133, 0.0, 8.45143674425E-4, 7.15121570667E-4, 0.0, 0.0, 0.0, 0.00130022103758, 0.0, 7.15121570667E-4, 0.0, 5.85099466909E-4, 0.0, 7.80132622546E-4, 0.00253543102327, 0.0, 0.00117019893382, 0.0, 7.15121570667E-4, 0.0, 0.00188532050449, 0.0, 0.0, 0.0, 6.50110518788E-4, 0.0, 9.75165778182E-4, 0.0, 8.45143674425E-4, 0.0, 0.00169028734885, 0.0, 3.25055259394E-4, 0.0, 0.0024704199714, 0.0, 0.00266545312703, 0.0, 8.45143674425E-4, 0.0, 0.00175529840073, 0.0, 0.0012352099857, 0.0, 0.00201534260824, 0.0, 0.0, 0.0, 0.0012352099857, 0.0, 0.0, 0.0, 4.55077363152E-4, 0.0, 0.00299050838643, 0.0, 0.00351059680146, 0.0, 0.00227538681576, 0.0, 7.15121570667E-4, 0.0, 0.0, 0.00149525419321, 0.0, 0.00130022103758, 0.0, 0.0, 0.0, 1.30022103758E-4, 0.0, 1.95033155636E-4, 0.0, 0.0, 3.25055259394E-4, 0.0, 0.00130022103758, 0.0, 0.0, 0.0, 4.55077363152E-4, 0.0, 0.0, 6.50110518788E-5, 0.0, 0.0, 0.0, 1.95033155636E-4, 0.0, 0.00117019893382, 0.0, 0.0, 6.50110518788E-5, 0.0, 3.90066311273E-4, 0.0, 2.60044207515E-4, 0.0, 0.0, 0.00149525419321, 0.0, 3.90066311273E-4, 0.0, 5.85099466909E-4, 0.0, 0.0, 6.50110518788E-4, 0.0, 6.50110518788E-5, 0.0, 0.0, 0.0, 0.0, 9.10154726303E-4, 0.0, 0.0, 0.0, 0.0, 5.85099466909E-4, 0.0, 0.0, 0.00117019893382, 0.0, 5.20088415031E-4, 0.0, 0.0, 5.85099466909E-4, 0.0, 7.80132622546E-4, 0.0, 0.0, 0.00104017683006, 0.0, 0.0, 0.0, 0.0, 0.00104017683006, 0.0, 0.0, 0.0, 0.0, 1.95033155636E-4, 0.0, 0.0, 7.80132622546E-4, 0.0, 3.25055259394E-4, 0.0, 0.0, 3.25055259394E-4, 0.0, 0.0, 1.30022103758E-4, 0.0, 0.0, 4.55077363152E-4, 0.0, 3.90066311273E-4, 0.0, 0.0, 7.15121570667E-4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.00110518788194, 0.0, 0.0, 0.0, 0.0, 1.95033155636E-4, 0.0, 0.0, 4.55077363152E-4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.00110518788194, 0.0, 0.0, 0.0, 0.0, 0.0, 4.55077363152E-4, 0.0, 0.0, 1.95033155636E-4, 0.0, 0.0, 0.00110518788194, 0.0, 0.0, 3.25055259394E-4, 0.0, 0.0, 2.60044207515E-4, 0.0, 0.0, 0.0, 0.0, 0.0, 3.25055259394E-4, 0.0, 0.0, 5.85099466909E-4, 0.0, 0.0, 3.25055259394E-4, 0.0, 0.0, 5.85099466909E-4, 0.0, 0.0, 3.90066311273E-4, 0.0, 0.0, 1.95033155636E-4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.90066311273E-4, 0.0, 0.0, 9.10154726303E-4, 0.0, 0.0, 3.90066311273E-4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.85099466909E-4, 0.0, 0.0, 5.20088415031E-4, 0.0, 0.0, 1.95033155636E-4, 0.0, 0.0, 0.0, 5.85099466909E-4, 0.0, 0.0, 0.0012352099857, 0.0, 0.0, 6.50110518788E-5, 0.0, 0.0, 0.0, 4.55077363152E-4, 0.0, 0.0, 3.25055259394E-4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.00117019893382, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 7.80132622546E-4, 0.0, 0.0, 0.0, 8.45143674425E-4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.00117019893382, 0.0, 0.0, 6.50110518788E-5, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.50110518788E-5, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0012352099857, 0.0, 0.0, 0.0, 3.90066311273E-4, 0.0, 0.0, 0.0, 0.00630607203225};
    private String genomicDBDir = null;
    private TrieDB[] TrieFiles = null;
    private MS2DB[] MS2DBFiles = null;
    private TrieDB proteomeTrie;
    private String novelPeptideDir;
    private String outputFileName;
    private int numPeps;
    private Random randomGen;
    private HashSet novelPeptideSet;
    private TrieDB[] genomicTrieDBs;
    private long[] dbLengths;
    private long totalGenomeLength;
    private boolean debugFlag;
    private String inputPepFileName = null;
    private boolean buildFromNovel = false;
    private Hashtable novelPeptideInfo;
    private boolean writeLocations = false;
    private boolean inputIsLocations = false;
    private int maxLocations = Integer.MAX_VALUE;
    private String searchDBDir = null;
    private int maxLocs;
    private int minUnique;

    public DecoyPeptideGenerator(String genomicDBDir, String proteomeFile, String novelDir, String outputFileName, int numPeps, String pepFileName, boolean buildFromNovel, boolean writeLocations, boolean inputIsLocations, int maxLocations, String searchDBDir) {
        if (genomicDBDir != null && !Utils.IsDir(genomicDBDir)) {
            ErrorThrower.ThrowError(1, genomicDBDir);
        } else if (genomicDBDir != null) {
            this.genomicDBDir = genomicDBDir;
            searchDBDir = null;
        }
        if (searchDBDir != null && !Utils.IsDir(searchDBDir)) {
            ErrorThrower.ThrowError(1, searchDBDir);
        } else if (searchDBDir != null) {
            this.searchDBDir = searchDBDir;
        }
        if (!Utils.IsFile(proteomeFile)) {
            ErrorThrower.ThrowError(1, proteomeFile);
        }
        this.proteomeTrie = new TrieDB(proteomeFile);
        if (!Utils.IsDir(novelDir)) {
            ErrorThrower.ThrowError(1, novelDir);
        }
        this.novelPeptideDir = novelDir;
        this.outputFileName = outputFileName;
        if (numPeps <= 0) {
            ErrorThrower.ThrowError(4, "Num peptides must be greater than 0");
        }
        this.numPeps = numPeps;
        this.randomGen = new Random();
        if (pepFileName != null && !Utils.IsFile(pepFileName)) {
            ErrorThrower.ThrowError(1, pepFileName);
        }
        if (pepFileName != null) {
            this.inputPepFileName = pepFileName;
        }
        this.buildFromNovel = buildFromNovel;
        this.writeLocations = writeLocations;
        this.inputIsLocations = inputIsLocations;
        this.maxLocations = maxLocations;
        if (searchDBDir != null && !Utils.IsDir(searchDBDir)) {
            ErrorThrower.ThrowError(1, searchDBDir);
        }
        if (this.inputPepFileName != null) {
            System.out.println("[RUN MODE] Updating the localFDRs in " + this.inputPepFileName + ".  No new peptides will be generated");
            this.writeLocations = this.inputIsLocations;
        } else {
            System.out.println("[RUN MODE] Generating new peptides");
            if (this.genomicDBDir != null) {
                System.out.println("[SEQ MODE] Generating peptide sequences from the genome");
                System.out.println("[TEST MODE] Decoy location count is not limited");
                this.maxLocations = Integer.MAX_VALUE;
                this.writeLocations = false;
            } else if (this.searchDBDir != null) {
                System.out.println("[SEQ MODE] Generating peptide sequences from search databases");
                System.out.println("[TEST MODE] Decoy location count limited to " + this.maxLocations + " by searching databases in " + this.searchDBDir);
            }
            if (this.buildFromNovel) {
                System.out.println("[ATTRIBUTE MODE] Generated decoy peptide attributes are drawn from novel peptides (localFDR, location count, length)");
            } else {
                System.out.println("[ATTRIBUTE MODE] Generated decoy peptide attributes are drawn from internal distributions");
            }
        }
        if (this.writeLocations) {
            System.out.println("[OUTPUT MODE] Writing output file as locations");
        } else {
            System.out.println("[OUTPUT MODE] Writing output file as Inspect-like");
        }
        if (this.maxLocs < 0) {
            ErrorThrower.ThrowError(4, "Maximum locations  must be non-negative");
        }
        this.maxLocs = this.maxLocs;
        if (this.minUnique < 0 || this.minUnique > this.numPeps) {
            ErrorThrower.ThrowError(4, "Minimum unique peptides must be non-negative, and less than total peptides");
        }
        this.minUnique = this.minUnique;
    }

    private void initializeDBs(String dbDir) {
        String Ext;
        int MS2DBCount = 0;
        int TrieCount = 0;
        String[] DBs = Utils.ListDir(dbDir);
        int i = 0;
        while (i < DBs.length) {
            Ext = Utils.GetFileExtension(DBs[i]);
            if (Ext.indexOf("trie") >= 0) {
                ++TrieCount;
            } else if (Ext.indexOf("ms2db") >= 0) {
                ++MS2DBCount;
            } else {
                System.out.println("WARNING: DB " + DBs[i] + " is not a valid format, we skip it!");
            }
            ++i;
        }
        this.TrieFiles = new TrieDB[TrieCount];
        this.MS2DBFiles = new MS2DB[MS2DBCount];
        TrieCount = 0;
        MS2DBCount = 0;
        i = 0;
        while (i < DBs.length) {
            Ext = Utils.GetFileExtension(DBs[i]);
            if (Ext.indexOf("trie") >= 0) {
                this.TrieFiles[TrieCount] = new TrieDB(DBs[i]);
                ++TrieCount;
            } else if (Ext.indexOf("ms2db") >= 0) {
                this.MS2DBFiles[MS2DBCount] = new MS2DB(DBs[i]);
                ++MS2DBCount;
            }
            ++i;
        }
    }

    public int getPeptideLength() {
        double myDoub = this.randomGen.nextDouble();
        double cum = 0.0;
        int i = 0;
        while (i < sizeDist.length) {
            if (myDoub <= (cum += sizeDist[i])) {
                return i + 5;
            }
            ++i;
        }
        return sizeDist.length - 1 + 5;
    }

    public double generateLocalFDR() {
        double myDoub = this.randomGen.nextDouble();
        double cum = 0.0;
        int i = 0;
        while (i < localFDRDist.length) {
            if (myDoub <= (cum += localFDRDist[i])) {
                return (double)(i + 1) / 500.0;
            }
            ++i;
        }
        return 1.0;
    }

    private void generatePeptides() {
        if (this.inputPepFileName != null && !this.inputIsLocations) {
            InspectAnnotation[] anns = InspectAnnotation.LoadInspectResultsFile(this.inputPepFileName);
            FileWriter f = null;
            if (this.debugFlag) {
                System.out.println("Updated localFDR " + anns.length + " to " + this.outputFileName);
            }
            try {
                f = new FileWriter(this.outputFileName);
                f.write("#" + Utils.JoinStringArray(anns[0].columnHeaders, "\t") + "\n");
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(5, this.outputFileName);
            }
            int i = 0;
            while (i < anns.length) {
                anns[i].LocalFDR = this.generateLocalFDR();
                try {
                    f.write(String.valueOf(anns[i].toString()) + "\n");
                }
                catch (IOException E) {
                    ErrorThrower.ThrowError(7, this.outputFileName);
                }
                ++i;
            }
            try {
                f.close();
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(7, this.outputFileName);
            }
            return;
        }
        if (this.inputPepFileName != null && this.inputIsLocations) {
            String[] anns = GenomicLocator.GenomicColumns.LoadAllPeptides(this.inputPepFileName);
            double[] newLocalFDRs = new double[anns.length];
            int i = 0;
            while (i < anns.length) {
                newLocalFDRs[i] = this.generateLocalFDR();
                ++i;
            }
            FileWriter f = Utils.openFileWriter(this.outputFileName);
            if (this.debugFlag) {
                System.out.println("Updated localFDR " + anns.length + " to " + this.outputFileName);
            }
            BufferedReader buf = Utils.openBufferedReader(this.inputPepFileName);
            String line = Utils.readNextLine(buf, this.inputPepFileName);
            while (line != null) {
                if ((line = line.trim()).length() == 0) {
                    line = Utils.readNextLine(buf, this.inputPepFileName);
                    continue;
                }
                if (line.charAt(0) == '#') {
                    Utils.writeLine(f, this.outputFileName, String.valueOf(line) + "\n");
                    line = Utils.readNextLine(buf, this.inputPepFileName);
                    continue;
                }
                Object[] loc = GenomicLocator.GenomicColumns.LoadFromString(line);
                String pep = Utils.RemoveFlankingAAs((String)loc[GenomicLocator.GenomicColumns.Peptide]);
                int index = Utils.FindStringInArray(anns, pep);
                if (index < 0) {
                    ErrorThrower.ThrowErrorCustum(100, "Unable to find peptide '" + pep + "' in loaded peptides!");
                }
                loc[GenomicLocator.GenomicColumns.BestLocalFDR] = new Double(newLocalFDRs[index]);
                if (this.debugFlag) {
                    System.out.println("Peptide " + pep + " appears at position " + index + " and has lFDR " + newLocalFDRs[index]);
                }
                String newLine = GenomicLocator.GenomicColumns.toString(loc, "\t");
                Utils.writeLine(f, this.outputFileName, String.valueOf(newLine) + "\n");
                line = Utils.readNextLine(buf, this.inputPepFileName);
            }
            try {
                f.close();
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(7, this.outputFileName);
            }
            return;
        }
        this.initializeNovelPeptides();
        this.initializeGenomicDBs();
        Object peptides = null;
        if (this.buildFromNovel) {
            Utils.ConvertHashSetToStringArray(this.novelPeptideSet);
            System.out.println(String.valueOf((peptides).length) + " = " + this.novelPeptideSet.size() + " = " + this.novelPeptideInfo.size());
        }
        int pepsGenerated = 0;
        String[] finalPeps = new String[this.numPeps];
        double[] finalPepLFDR = new double[this.numPeps];
        int[] finalPepLocCount = new int[this.numPeps];
        String[] finalPepLocation = new String[this.numPeps];
        boolean startToSearch = false;
        int endToSearch = 0;
        HashSet<String> finalPepSet = new HashSet<String>();
        ArrayList currPeptides = new ArrayList();
        int uniquePeptides = 0;
        Hashtable locs = new Hashtable();
        while (finalPepSet.size() < this.numPeps && uniquePeptides < this.minUnique) {
            if (pepsGenerated % 100 == 0) {
                System.out.println("Generated " + pepsGenerated + "/" + this.numPeps + " peptides (" + uniquePeptides + " unique, " + currPeptides.size() + " in the pipe)");
            }
            int length = -1;
            int pepIndex = -1;
            if (this.buildFromNovel) {
                pepIndex = this.randomGen.nextInt((peptides).length);
                length = peptides[pepIndex].length();
                if (this.debugFlag) {
                    System.out.println(" - basing the peptide on " + (String)peptides[pepIndex]);
                }
            } else {
                if (this.debugFlag) {
                    System.out.println(" - generating peptide length from random");
                }
                length = this.getPeptideLength();
            }
            int Pos = this.generateRandomPos();
            boolean strand = this.randomGen.nextBoolean();
            String strandStr = strand ? "+" : "-";
            String pepLocation = "FakeChr_0_0_0";
            long[] dbInfo = this.getDBIndex(Pos);
            String peptide = this.getPeptide((int)dbInfo[0], dbInfo[1], length, strand);
            if (this.writeLocations) {
                pepLocation = String.valueOf(this.genomicTrieDBs[(int)dbInfo[0]].getProteinName((int)dbInfo[1])) + "_" + dbInfo[1] + "_" + (dbInfo[1] + (long)length) + "_" + strandStr;
                if (this.debugFlag) {
                    System.out.println("Peptide location: " + pepLocation);
                }
            }
            if (this.debugFlag) {
                System.out.println("[" + pepsGenerated + "] : pos=" + Pos + ", length=" + length + ", peptide=" + peptide + ", forwardFlag=" + strand);
            }
            if (this.containedInKnownProtein(peptide)) {
                if (!this.debugFlag) continue;
                System.out.println(" - Found in a known protein!");
                continue;
            }
            if (this.containedInNovelPeptides(peptide)) {
                if (!this.debugFlag) continue;
                System.out.println(" - Found as a novel peptide already!");
                continue;
            }
            if (peptide.indexOf(88) >= 0) {
                if (!this.debugFlag) continue;
                System.out.println(" - Found a stop codon!");
                continue;
            }
            if (finalPepSet.contains(peptide)) continue;
            double localFDR = 0.0;
            int locCount = -1;
            if (this.buildFromNovel) {
                if (!this.novelPeptideInfo.containsKey(peptides[pepIndex])) {
                    ErrorThrower.ThrowError(13, "Cannot find '" + (String)peptides[pepIndex] + "' in novelPeptideInfo");
                }
                double[] info = (double[])this.novelPeptideInfo.get(peptides[pepIndex]);
                localFDR = info[0];
                locCount = (int)info[1];
            } else {
                localFDR = this.generateLocalFDR();
            }
            finalPepSet.add(peptide);
            finalPeps[pepsGenerated] = peptide;
            finalPepLFDR[pepsGenerated] = localFDR;
            finalPepLocCount[pepsGenerated] = locCount;
            finalPepLocation[pepsGenerated] = pepLocation;
            ++pepsGenerated;
            ++endToSearch;
        }
        if (this.writeLocations) {
            this.writeFinalPepLocations(finalPeps, finalPepLFDR, finalPepLocCount, finalPepLocation);
        } else {
            this.writeFinalPeps(finalPeps, finalPepLFDR, finalPepLocCount);
        }
    }

    private void writeFinalPepLocations(String[] finalPeps, double[] finalPepLFDR, int[] finalPepLocCount, String[] finalPepLocation) {
        FileWriter f = null;
        if (this.debugFlag) {
            System.out.println("Writing " + finalPeps.length + " to " + this.outputFileName);
        }
        String[] columns = GenomicLocator.GenomicColumns.GenomicHeader.trim().substring(1).split("\t");
        try {
            f = new FileWriter(this.outputFileName);
            f.write("#" + Utils.JoinStringArray(columns, "\t") + "\n");
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(5, this.outputFileName);
        }
        int i = 0;
        while (i < finalPeps.length) {
            String[] locBits = finalPepLocation[i].split("_");
            String outLine = "";
            outLine = String.valueOf(outLine) + "FakeFile";
            outLine = String.valueOf(outLine) + "\t-1";
            outLine = String.valueOf(outLine) + "\t" + finalPeps[i];
            outLine = String.valueOf(outLine) + "\t-1";
            outLine = String.valueOf(outLine) + "\t-1";
            outLine = String.valueOf(outLine) + "\t-1";
            outLine = String.valueOf(outLine) + "\t0.0";
            outLine = String.valueOf(outLine) + "\t0.0";
            outLine = String.valueOf(outLine) + "\tFakeProtein";
            outLine = String.valueOf(outLine) + "\tFakeDB";
            outLine = String.valueOf(outLine) + "\t" + locBits[0];
            outLine = String.valueOf(outLine) + "\t" + locBits[1];
            outLine = String.valueOf(outLine) + "\t" + locBits[2];
            outLine = String.valueOf(outLine) + "\t" + locBits[3];
            outLine = String.valueOf(outLine) + "\t" + finalPeps[i];
            outLine = String.valueOf(outLine) + "\t";
            outLine = String.valueOf(outLine) + "\t0";
            outLine = finalPepLocCount[i] == 1 ? String.valueOf(outLine) + "\ttrue" : String.valueOf(outLine) + "\tfalse";
            outLine = String.valueOf(outLine) + "\ttrue";
            outLine = String.valueOf(outLine) + "\t0";
            outLine = String.valueOf(outLine) + "\t" + finalPepLocCount[i];
            outLine = String.valueOf(outLine) + "\t" + finalPepLFDR[i];
            outLine = String.valueOf(outLine) + "\n";
            try {
                f.write(outLine);
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(7, this.outputFileName);
            }
            ++i;
        }
        try {
            f.close();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(8, this.outputFileName);
        }
    }

    private void writeFinalPeps(String[] finalPeps, double[] finalPepLFDR, int[] finalPepLocCount) {
        String[] columns;
        FileWriter f = null;
        if (this.debugFlag) {
            System.out.println("Writing " + finalPeps.length + " to " + this.outputFileName);
        }
        if (Utils.FindStringInArray(columns = InspectAnnotation.HeaderLineDefault.trim().substring(1).split("\t"), "LocalFDR") < 0) {
            columns = Utils.appendToArray(columns, "LocalFDR");
        }
        if (Utils.FindStringInArray(columns, "IsNovel") < 0) {
            columns = Utils.appendToArray(columns, "IsNovel");
        }
        try {
            f = new FileWriter(this.outputFileName);
            f.write("#" + Utils.JoinStringArray(columns, "\t") + "\n");
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(5, this.outputFileName);
        }
        int i = 0;
        while (i < finalPeps.length) {
            InspectAnnotation newAnn = new InspectAnnotation();
            newAnn.SpectrumFile = "FakeFile";
            newAnn.ScanNumber = -1;
            newAnn.Annotation = finalPeps[i];
            newAnn.AnnotationWithFlanking = "X." + finalPeps[i] + ".X";
            newAnn.ProteinName = "FakeProtein";
            newAnn.Charge = 0;
            newAnn.IsNovel = true;
            newAnn.FDR = 0.01;
            newAnn.LocalFDR = finalPepLFDR[i];
            newAnn.columnHeaders = columns;
            newAnn.Length = finalPeps[i].length();
            try {
                f.write(String.valueOf(newAnn.toString()) + "\n");
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(7, this.outputFileName);
            }
            ++i;
        }
        try {
            f.close();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(8, this.outputFileName);
        }
    }

    private boolean containedInNovelPeptides(String peptide) {
        return this.novelPeptideSet.contains(peptide);
    }

    private boolean containedInKnownProtein(String peptide) {
        ArrayList locs = this.proteomeTrie.GetAllLocations(peptide);
        return locs != null && locs.size() != 0;
    }

    private long[] getDBIndex(int pos) {
        int cumLength = 0;
        long triePos = 0L;
        int dbIndex = -1;
        int i = 0;
        while (i < this.dbLengths.length) {
            triePos = pos - cumLength;
            if (pos <= (cumLength += (int)this.dbLengths[i])) {
                dbIndex = i;
                break;
            }
            ++i;
        }
        long[] ret = new long[]{dbIndex, triePos};
        return ret;
    }

    private String getPeptide(int dbIndex, long triePos, int length, boolean forwardFlag) {
        if (this.genomicDBDir != null) {
            if (this.debugFlag) {
                System.out.println(" db " + this.genomicTrieDBs[dbIndex].GetDBFileName() + " at triepos " + triePos + " with strand");
            }
            String nucSeq = this.genomicTrieDBs[dbIndex].getPeptideSeq(triePos, length * 3);
            if (forwardFlag) {
                return Utils.translateSequence(nucSeq, 0);
            }
            return Utils.translateSequence(Utils.ReverseComplement(nucSeq), 0);
        }
        if (this.searchDBDir != null) {
            if (dbIndex < this.TrieFiles.length) {
                return this.TrieFiles[dbIndex].getPeptideSeq(triePos, length);
            }
            return this.MS2DBFiles[dbIndex - this.TrieFiles.length].getPeptideSeq(triePos, length);
        }
        return null;
    }

    private int generateRandomPos() {
        int pos = this.randomGen.nextInt((int)this.totalGenomeLength);
        return pos;
    }

    private void initializeGenomicDBs() {
        block14: {
            block13: {
                if (this.genomicDBDir == null) break block13;
                System.out.println("Initializing genomic dbs");
                ArrayList<String> genomeDBs = new ArrayList<String>();
                String[] dbFiles = Utils.ListDir(this.genomicDBDir);
                int i = 0;
                while (i < dbFiles.length) {
                    String currFileName = dbFiles[i];
                    if (Utils.GetFileExtension(currFileName).equals(".trie")) {
                        System.out.println(" - Adding database file " + currFileName);
                        genomeDBs.add(currFileName);
                    } else if (Utils.GetFileExtension(currFileName).equalsIgnoreCase(".fasta") || Utils.GetFileExtension(currFileName).equalsIgnoreCase(".fa")) {
                        System.out.println(" - Prepping database file " + currFileName);
                        String[] dbInfo = TrieDB.prepDB(currFileName, null);
                        genomeDBs.add(dbInfo[0]);
                    }
                    ++i;
                }
                this.genomicTrieDBs = new TrieDB[genomeDBs.size()];
                this.dbLengths = new long[genomeDBs.size()];
                this.totalGenomeLength = 0L;
                i = 0;
                while (i < genomeDBs.size()) {
                    String currDB = (String)genomeDBs.get(i);
                    this.genomicTrieDBs[i] = new TrieDB(currDB);
                    this.dbLengths[i] = this.genomicTrieDBs[i].GetNumAA();
                    this.totalGenomeLength += this.dbLengths[i];
                    ++i;
                }
                break block14;
            }
            if (this.searchDBDir == null) break block14;
            System.out.println("Initializing search dbs");
            ArrayList<String> trieDBs = new ArrayList<String>();
            ArrayList<String> ms2dbs = new ArrayList<String>();
            String[] dbFiles = Utils.ListDir(this.genomicDBDir);
            int i = 0;
            while (i < dbFiles.length) {
                String currFileName = dbFiles[i];
                if (Utils.GetFileExtension(currFileName).equals(".trie")) {
                    System.out.println(" - Adding database file " + currFileName);
                    trieDBs.add(currFileName);
                } else if (Utils.GetFileExtension(currFileName).equalsIgnoreCase(".fasta") || Utils.GetFileExtension(currFileName).equalsIgnoreCase(".fa")) {
                    System.out.println(" - Prepping database file " + currFileName);
                    String[] dbInfo = TrieDB.prepDB(currFileName, null);
                    trieDBs.add(dbInfo[0]);
                } else if (Utils.GetFileExtension(currFileName).equalsIgnoreCase(".ms2db")) {
                    ms2dbs.add(currFileName);
                }
                ++i;
            }
            this.TrieFiles = new TrieDB[trieDBs.size()];
            this.MS2DBFiles = new MS2DB[ms2dbs.size()];
            i = 0;
            while (i < trieDBs.size()) {
                this.TrieFiles[i] = new TrieDB((String)trieDBs.get(i));
                this.dbLengths[i] = this.TrieFiles[i].GetNumAA();
                this.totalGenomeLength += this.dbLengths[i];
                ++i;
            }
            i = 0;
            while (i < ms2dbs.size()) {
                this.MS2DBFiles[i] = new MS2DB((String)ms2dbs.get(i));
                this.dbLengths[i + trieDBs.size()] = this.MS2DBFiles[i].GetNumAA();
                this.totalGenomeLength += this.dbLengths[i + trieDBs.size()];
                ++i;
            }
        }
    }

    private void initializeNovelPeptides() {
        System.out.println("Initializing novel peptides");
        String[] novelPeptideFiles = Utils.ListDir(this.novelPeptideDir);
        this.novelPeptideSet = new HashSet();
        this.novelPeptideInfo = new Hashtable();
        int i = 0;
        while (i < novelPeptideFiles.length) {
            ArrayList locs = GenomicLocator.GenomicColumns.LoadFromFile(novelPeptideFiles[i]);
            int j = 0;
            while (j < locs.size()) {
                Object[] currLoc = (Object[])locs.get(j);
                String pep = (String)currLoc[GenomicLocator.GenomicColumns.Peptide];
                if (pep.indexOf(46) >= 0) {
                    pep = pep.split("[.]")[1];
                }
                this.novelPeptideSet.add(pep);
                double[] info = new double[]{(Double)currLoc[GenomicLocator.GenomicColumns.BestLocalFDR], ((Integer)currLoc[GenomicLocator.GenomicColumns.LocationCount]).intValue()};
                this.novelPeptideInfo.put(pep, info);
                ++j;
            }
            ++i;
        }
        System.out.println("Loaded " + this.novelPeptideSet.size() + " novel peptide sequences");
    }

    public static void main(String[] args) {
        String[] options = new String[]{"-g", "-p", "-n", "-o", "-s", "-d", "-i", "-c", "-l", "-b", "-m", "-q"};
        boolean[] blArray = new boolean[13];
        blArray[0] = true;
        blArray[1] = true;
        blArray[2] = true;
        blArray[3] = true;
        blArray[4] = true;
        blArray[6] = true;
        blArray[7] = true;
        blArray[11] = true;
        blArray[12] = true;
        boolean[] values = blArray;
        Hashtable CommandLineArgs = Utils.ParseCommandLine(args, options, values);
        if (!(CommandLineArgs.containsKey("-p") && CommandLineArgs.containsKey("-n") && CommandLineArgs.containsKey("-o") && CommandLineArgs.containsKey("-s"))) {
            System.err.println(usageInfo);
            ErrorThrower.ThrowError(2);
        }
        if (!CommandLineArgs.containsKey("-g") && !CommandLineArgs.containsKey("-q")) {
            System.err.println(usageInfo);
            ErrorThrower.ThrowErrorCustum(100, "Must specify either a genome directory (-g) or search DB directory (-q)");
        }
        String genomicDBDir = null;
        if (CommandLineArgs.containsKey("-g")) {
            genomicDBDir = (String)CommandLineArgs.get("-g");
        }
        String searchDBDir = null;
        if (CommandLineArgs.containsKey("-q")) {
            searchDBDir = (String)CommandLineArgs.get("-q");
        }
        String proteomeFile = (String)CommandLineArgs.get("-p");
        String novelDir = (String)CommandLineArgs.get("-n");
        String outputFileName = (String)CommandLineArgs.get("-o");
        int numPeps = Integer.parseInt((String)CommandLineArgs.get("-s"));
        String pepFileName = null;
        if (CommandLineArgs.containsKey("-i")) {
            pepFileName = (String)CommandLineArgs.get("-i");
        }
        boolean buildFromNovel = false;
        if (CommandLineArgs.containsKey("-c")) {
            buildFromNovel = true;
        }
        boolean writeLocations = false;
        if (CommandLineArgs.containsKey("-l")) {
            writeLocations = true;
        }
        boolean inputIsLocations = false;
        if (CommandLineArgs.containsKey("-b")) {
            inputIsLocations = true;
        }
        int maxLocations = Integer.MAX_VALUE;
        if (CommandLineArgs.containsKey("-m")) {
            maxLocations = Integer.parseInt((String)CommandLineArgs.get("-m"));
        }
        DecoyPeptideGenerator g = new DecoyPeptideGenerator(genomicDBDir, proteomeFile, novelDir, outputFileName, numPeps, pepFileName, buildFromNovel, writeLocations, inputIsLocations, maxLocations, searchDBDir);
        if (CommandLineArgs.containsKey("-d")) {
            g.debugFlag = true;
        }
        g.generatePeptides();
    }
}

