/*
 * Decompiled with CFR 0.152.
 */
package proteogenomicUtils;

import basicUtils.InspectAnnotation;
import basicUtils.Utils;
import errorUtils.ErrorThrower;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;

public class DetermineSourceSpecCounts {
    private static final double NORMALIZED_SPEC_COUNT = 3.0E7;
    public static String usageInfo = "proteogenomicUtils.DetermineSourceSpecCounts version 2012.03.12\nThis class is used to get per peptide spectral counts for each identified protein/peptide.  Only unique peptides areconsidered.\n[Required]:\n -i [FILE] Event file in table format\n OR\n -p [FILE] Identified protein file\n OR\n -r [FILE] Generic table with proteins listed in one column (comma-delimited)\n -n [DIR] Directory of Inspect results containing peptides that were grouped into events/proteins\n -t [NAME:FILE] A case-insensitive sample label that is contained in the spectrum file names:A file containing the cluster to spectrum info for the sample (Can specify multiple times)\n -d [DIR] Directory containing map files from filtered clusters to raw clusters\n -o [FILE] Output file to write updated event information (an added column containing the sources)\n( -c [NUM] Column number containing the peptides if a generic table is input, default=0)\n";
    private String inputEventFileName = null;
    private String proteinFileName = null;
    private String[] labels = null;
    private String[] mapFileNames = null;
    private String[] peptideResultFiles = null;
    private String genericInputTable = null;
    private int colNum = 0;
    private String outputFileName = null;
    private Hashtable novelPeptideSourceHash = null;
    private String mapDirectory;
    private static String[] specFileNames = new String[]{"Maize-Embryo", "Maize-Endosperm-8DAP", "Maize-Endosperm-12DAP", "Maize-Endosperm-10DAP", "Maize-Pericarp", "Maize-GermKernel"};
    private static double[] scaleFactors = new double[]{2.3200878E7, 2.8111693E7, 7473551.0, 7658654.0, 2.4414192E7, 1.744308E7};
    private static String[] specFileNamesOld = new String[]{"Maize-Endosperm-10DAP-10-Try-150ug-2D29-122109-LTQ3", "Maize-Endosperm-10DAP-10_11-try-4mg-1st2mgCeO2-Elu1-2D19-122209-LTQ2", "Maize-Endosperm-10DAP-11-Try-150ug-2D29-122409-LTQ3", "Maize-Endosperm-10DAP-12-Try-100ug-C18FT-2D19-012010-LTQ3", "Maize-Endosperm-10DAP-12-Try-150ug-2D29-010410-LTQ3", "Maize-Endosperm-10DAP-12-Try-2mg-1mgCeO2-Elu-2D19-011810-LTQ3", "Maize-Endosperm-10DAP-12-try-100ug-2D29-b-011210-LTQ2", "Maize-Endosperm-10DAP-9-HClWash-try-1mg-1mgCeO2-2D19-122409-LTQ2", "Maize-Endosperm-10DAP-9-Try-150ug-2D29-121709-LTQ3", "Maize-Endosperm-10DAP-9-Try-2mg-noAntiP-2mgCeO2-Elu-2D9-121509-LTQ3", "Maize-Endosperm-10DAP-9-try-1mg-1mgCeO2-2D19-122409-LTQ1", "Maize-Endosperm-12DAP-13-try-150ug-2D29-010410-LTQ1", "Maize-Endosperm-12DAP-13-try-5mg-1mgCeO2-2D19-012510-LTQ1", "Maize-Endosperm-12DAP-14-try-150ug-2D29-010410-LTQ2", "Maize-Endosperm-12DAP-14-try-5mg-1mgCeO2-Elu-2D19-012610-LTQ2", "Maize-Endosperm-12DAP-15-Try-150ug-2D29-010710-LTQ3", "Maize-Endosperm-12DAP-15-try-5mg-1mgCeO2-2D19-012810-LTQ1", "Maize-Endosperm-12DAP-16-try-150ug-2D29-010710-LTQ1", "Maize-Endosperm-12DAP-16-try-5mg-1mgCeO2-2D19-011510-LTQ1", "Maize-Endosperm-12DAP-17-try-150ug-2D29-010710-LTQ2", "Maize-Endosperm-12DAP-17-try-5mg-1mgCeO2-Elu-2D19-011510-LTQ2", "Maize-Endosperm-12DAP-18-Try-150ug-2D29-011110-LTQ3", "Maize-Endosperm-12DAP-18-try-5mg-1mgCeO2-2D19-011210-LTQ1", "Maize-Endosperm-8DAP-2-Try-4mg-1st5mgCeO2-2D19-101209-LTQ3", "Maize-Endosperm-8DAP-2-Try-4mg-2nd5mgCeO2-2D19-101509-LTQ3", "Maize-Endosperm-8DAP-2a-Try-100ug-2D29-101609-LTQ3", "Maize-Endosperm-8DAP-2a-Try-150ug-2D29-100909-LTQ3", "Maize-Endosperm-8DAP-2a-Try-200ug-2D43-093009-LTQ3", "Maize-Endosperm-8DAP-2a-Try-300ug-2D29-100609-LTQ3", "Maize-Endosperm-8DAP-2a-trypsin-150ug-2D29-100909-LTQ1", "Maize-Endosperm-8DAP-2a-trypsin-150ug-2D38-091709-LTQ1", "Maize-Endosperm-8DAP-2a-trypsin-1mg-1st5mgCeO2-Elu-2D24-0918090-LTQ2", "Maize-Endosperm-8DAP-2a-trypsin-1mg-2nd5mgCeO2-Elu-2D24-0922090-LTQ2", "Maize-Endosperm-8DAP-2a-trypsin-300ug-2D36-092409-LTQ2", "Maize-Endosperm-8DAP-3-Try-10mg-1st10mgCeO2-Elu1-2D19-103009-LTQ3", "Maize-Endosperm-8DAP-3-Try-10mg-1st10mgCeO2-Elu1-C18FT-2D19-110209-LTQ3", "Maize-Endosperm-8DAP-3-Try-10mg-1st10mgCeO2-Elu3-2D19-110409-LTQ3", "Maize-Endosperm-8DAP-3-try-10mg-1st10mgCeO2-Elu2-2D19-1030090-LTQ2", "Maize-Endosperm-8DAP-3-try-10mg-1st10mgCeO2-Elu3-10pct-2D2hr-110209-LTQ2", "Maize-Endosperm-8DAP-3a-Try-150ug-2D29-102309-LTQ3", "Maize-Endosperm-8DAP-4-Try-10mg-1st10mgCeO2-Elu-2D19-111109-LTQ3", "Maize-Endosperm-8DAP-4-Try-10mg-1st10mgCeO2-Elu-2D9-110909-LTQ3", "Maize-Endosperm-8DAP-4a-Try-150ug-2D29-102609-LTQ3", "Maize-Endosperm-8DAP-5-Try-10mg-1st10mgCeO2-Elu-10pct-2D9-111009-LTQ3", "Maize-Endosperm-8DAP-5-Try-10mg-1st10mgCeO2-Elu-90pct-2D9-111609-LTQ3", "Maize-Endosperm-8DAP-5-Try-150ug-2D29-110609-LTQ3", "Maize-Endosperm-8DAP-6-Try-10mg-1st10mgCeO2-Elu-2D9-111909-LTQ3", "Maize-Endosperm-8DAP-6-Try-10mg-1st10mgCeO2-Elu2-2D9-112309-LTQ3", "Maize-Endosperm-8DAP-6-Try-150ug-2D29-111309-LTQ3", "Maize-Endosperm-8DAP-6-Try-150ug-b-2D29-112509-LTQ3", "Maize-Endosperm-8DAP-6-Try-150ug-b-2D29-120109-LTQ3", "Maize-Endosperm-8DAP-7-Try-10mg-1st10mgCeO2-Elu-2D9-112409-LTQ3", "Maize-Endosperm-8DAP-7-Try-150ug-2D19-111709-LTQ3", "Maize-Endosperm-8DAP-7-Try-150ug-2D29-112009-LTQ3", "Maize-Endosperm-8DAP-7-Try-150ug-b-2D29-112809-LTQ3", "Maize-Endosperm-8DAP-7-Try-150ug-b-2D29-120409-LTQ3", "Maize-Endosperm-8DAP-7-Try-5mg-10mgCeO2-Elu-2D9-120709-LTQ3", "Maize-Endosperm-8DAP-8-Try-150ug-2D29-121109-LTQ3", "Maize-Endosperm-8DAP-8-Try-2mg-5mgCeO2-Elu-2D9-121009-LTQ3", "Maize-Endosperm-8DAP-8-Try-2mg-5mgCeO2-Elu2-10pct-2D2hr-121109-LTQ3", "Maize-Endosperm-8DAP-8-Try-2mg-5mgCeO2-Elu2-NH3-2D9-121409-LTQ3", "Maize-Endosperm-8DAP-8-Try-SepPak-5mg-10mgCeO2-Elu-2D9-120809-LTQ3", "Maize-Endosperm-8DAP-8-trypsin-SepPak-150ug-2D29-120409-LTQ1", "Maize-Embryo-41-Try-300ug-2D29-022610-LTQ3", "Maize-Embryo-41-Try-3mg-1st3mgCeO2-Elu-2D19-030110-LTQ3", "Maize-Embryo-41-Try-3mg-2nd1mgCeO2-Elu-2D19-022310-LTQ3", "Maize-Embryo-42-Try-4mg-5mgCeO2-Elu-2D19-041410-LTQ3", "Maize-Embryo-42-Try-50ug-2D29-030810-LTQ3", "Maize-Embryo-42-try-150ug-2D29-032210-LTQ1", "Maize-Embryo-42_43_44_46-Try-8mg-4mgCeO2-Elu-2D19-031210-LTQ3", "Maize-Embryo-43-Try-100ug-2D29-030510-LTQ3", "Maize-Embryo-43-Try-4mg-5mgCeO2-Elu-2D19-041210-LTQ3", "Maize-Embryo-43-try-150ug-2D29-032210-LTQ2", "Maize-Embryo-44-try-100ug-2D29-030510-LTQ1", "Maize-Embryo-44-try-4mg-5mgCeO2-Elu-2D19-041210-LTQ1", "Maize-Embryo-46-try-100ug-2D29-030610-LTQ2", "Maize-Embryo-46-try-4mg-5mgCeO2-Elu-2D19-041210-LTQ2", "Maize-Embryo-47-Try-150ug-2D29-031910-LTQ3", "Maize-Embryo-47-Try-8mg-2nd1mgCeO2-Elu-2D9-040110-LTQ3", "Maize-Embryo-47-Try-8mg-3mgCeO2-Elu-2D19-032610-LTQ3", "Maize-Embryo-47-Try-8mg-3mgCeO2-Elu2-2D9-040610-LTQ3", "Maize-Embryo-47-try-8mg-3rd-5mgCeO2-Elu-2D19-040610-LTQ2", "Maize-Embryo-48-Try-10mg-1st1mgCeO2-Elu-2D19-022210-LTQ3", "Maize-Embryo-48-Try-300ug-2D29-021810-LTQ3", "Maize-Embryo-48-try-10mg-2nd2mgCeO2-Elu-2D9-022210-LTQ2", "Maize-Embryo-48-try-3mg-2mgCeO2-Elu-2D19-041610-LTQ1", "Maize-Embryo-48-try-3mg-2mgCeO2-Elu-2D9-030410-LTQ2", "Maize-Embryo-48a-3KMWCO-try-1mg-1mgCeO2-Na3PO4Elu-2D9-022410-LTQ2", "Maize-Embryo-48b-3kMWCO-try-3mg-4mgCeO2-EDTA_NH3Elu-2D19-022610-LTQ1", "Maize-Embryo-48b-noMWCO-Try-1mg-1mgCeO2-EDTAElu2-2D9-022510-LTQ3", "Maize-Embryo-48b-noMWCO-try-1mg-1mgCeO2-NH3Elu-2D9-022510-LTQ2", "Maize-Embryo-48b-noMWCO-try-4mg-4mgCeO2-EDTA_NH3Elu-2D19-022610-LTQ2", "Maize-Embryo-49-Try-10mg-3rd5mgCeO2-Elu-2D19-040710-LTQ3", "Maize-Embryo-49-try-10mg-2nd-1mgCeO2-Elu-2D19-040110-LTQ2", "Maize-Embryo-49-try-10mg-3mgCeO2-Elu-2D19-032610-LTQ1", "Maize-Embryo-49-try-150ug-2D29-031910-LTQ1", "Maize-Embryo-50-try-10mg-3rd5mgCeO2-Elu-2D19-040810-LTQ1", "Maize-Embryo-50-try-12mg-3mgCeO2-2D19-032610-LTQ2", "Maize-Embryo-50-try-12mg-3mgCeO2-Elu2-2D19-032910-LTQ2", "Maize-Embryo-50-try-150ug-2D29-031910-LTQ2", "Maize-Embryo-51-Try-13mg-3mgCeO2-Elu-2D19-032910-LTQ3", "Maize-Embryo-51-Try-150ug-2D29-032210-LTQ3", "Maize-PeriAleu-21-Try-2DClean-250ug-500ugCeO2-2D9-071510-LTQ3", "Maize-PeriAleu-21-Try-C18SP-500ug-1mgCeO2-5pctTFA-PQD-2D9-071410", "Maize-PericarpAleurone-19-20-21-22-23-Try-100ug-1mgCeO2-2D9-042710-LTQ3", "Maize-PericarpAleurone-19-20-21-22-23-Try-1mg-1mgCeO2-2D9-042910-LTQ3", "Maize-PericarpAleurone-19-20-21-22-23-Try-250ug-1mgCeO2-2D9-042810-LTQ3", "Maize-PericarpAleurone-19-20-21-22-23-Try-500ug-1mgCeO2-2D9-050610-LTQ3", "Maize-PericarpAleurone-19-Try-150ug-2D29-043010-LTQ3", "Maize-PericarpAleurone-19-Try-300ug-2D29-b-040210-LTQ3", "Maize-PericarpAleurone-19-Try-5mg-5mgCeO2-Elu-2D19-051010-LTQ3", "Maize-PericarpAleurone-19-Try-7mg-2mgCeO2-Elu-2D19-021610-LTQ3", "Maize-PericarpAleurone-20-Try-150ug-2D29-050310-LTQ3", "Maize-PericarpAleurone-20-Try-150ug-2D29-050310-LTQ3", "Maize-PericarpAleurone-20-try-5mg-5mgCeO2-Elu-2D19-051310-LTQ2", "Maize-PericarpAleurone-20_21-try-3mg-1mgCeO2-Elu-2D19-021310-LTQ2", "Maize-PericarpAleurone-21-Try-2ndC18SepPak-20ug-2D9-071210-LTQ3", "Maize-PericarpAleurone-21-Try-2ndC18SepPak-20ug-PQD-2D9-071310-LTQ3", "Maize-PericarpAleurone-21-Try-3mg-3mgCeO2-Elu-2D19-050710-LTQ3", "Maize-PericarpAleurone-21-Try-C18SepPak-TMT0-100ug-2D29-070910-LTQ3", "Maize-PericarpAleurone-21-try-150ug-2D29-050310-LTQ1", "Maize-PericarpAleurone-21-try-C18SepPak-150ug-2D29-070910-LTQ1", "Maize-PericarpAleurone-22-5mg-5mgCeO2-Elu-2D19-051010-LTQ1", "Maize-PericarpAleurone-22-Try-2mg-2mgCeO2-Elu-50pct-2D19-030310-LTQ3", "Maize-PericarpAleurone-22-try-150ug-2D29-050310-LTQ2", "Maize-PericarpAleurone-22-try-2mg-2mgCeO2-Elu-50pct-2D19-030210-LTQ2", "Maize-PericarpAleurone-23-try-150ug-2D29-b-040210-LTQ2", "Maize-PericarpAleurone-23-try-5mg-5mgCeO2-Elu-2D19-051010-LTQ2", "Maize-PericarpAleurone-23-try-7mg-2mgCeO2-Elu-2D19-021610-LTQ2", "Maize-PericarpAleurone-24-try-215ug-250uMx30cm-043010-LTQ1", "Maize-PericarpAluerone-19-Try-300ug-2D29-020810-LTQ3", "Maize-PericarpAluerone-20-Try-300ug-250uMx30cm-2D29-020510-LTQ3", "Maize-PericarpAluerone-20-Try-3mg-1mgCeO2-Elu-2D9-021110-LTQ3", "Maize-PericarpAluerone-20-Try-5mg-2mgCeO2-Elu-2D9-020410-LTQ3", "Maize-PericarpAluerone-20-Try-5mg-2mgCeO2-Elu-5pct-2D2hr-020510-LTQ3", "Maize-PericarpAluerone-21-Try-300ug-250uMx30cm-2D29-020110-LTQ3", "Maize-PericarpAluerone-21-Try-4mg-2mgCeO2-Elu-2D19-012710-LTQ3", "Maize-PericarpAluerone-21-try-200ug-250uMx30cm-020110-LTQ1", "Maize-PericarpAluerone-22-Try-100ug-2D29-011410-LTQ3", "Maize-PericarpAluerone-22b-Try-150ug-250uMx30cm-2D29-012210-LTQ3", "Maize-PericarpAluerone-22b-Try-5mg-2mgCeO2-Elu-2D9-012510-LTQ3", "Maize-PericarpAluerone-23-Try-300ug-2D29-021310-LTQ3", "Maize-GermKernel-52-SDS-try-8mg-MCX-1mgCeO2-2D19-092210-LTQ1", "Maize-GermKernel-52-Try-150ug-2D29-082010-LTQ3", "Maize-GermKernel-52-try-10mg-MCX-1mgCeO2-2D29-083110-LTQ1", "Maize-GermKernel-52_53_54-55_56-try-20mg-SDS-MCX-1mgCeO2-2D29-082010-LTQ1", "Maize-GermKernel-52to56-Try-20mg-MCX-2nd500ugCeO2-2D19-082310-LTQ3", "Maize-GermKernel-52to56-Try-30mg-SDS-MCX-2nd1mgCeO2-2D19-082610-LTQ2", "Maize-GermKernel-52to56-Try-MCX-150ug-2D29-090610-LTQ3", "Maize-GermKernel-52to56-Try-MCX-150ug-2D29-090610-LTQ3", "Maize-GermKernel-53-SDS-Try-10mg-MCX-500ugCeO2-2D19-091510-LTQ2", "Maize-GermKernel-53-SDS-try-10mg-MCX-2nd500ugCeO2-2D19-092010-LTQ1", "Maize-GermKernel-53-Try-10mg-MCX-1mgCeO2-2D19-083110-LTQ2", "Maize-GermKernel-53-Try-SDS-MCX-150ug-2D29-091310-LTQ3", "Maize-GermKernel-53-try-150ug-2D29-082410-LTQ1", "Maize-GermKernel-54-SDS-Try-10mg-MCX-2nd500ugCeO2-2D19-092010-LTQ2", "Maize-GermKernel-54-SDS-Try-10mg-MCX-500ugCeO2-2D19-091310-LTQ2", "Maize-GermKernel-54-Try-150ug-2D29-082710-LTQ3", "Maize-GermKernel-54-Try-SDS-MCX-150ug-2D29-091010-LTQ3", "Maize-GermKernel-55-SDS-Try-10mg-MCX-500ugCeO2-2D9-092310-LTQ3", "Maize-GermKernel-55-Try-150ug-2D29-083010-LTQ3", "Maize-GermKernel-55-Try-9mg-MCX-500ugCeO2-2D9-102110-LTQ3", "Maize-GermKernel-56-SDS-Try-6mg-MCX-1mgCeO2-2D19-092210-LTQ2", "Maize-GermKernel-56-try-150ug-2D29-090310-LTQ1"};
    private static double[] scaleFactorsOld = new double[]{983635.0, 457088.0, 1028634.0, 299258.0, 982630.0, 715392.0, 689452.0, 656309.0, 1101893.0, 282990.0, 461373.0, 672642.0, 477407.0, 670775.0, 419649.0, 998430.0, 457447.0, 668835.0, 473743.0, 716217.0, 415590.0, 1040956.0, 461860.0, 673808.0, 396544.0, 1381628.0, 1216524.0, 956421.0, 1138164.0, 731131.0, 943162.0, 607741.0, 559728.0, 949337.0, 637200.0, 628490.0, 719761.0, 471968.0, 20553.0, 1115298.0, 644812.0, 325760.0, 1144394.0, 269382.0, 389184.0, 1076655.0, 358274.0, 285490.0, 1114260.0, 579829.0, 1212772.0, 328470.0, 993534.0, 1119651.0, 662850.0, 1222221.0, 381072.0, 1106608.0, 319568.0, 27984.0, 319308.0, 378032.0, 704125.0, 950691.0, 724112.0, 565657.0, 725854.0, 968902.0, 700656.0, 738782.0, 1053807.0, 700773.0, 676306.0, 691333.0, 586052.0, 699361.0, 521092.0, 1116802.0, 260138.0, 734486.0, 253715.0, 503914.0, 350665.0, 1084310.0, 206789.0, 495260.0, 190800.0, 204171.0, 439967.0, 280512.0, 206870.0, 495604.0, 677781.0, 211817.0, 497488.0, 729452.0, 597282.0, 534127.0, 425709.0, 724040.0, 571662.0, 1078889.0, 270980.0, 323164.0, 295936.0, 294176.0, 306544.0, 294571.0, 994493.0, 1020419.0, 720125.0, 684165.0, 889993.0, 710048.0, 465507.0, 497674.0, 315316.0, 298605.0, 791760.0, 1084844.0, 684148.0, 700344.0, 497512.0, 644448.0, 622846.0, 479121.0, 678342.0, 526789.0, 447247.0, 686056.0, 1108211.0, 1125284.0, 272861.0, 17570.0, 27946.0, 1029418.0, 475563.0, 680921.0, 1006437.0, 1071725.0, 308178.0, 1064905.0, 529149.0, 1073775.0, 536089.0, 681141.0, 923025.0, 520443.0, 1528708.0, 778528.0, 474592.0, 490575.0, 532542.0, 1475602.0, 726774.0, 480989.0, 474467.0, 1385693.0, 1505177.0, 354794.0, 1473981.0, 318416.0, 504576.0, 674044.0};

    public DetermineSourceSpecCounts(String inputFileName, String proteinFileName, String genericInputTable, int colNum, String peptideDir, String[] labels, String outputFileName, String mapDirectory) {
        if (inputFileName != null && (proteinFileName != null || genericInputTable != null)) {
            ErrorThrower.ThrowWarning(101, "Should only specify an event file OR a proteome file OR a generic table.  We will ignore the proteome file/generic table");
            proteinFileName = null;
            genericInputTable = null;
        }
        if (inputFileName != null && !Utils.IsFile(inputFileName)) {
            ErrorThrower.ThrowError(1, inputFileName);
        }
        this.inputEventFileName = inputFileName;
        if (proteinFileName != null && genericInputTable != null) {
            ErrorThrower.ThrowWarning(101, "Should only specify an event file OR a proteome file OR a generic table.  We will ignore the generic table");
            genericInputTable = null;
        }
        if (proteinFileName != null && !Utils.IsFile(proteinFileName)) {
            ErrorThrower.ThrowError(1, proteinFileName);
        }
        this.proteinFileName = proteinFileName;
        if (genericInputTable != null && !Utils.IsFile(genericInputTable)) {
            ErrorThrower.ThrowError(1, genericInputTable);
        }
        this.genericInputTable = genericInputTable;
        if (genericInputTable != null && colNum < 0) {
            ErrorThrower.ThrowError(4, "Column number must be non-negative: " + colNum);
        }
        this.colNum = colNum;
        if (!Utils.IsDir(peptideDir)) {
            ErrorThrower.ThrowError(1, peptideDir);
        }
        this.peptideResultFiles = Utils.ListDir(peptideDir);
        this.labels = new String[labels.length + 1];
        this.mapFileNames = new String[labels.length];
        int i = 0;
        while (i < labels.length) {
            String[] bits = labels[i].split(":");
            if (bits.length != 2) {
                ErrorThrower.ThrowError(4, "Must specify label as 'STR:FILE'");
            }
            this.labels[i] = bits[0].toLowerCase();
            this.mapFileNames[i] = bits[1];
            if (!Utils.IsFile(this.mapFileNames[i])) {
                ErrorThrower.ThrowError(1, this.mapFileNames[i]);
            }
            ++i;
        }
        this.labels[labels.length] = "other";
        if (!Utils.IsDir(mapDirectory)) {
            ErrorThrower.ThrowError(1, mapDirectory);
        }
        this.mapDirectory = mapDirectory;
        this.outputFileName = outputFileName;
    }

    public static void main(String[] args) {
        String[] options = new String[]{"-i", "-n", "-t", "-o", "-p", "-d", "-r", "-c"};
        boolean[] values = new boolean[]{true, true, true, true, true, true, true, true};
        Hashtable commandLineArgs = Utils.ParseCommandLineMulti(args, options, values);
        if (!(commandLineArgs.containsKey("-n") && commandLineArgs.containsKey("-t") && commandLineArgs.containsKey("-o") && commandLineArgs.containsKey("-d"))) {
            System.out.println(usageInfo);
            ErrorThrower.ThrowError(2, "Must specify an output file (-o), an Inspect result directory (-n), at least one sample label (-t), and a direcotry containing filtered cluster maps (-d)");
        }
        if (!(commandLineArgs.containsKey("-i") || commandLineArgs.containsKey("-p") || commandLineArgs.containsKey("-r"))) {
            System.out.println(usageInfo);
            ErrorThrower.ThrowError(2, "Must specify an input event file (-i) or a selected protein file (-p) or a generic table file (-r)");
        }
        if (commandLineArgs.containsKey("-i") && commandLineArgs.containsKey("-p") || commandLineArgs.containsKey("-i") && commandLineArgs.containsKey("-r") || commandLineArgs.containsKey("-r") && commandLineArgs.containsKey("-p")) {
            System.out.println(usageInfo);
            ErrorThrower.ThrowError(2, "Must specify exactly one input event file (-i) or a selected protein file (-p) or a generic table file (-r)");
        }
        String inputFileName = null;
        if (commandLineArgs.containsKey("-i")) {
            inputFileName = (String)((ArrayList)commandLineArgs.get("-i")).get(0);
        }
        String proteinFileName = null;
        if (commandLineArgs.containsKey("-p")) {
            proteinFileName = (String)((ArrayList)commandLineArgs.get("-p")).get(0);
        }
        String genericFileName = null;
        if (commandLineArgs.containsKey("-r")) {
            genericFileName = (String)((ArrayList)commandLineArgs.get("-r")).get(0);
        }
        int colNum = 0;
        if (commandLineArgs.containsKey("-c")) {
            colNum = Integer.parseInt((String)((ArrayList)commandLineArgs.get("-c")).get(0));
        }
        String peptideDir = (String)((ArrayList)commandLineArgs.get("-n")).get(0);
        String[] labels = Utils.ConvertArraylistToStringArray((ArrayList)commandLineArgs.get("-t"));
        String outputFileName = (String)((ArrayList)commandLineArgs.get("-o")).get(0);
        String mapDirectory = (String)((ArrayList)commandLineArgs.get("-d")).get(0);
        DetermineSourceSpecCounts d = new DetermineSourceSpecCounts(inputFileName, proteinFileName, genericFileName, colNum, peptideDir, labels, outputFileName, mapDirectory);
        d.countSpectra();
    }

    private void countSpectra() {
        if (this.inputEventFileName != null) {
            System.out.println("[1] - Load peptides of interest from events\n");
            this.loadEventPeptides();
            System.out.println(" - Total novel peptides in events: " + this.novelPeptideSourceHash.size());
        } else if (this.proteinFileName != null) {
            System.out.println("[1] - Load peptides of interest from selected proteins\n");
            this.loadProteinPeptides();
            System.out.println(" - Total peptides in selected proteins: " + this.novelPeptideSourceHash.size());
        } else {
            System.out.println("[1] - Load peptides of interest from generic table\n");
            this.loadGenericPeptides();
            System.out.println(" - Total peptides in generic table: " + this.novelPeptideSourceHash.size());
        }
        System.out.println("[2] - Identify sources of peptides\n");
        this.identifyPeptideSources();
        if (this.inputEventFileName != null) {
            System.out.println("[3] - Write out events with sources\n");
            this.writeEventsWithSources();
        } else if (this.proteinFileName != null) {
            System.out.println("[3] - Write out proteins with sources\n");
            this.writeProteinsWithSources();
        } else {
            System.out.println("[3] - Write out generic table with sources\n");
            this.writeGenericWithSources();
        }
    }

    private void writeGenericWithSources() {
        BufferedReader buf = Utils.openBufferedReader(this.genericInputTable);
        FileWriter f = Utils.openFileWriter(this.outputFileName);
        Utils.writeLine(f, this.outputFileName, "#Protein\tGene\tCoverage\tPeptides\tUniquePeptides\tSharedPeptides\tSources\t" + Utils.JoinStringArray(this.labels, "\t") + "\n");
        String line = Utils.readNextLine(buf, this.genericInputTable);
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, this.proteinFileName);
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits[this.colNum].split(",");
            int[] specCounts = new int[this.labels.length];
            HashSet<String> sources = new HashSet<String>();
            int i = 0;
            while (i < pepList.length) {
                String ann = Utils.GetUnModded(pepList[i]);
                int[] vals = (int[])this.novelPeptideSourceHash.get(ann);
                int j = 0;
                while (j < vals.length) {
                    int n = j;
                    specCounts[n] = specCounts[n] + vals[j];
                    sources.add(this.labels[j]);
                    ++j;
                }
                ++i;
            }
            i = 0;
            while (i < specCounts.length) {
                int n = i++;
                specCounts[n] = specCounts[n] / pepList.length;
            }
            if (sources.size() == 0) {
                ErrorThrower.ThrowWarningCustom(101, "No sources found for event: " + line);
            }
            String sEls = Utils.JoinIntArray(specCounts, "\t");
            line = String.valueOf(line) + "\t" + sEls;
            Utils.writeLine(f, this.outputFileName, String.valueOf(line) + "\n");
            line = Utils.readNextLine(buf, this.genericInputTable);
        }
        Utils.closeBufferedReader(buf, this.genericInputTable);
        Utils.closeFileWriter(f, this.outputFileName);
    }

    private void loadGenericPeptides() {
        this.novelPeptideSourceHash = new Hashtable();
        BufferedReader buf = Utils.openBufferedReader(this.genericInputTable);
        String line = Utils.readNextLine(buf, this.genericInputTable);
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, this.genericInputTable);
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits[this.colNum].split(",");
            int i = 0;
            while (i < pepList.length) {
                String ann = Utils.GetUnModded(pepList[i]);
                int[] vals = new int[this.labels.length];
                vals = Utils.initializeIntArray(vals, 0);
                this.novelPeptideSourceHash.put(ann, vals);
                ++i;
            }
            line = Utils.readNextLine(buf, this.genericInputTable);
        }
        Utils.closeBufferedReader(buf, this.genericInputTable);
    }

    private void writeProteinsWithSources() {
        BufferedReader buf = Utils.openBufferedReader(this.proteinFileName);
        FileWriter f = Utils.openFileWriter(this.outputFileName);
        Utils.writeLine(f, this.outputFileName, "#Protein\tGene\tCoverage\tPeptides\tUniquePeptides\tSharedPeptides\tSources\t" + Utils.JoinStringArray(this.labels, "\t") + "\n");
        String line = Utils.readNextLine(buf, this.proteinFileName);
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, this.proteinFileName);
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits[4].split(",");
            int[] specCounts = new int[this.labels.length];
            HashSet<String> sources = new HashSet<String>();
            int i = 0;
            while (i < pepList.length) {
                String ann = Utils.GetUnModded(pepList[i]);
                int[] vals = (int[])this.novelPeptideSourceHash.get(ann);
                int j = 0;
                while (j < vals.length) {
                    int n = j;
                    specCounts[n] = specCounts[n] + vals[j];
                    sources.add(this.labels[j]);
                    ++j;
                }
                ++i;
            }
            i = 0;
            while (i < specCounts.length) {
                int n = i++;
                specCounts[n] = specCounts[n] / pepList.length;
            }
            if (sources.size() == 0) {
                ErrorThrower.ThrowWarningCustom(101, "No sources found for event: " + line);
            }
            String sEls = Utils.JoinIntArray(specCounts, "\t");
            line = String.valueOf(line) + "\t" + sEls;
            Utils.writeLine(f, this.outputFileName, String.valueOf(line) + "\n");
            line = Utils.readNextLine(buf, this.proteinFileName);
        }
        Utils.closeBufferedReader(buf, this.proteinFileName);
        Utils.closeFileWriter(f, this.outputFileName);
    }

    private void loadProteinPeptides() {
        this.novelPeptideSourceHash = new Hashtable();
        BufferedReader buf = Utils.openBufferedReader(this.proteinFileName);
        String line = Utils.readNextLine(buf, this.proteinFileName);
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, this.proteinFileName);
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits[4].split(",");
            int i = 0;
            while (i < pepList.length) {
                String ann = Utils.GetUnModded(pepList[i]);
                int[] vals = new int[this.labels.length];
                vals = Utils.initializeIntArray(vals, 0);
                this.novelPeptideSourceHash.put(ann, vals);
                ++i;
            }
            line = Utils.readNextLine(buf, this.proteinFileName);
        }
        Utils.closeBufferedReader(buf, this.proteinFileName);
    }

    private void writeEventsWithSources() {
        BufferedReader buf = null;
        FileWriter f = null;
        String line = null;
        try {
            buf = new BufferedReader(new FileReader(this.inputEventFileName));
            line = buf.readLine();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(5, this.inputEventFileName);
        }
        try {
            f = new FileWriter(this.outputFileName);
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(5, this.outputFileName);
        }
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                try {
                    line = buf.readLine();
                }
                catch (IOException E) {
                    ErrorThrower.ThrowError(6, this.inputEventFileName);
                }
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits.length == 5 ? eventBits[3].split(",") : eventBits[2].split(",");
            HashSet<String> sources = new HashSet<String>();
            int i = 0;
            while (i < pepList.length) {
                String[] pepInfo = pepList[i].split(":");
                String ann = Utils.GetUnModded(pepInfo[0]);
                int[] vals = (int[])this.novelPeptideSourceHash.get(ann);
                int j = 0;
                while (j < vals.length) {
                    if (vals[j] > 0) {
                        sources.add(this.labels[j]);
                    }
                    ++j;
                }
                ++i;
            }
            if (sources.size() == 0) {
                ErrorThrower.ThrowWarningCustom(101, "No sources found for event: " + line);
            }
            String sEls = Utils.JoinHashSet(sources, ",");
            line = String.valueOf(line) + "\t" + sEls;
            try {
                f.write(String.valueOf(line) + "\n");
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(7, this.outputFileName);
            }
            try {
                line = buf.readLine();
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(6, this.inputEventFileName);
            }
        }
        try {
            buf.close();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(8, this.inputEventFileName);
        }
        try {
            f.close();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(8, this.outputFileName);
        }
    }

    private void identifyPeptideSources() {
        boolean debug = false;
        int i = 0;
        while (i < this.peptideResultFiles.length) {
            System.out.println(" - Loading peptide sources from " + this.peptideResultFiles[i]);
            Hashtable specHash = InspectAnnotation.LoadInspectAnnotationsIntoSpec2PeptideHash(this.peptideResultFiles[i]);
            Enumeration e = specHash.keys();
            while (e.hasMoreElements()) {
                String hashKey = (String)e.nextElement();
                String peptide = (String)specHash.get(hashKey);
                if (peptide.indexOf(46) >= 0) {
                    peptide = peptide.split("[.]")[1];
                }
                if (!this.novelPeptideSourceHash.containsKey(peptide)) continue;
                if (debug) {
                    System.out.println(String.valueOf(hashKey) + " -> " + peptide + " is in an event!");
                }
                int[] vals = (int[])this.novelPeptideSourceHash.get(peptide);
                boolean found = false;
                int j = 0;
                while (j < this.labels.length - 1 && !found) {
                    if (hashKey.toLowerCase().indexOf(this.labels[j]) >= 0) {
                        if (debug) {
                            System.out.println(" source found: " + this.labels[j]);
                        }
                        int n = j;
                        vals[n] = vals[n] + this.getSpecCount(hashKey, this.mapFileNames[j]);
                        found = true;
                    }
                    ++j;
                }
                if (!found) {
                    System.err.println("ERROR: No source match for " + hashKey);
                    Utils.WaitForEnter();
                    int n = this.labels.length - 1;
                    vals[n] = vals[n] + 1;
                }
                if (debug) {
                    Utils.WaitForEnter();
                }
                this.novelPeptideSourceHash.put(peptide, vals);
            }
            ++i;
        }
    }

    private String getRawClusterKey(String filteredHashKey) {
        int fileNum;
        String[] nameBits;
        String[] bits = filteredHashKey.split("@");
        String base = Utils.GetBaseNameNoExtension(bits[0]);
        String mapFileName = null;
        int scanNum = Integer.parseInt(bits[1]);
        if (base.indexOf("Embryo") >= 0 || base.indexOf("Pericarp") >= 0) {
            nameBits = base.split("[.]");
            fileNum = Integer.parseInt(nameBits[1]);
            if (fileNum % 2 == 1) {
                scanNum += 20000;
            }
            mapFileName = String.valueOf(this.mapDirectory) + File.separator + nameBits[0] + "." + (fileNum /= 2) + ".fil.map";
        } else if (base.indexOf("Maize-GermKernel_Clustered") >= 0 && base.indexOf("Maize-GermKernel_Clustered_Clustered") < 0) {
            nameBits = base.split("[.]");
            fileNum = Integer.parseInt(nameBits[1]);
            mapFileName = String.valueOf(this.mapDirectory) + File.separator + nameBits[0] + "_Clustered" + "." + fileNum + ".fil.map";
        } else {
            mapFileName = String.valueOf(this.mapDirectory) + File.separator + base + ".map";
        }
        BufferedReader buf = Utils.openBufferedReader(mapFileName);
        String line = Utils.readNextLine(buf, mapFileName);
        String lastFileName = null;
        int lastScan = -1;
        int lastIndex = -1;
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, mapFileName);
                continue;
            }
            String[] lineBits = line.split("\t");
            if (Integer.parseInt(lineBits[0]) == scanNum) {
                Utils.closeBufferedReader(buf, mapFileName);
                return String.valueOf(Utils.GetBaseName(lineBits[1])) + "@" + lineBits[2];
            }
            lastIndex = Integer.parseInt(lineBits[0]);
            lastFileName = lineBits[1];
            lastScan = Integer.parseInt(lineBits[2]);
            line = Utils.readNextLine(buf, mapFileName);
        }
        Utils.closeBufferedReader(buf, mapFileName);
        if (lastScan + (scanNum - lastIndex) < 40000) {
            return String.valueOf(Utils.GetBaseName(lastFileName)) + "@" + (lastScan + (scanNum - lastIndex));
        }
        ErrorThrower.ThrowErrorCustum(100, "Unable to find filter map file for " + filteredHashKey);
        return null;
    }

    private int getSpecCount(String hashKey, String mapFileName) {
        hashKey = this.getRawClusterKey(hashKey);
        String[] bits = hashKey.split("@");
        BufferedReader buf = Utils.openBufferedReader(mapFileName);
        String line = Utils.readNextLine(buf, mapFileName);
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, mapFileName);
                continue;
            }
            int specCount = 0;
            String[] lineBits = line.split("\t");
            if (lineBits[1].equals(bits[1])) {
                int runIndex = Utils.findMatchingPrefix(Utils.GetBaseName(bits[0]), specFileNames);
                if (runIndex < 0) {
                    ErrorThrower.ThrowWarningCustom(101, "Cannot find matching prefix for '" + bits[0] + "'");
                    specCount += Integer.parseInt(lineBits[3]);
                } else {
                    double scaleFactor = 3.0E7 / scaleFactors[runIndex];
                    specCount = (int)((double)specCount + scaleFactor * (double)Integer.parseInt(lineBits[3]));
                }
                Utils.closeBufferedReader(buf, mapFileName);
                return specCount;
            }
            line = Utils.readNextLine(buf, mapFileName);
        }
        Utils.closeBufferedReader(buf, mapFileName);
        return 0;
    }

    private void loadEventPeptides() {
        this.novelPeptideSourceHash = new Hashtable();
        BufferedReader buf = null;
        String line = null;
        try {
            buf = new BufferedReader(new FileReader(this.inputEventFileName));
            line = buf.readLine();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(5, this.inputEventFileName);
        }
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                try {
                    line = buf.readLine();
                }
                catch (IOException E) {
                    ErrorThrower.ThrowError(6, this.inputEventFileName);
                }
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits.length == 5 ? eventBits[3].split(",") : eventBits[2].split(",");
            int i = 0;
            while (i < pepList.length) {
                String[] pepInfo = pepList[i].split(":");
                String ann = Utils.GetUnModded(pepInfo[0]);
                int[] vals = new int[this.labels.length];
                vals = Utils.initializeIntArray(vals, 0);
                this.novelPeptideSourceHash.put(ann, vals);
                ++i;
            }
            try {
                line = buf.readLine();
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(6, this.inputEventFileName);
            }
        }
        try {
            buf.close();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(8, this.inputEventFileName);
        }
    }
}

