/*
 * Decompiled with CFR 0.152.
 */
package itraqUtils;

import basicUtils.Utils;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import org.systemsbiology.jrap.stax.Scan;
import org.systemsbiology.jrap.stax.ScanHeader;
import specUtils.SpectrumLoader;

public class ITRAQExtractor {
    public static final String UsageInfo = "itraqUtils.ITRAQExtractor, version 2010.6.22\nExtracts the raw reporter ion intensities of one or more predefined\nreporter ion types.\n[REQUIRED]\n -r [FILE] Inspect results file\n -w [FILE] Output File\n -d [FILE] Spectrum directory\n -m [NUM] either 4 (4-plex) or 8 (8-plex)\n -o [NUM] 0: CID precedes PQD, 1: PQD precedes CID\n";
    private int Order = -1;
    private int Plex = -1;
    private String SpectrumDir = null;
    private String InspectResults = null;
    private String OutputFile = null;
    private Hashtable IonScanHash = null;
    private Hashtable CIDScanHash = null;
    private Hashtable UsedScanHash = null;
    private boolean Debug = false;
    private double[] FourPlex = new double[]{114.1112, 115.1083, 116.1116, 117.115};
    private double[] EightPlex = new double[]{113.1079, 114.1112, 115.1083, 116.1116, 117.115, 118.112, 119.1154, 121.1221};

    public ITRAQExtractor(String InspectFile, String OutputFile, int Plex, String SpectrumDir, int Order) {
        if (!Utils.IsFile(InspectFile)) {
            System.err.println("ERROR: " + InspectFile + " is not a valid input file name");
            System.exit(-1);
        }
        this.InspectResults = InspectFile;
        this.OutputFile = OutputFile;
        if (!Utils.IsDir(SpectrumDir)) {
            System.err.println("ERROR: " + SpectrumDir + " is not a valid directory");
            System.exit(-1);
        }
        this.SpectrumDir = SpectrumDir;
        if (Plex != 4 && Plex != 8) {
            System.out.println("ERROR: " + Plex + " is not a valid plex (must be either 4 or 8)");
            System.exit(-1);
        }
        this.Plex = Plex;
        if (Order != 0 && Order != 1) {
            System.out.println("ERROR: " + Order + " is not a valid spectrum order (must be either 0 or 1)");
            System.exit(-1);
        }
        this.Order = Order;
        if (this.Debug) {
            this.PrintParams();
        }
    }

    public void PrintParams() {
        System.out.println("Input File   : " + this.InspectResults);
        System.out.println("Output File  : " + this.OutputFile);
        System.out.println("Spectrum Dir : " + this.SpectrumDir);
        System.out.println("Reporter Ions: " + this.Plex + "-plex");
        if (this.Order == 0) {
            System.out.println("Order        : CID->PQD");
        } else {
            System.out.println("Order        : PQD->CID");
        }
    }

    public void extract() {
        System.out.println("Loading used scan numbers...");
        this.PopulateUsedScanHash();
        System.out.println("Loading CID/PQD scan pairs...");
        this.LoadIonHash();
        System.out.println("Writing updated results...");
        this.WriteUpdatedFile();
    }

    private void WriteUpdatedFile() {
        OutputStreamWriter w = null;
        String Line = null;
        BufferedReader r = null;
        try {
            r = new BufferedReader(new FileReader(this.InspectResults));
            w = new FileWriter(this.OutputFile);
            Line = r.readLine();
        }
        catch (IOException E) {
            E.printStackTrace();
            System.exit(-1);
        }
        while (Line != null) {
            Hashtable Scans;
            if ((Line = Line.trim()).length() == 0) {
                try {
                    Line = r.readLine();
                }
                catch (IOException E) {
                    E.printStackTrace();
                    System.exit(-1);
                }
                continue;
            }
            if (Line.charAt(0) == '#') {
                try {
                    w.write(String.valueOf(Line) + "\tReporterScan\tScanType");
                    int i = 0;
                    while (i < this.Plex) {
                        if (this.Plex == 4) {
                            w.write("\t" + this.FourPlex[i]);
                        }
                        if (this.Plex == 8) {
                            w.write("\t" + this.EightPlex[i]);
                        }
                        ++i;
                    }
                    w.write("\n");
                    Line = r.readLine();
                }
                catch (IOException E) {
                    E.printStackTrace();
                    System.exit(-1);
                }
                continue;
            }
            String[] Bits = Line.split("\t");
            String FileName = Utils.GetBaseName(Bits[0]);
            int ScanNumber = Integer.parseInt(Bits[1]);
            if (!this.IonScanHash.containsKey(FileName)) {
                System.err.println("ERROR: We found a file with no ions '" + FileName);
                System.exit(-1);
            }
            if (!(Scans = (Hashtable)this.IonScanHash.get(FileName)).containsKey(new Integer(ScanNumber))) {
                System.err.println("ERROR: We found a scan with no ions '" + FileName + ":" + ScanNumber + "'");
                System.exit(-1);
            }
            double[] Ions = (double[])Scans.get(new Integer(ScanNumber));
            int PQDScan = ScanNumber;
            Hashtable ScanHash = (Hashtable)this.CIDScanHash.get(FileName);
            if (ScanHash.containsKey(new Integer(ScanNumber))) {
                PQDScan = (Integer)ScanHash.get(new Integer(ScanNumber));
                Line = String.valueOf(Line) + "\t" + PQDScan + "\tCID";
            } else {
                Line = String.valueOf(Line) + "\t" + PQDScan + "\tPQD";
            }
            int i = 0;
            while (i < Ions.length) {
                Line = String.valueOf(Line) + "\t" + Ions[i];
                ++i;
            }
            Line = String.valueOf(Line) + "\n";
            try {
                w.write(Line);
                Line = r.readLine();
            }
            catch (IOException E) {
                E.printStackTrace();
                System.exit(-1);
            }
        }
        try {
            r.close();
            w.close();
        }
        catch (IOException E) {
            E.printStackTrace();
            System.exit(-1);
        }
    }

    private void PopulateUsedScanHash() {
        this.UsedScanHash = new Hashtable();
        String Line = null;
        BufferedReader r = null;
        try {
            r = new BufferedReader(new FileReader(this.InspectResults));
            Line = r.readLine();
        }
        catch (IOException E) {
            E.printStackTrace();
            System.exit(-1);
        }
        while (Line != null) {
            if ((Line = Line.trim()).length() == 0 || Line.charAt(0) == '#') {
                try {
                    Line = r.readLine();
                }
                catch (IOException E) {
                    E.printStackTrace();
                    System.exit(-1);
                }
                continue;
            }
            String[] Bits = Line.split("\t");
            String FileName = Utils.GetBaseName(Bits[0]);
            int ScanNumber = Integer.parseInt(Bits[1]);
            ArrayList ScanList = null;
            ScanList = this.UsedScanHash.containsKey(FileName) ? (ArrayList)this.UsedScanHash.get(FileName) : new ArrayList();
            ScanList = Utils.InsertInAscendingOrder(ScanList, ScanNumber);
            this.UsedScanHash.put(FileName, ScanList);
            try {
                Line = r.readLine();
            }
            catch (IOException E) {
                E.printStackTrace();
                System.exit(-1);
            }
        }
    }

    private void LoadIonHash() {
        Scan PrevScan = null;
        ScanTypes CurrScanType = ScanTypes.CID;
        if (this.Order == 1) {
            CurrScanType = ScanTypes.PQD;
        }
        this.IonScanHash = new Hashtable();
        this.CIDScanHash = new Hashtable();
        File Dir = new File(this.SpectrumDir);
        String[] FileList = Dir.list();
        int i = 0;
        while (i < FileList.length) {
            String CurrBaseName = FileList[i];
            String CurrFullName = String.valueOf(this.SpectrumDir) + File.separator + CurrBaseName;
            SpectrumLoader l = new SpectrumLoader(CurrFullName);
            if (!this.UsedScanHash.containsKey(CurrBaseName)) {
                if (this.Debug) {
                    System.out.println("Skipping spectrum file " + CurrBaseName);
                }
            } else {
                Hashtable<Integer, double[]> IonList = new Hashtable<Integer, double[]>();
                ArrayList ScanList = (ArrayList)this.UsedScanHash.get(CurrBaseName);
                Hashtable<Integer, Integer> ScanPairList = new Hashtable<Integer, Integer>();
                try {
                    l.openSequential();
                }
                catch (Exception E) {
                    E.printStackTrace();
                    System.exit(-1);
                }
                Scan currScan = l.getNextScan();
                while (currScan != null) {
                    if (currScan.header.getMsLevel() == 2) {
                        double[] Ions;
                        int ScanNumber = currScan.header.getNum();
                        if (this.Order == 0) {
                            if (CurrScanType == ScanTypes.CID) {
                                PrevScan = currScan;
                            } else {
                                Ions = null;
                                if (Utils.IntegerArrayListContains(ScanList, ScanNumber)) {
                                    Ions = this.GetReporterIons(currScan);
                                    if (this.Debug) {
                                        System.out.println("PQDScan: " + ScanNumber + ", IONS: " + Utils.DoubleArrayToString(Ions));
                                        Utils.WaitForEnter();
                                    }
                                    IonList.put(new Integer(ScanNumber), Ions);
                                    ScanPairList.put(new Integer(PrevScan.header.getNum()), new Integer(ScanNumber));
                                }
                                if (Utils.IntegerArrayListContains(ScanList, PrevScan.header.getNum())) {
                                    if (Ions == null) {
                                        Ions = this.GetReporterIons(currScan);
                                        ScanPairList.put(new Integer(PrevScan.header.getNum()), new Integer(ScanNumber));
                                    }
                                    if (this.Debug) {
                                        System.out.println("CIDScan: " + PrevScan.header.getNum() + ", IONS: " + Utils.DoubleArrayToString(Ions));
                                        Utils.WaitForEnter();
                                    }
                                    IonList.put(new Integer(PrevScan.header.getNum()), Ions);
                                }
                                PrevScan = currScan;
                            }
                        } else if (CurrScanType == ScanTypes.CID) {
                            Ions = null;
                            if (Utils.IntegerArrayListContains(ScanList, ScanNumber)) {
                                Ions = this.GetReporterIons(PrevScan);
                                IonList.put(new Integer(ScanNumber), Ions);
                                ScanPairList.put(new Integer(ScanNumber), new Integer(PrevScan.header.getNum()));
                            }
                            if (Utils.IntegerArrayListContains(ScanList, PrevScan.header.getNum())) {
                                if (Ions == null) {
                                    Ions = this.GetReporterIons(PrevScan);
                                    ScanPairList.put(new Integer(ScanNumber), new Integer(PrevScan.header.getNum()));
                                }
                                IonList.put(new Integer(PrevScan.header.getNum()), Ions);
                            }
                            PrevScan = currScan;
                        } else {
                            PrevScan = currScan;
                        }
                        CurrScanType = CurrScanType == ScanTypes.CID ? ScanTypes.PQD : ScanTypes.CID;
                    } else if (currScan.header.getMsLevel() == 1) {
                        if (this.Order == 0 && CurrScanType != ScanTypes.CID) {
                            System.err.println("WARNING: We have a lone CID scan (" + PrevScan.header.getNum() + "without a mate!!");
                            Utils.WaitForEnter();
                            CurrScanType = ScanTypes.CID;
                        } else if (this.Order == 1 && CurrScanType != ScanTypes.PQD) {
                            System.err.println("WARNING: We have a lone PQD scan (" + PrevScan.header.getNum() + "without a mate!!");
                            Utils.WaitForEnter();
                            CurrScanType = ScanTypes.PQD;
                        }
                    }
                    currScan = l.getNextScan();
                }
                this.IonScanHash.put(CurrBaseName, IonList);
                this.CIDScanHash.put(CurrBaseName, ScanPairList);
                try {
                    l.closeSequential();
                }
                catch (Exception E) {
                    E.printStackTrace();
                    System.exit(-1);
                }
            }
            ++i;
        }
    }

    private double[] GetReporterIons(Scan CurrScan) {
        double[] Ret = null;
        Ret = new double[this.Plex];
        int i = 0;
        while (i < this.Plex) {
            double[] Peak;
            if (this.Plex == 4) {
                Peak = this.GetPeak(CurrScan, this.FourPlex[i], 0.3);
                Ret[i] = Peak != null ? Peak[1] : 0.0;
            }
            if (this.Plex == 8) {
                Peak = this.GetPeak(CurrScan, this.EightPlex[i], 0.3);
                Ret[i] = Peak != null ? Peak[1] : 0.0;
            }
            ++i;
        }
        return Ret;
    }

    private double[] GetPeak(Scan CurrScan, double Mass, double Tolerance) {
        if (this.Debug) {
            System.out.println("Getting peak near " + Mass + " for " + CurrScan.header.getNum());
        }
        double[][] PeakList = CurrScan.getMassIntensityList();
        if (this.Debug) {
            System.out.println("Total peaks: " + PeakList[0].length);
        }
        if (PeakList == null) {
            System.err.println("Unable to retrieve peak list for " + CurrScan.header.getNum());
            Utils.WaitForEnter();
        }
        int ClosestIndex = -1;
        double ClosestError = Double.MAX_VALUE;
        int i = 0;
        while (i < PeakList[0].length) {
            if (Math.abs(Mass - PeakList[0][i]) < Tolerance && Math.abs(Mass - PeakList[0][i]) < ClosestError) {
                ClosestError = Math.abs(Mass - PeakList[0][i]);
                ClosestIndex = i;
            }
            if (PeakList[0][i] > Mass + Tolerance) break;
            ++i;
        }
        if (ClosestIndex == -1) {
            return null;
        }
        double[] Ret = new double[]{PeakList[0][ClosestIndex], PeakList[1][ClosestIndex]};
        if (this.Debug) {
            System.out.println("Found intensity of " + Ret[1]);
        }
        return Ret;
    }

    private void DebugPrintScan(Scan S) {
        double[][] PeakList = S.getMassIntensityList();
        int i = 0;
        while (i < PeakList[0].length) {
            System.out.println(String.valueOf(PeakList[0][i]) + " " + PeakList[1][i]);
            ++i;
        }
    }

    private void PrintScanHeader(ScanHeader S) {
        System.out.println("ScanNumber: " + S.getNum());
        System.out.println("MSLevel: " + S.getMsLevel());
        System.out.println("ParentMZ: " + S.getPrecursorMz());
        System.out.println("PeakCount: " + S.getPeaksCount());
        Utils.WaitForEnter();
    }

    private void DebugPrintCIDHash() {
        Enumeration Keys = this.IonScanHash.keys();
        while (Keys.hasMoreElements()) {
            String FileName = (String)Keys.nextElement();
            Enumeration ScanNums = ((Hashtable)this.IonScanHash.get(FileName)).keys();
            while (ScanNums.hasMoreElements()) {
                int CIDScan = (Integer)Keys.nextElement();
                int PQDScan = (Integer)this.CIDScanHash.get(new Integer(CIDScan));
                System.out.println("CID: " + CIDScan + " = PQD: " + PQDScan);
            }
        }
    }

    public static void main(String[] args) {
        String[] options = new String[]{"-r", "-d", "-m", "-o", "-w", "-v"};
        boolean[] blArray = new boolean[6];
        blArray[0] = true;
        blArray[1] = true;
        blArray[2] = true;
        blArray[3] = true;
        blArray[4] = true;
        boolean[] values = blArray;
        Hashtable CommandLineArgs = Utils.ParseCommandLine(args, options, values);
        if (!(CommandLineArgs.containsKey("-r") && CommandLineArgs.containsKey("-d") && CommandLineArgs.containsKey("-m") && CommandLineArgs.containsKey("-o") && CommandLineArgs.containsKey("-w"))) {
            System.err.println("ERROR: Must specify an input file, spectrum directory, output directory, 'plexi'hood, and order!");
            System.err.println(UsageInfo);
            System.exit(-1);
        }
        String InputFile = (String)CommandLineArgs.get("-r");
        String OutputFile = (String)CommandLineArgs.get("-w");
        String SpectrumDir = (String)CommandLineArgs.get("-d");
        int Plex = Integer.parseInt((String)CommandLineArgs.get("-m"));
        int Order = Integer.parseInt((String)CommandLineArgs.get("-o"));
        ITRAQExtractor e = new ITRAQExtractor(InputFile, OutputFile, Plex, SpectrumDir, Order);
        if (CommandLineArgs.containsKey("-v")) {
            e.Debug = true;
        }
        e.extract();
    }

    private static enum ScanTypes {
        CID,
        PQD;

    }
}

