/*
 * Decompiled with CFR 0.152.
 */
package proteogenomicUtils;

import basicUtils.InspectAnnotation;
import basicUtils.Utils;
import errorUtils.ErrorThrower;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;

public class AnalyzeNovelEventSource {
    public static String usageInfo = "proteogenomicUtils.AnalyzeNovelEventSource version 2012.04.11\nThis class is used to identify the sources of events that are discovered.\n[Required]:\n -i [FILE] Event file in table format\n OR\n -p [FILE] Identified protein file\n OR\n -r [FILE] Generic table with proteins listed in one column (comma-delimited)\n -n [DIR] Directory of Inspect results containing peptides that were grouped into events\n -t [NAME] A case-insensitive label that is contained in the spectrum files (Can specify multiple times)\n -o [FILE] Output file to write updated event information (an added column containing the sources)\n( -c [NUM] Column number containing the peptides if a generic table is input, default=0)\n";
    private String[] peptideResultFiles;
    private String[] labels;
    private String outputFileName;
    private Hashtable novelPeptideSourceHash;
    private String proteinFileName = null;
    private String inputEventFileName = null;
    private String genericInputTable = null;
    private int colNum = 0;

    public AnalyzeNovelEventSource(String inputFileName, String proteinFileName, String genericInputTable, int colNum, String peptideDir, String[] labels, String outputFileName) {
        if (inputFileName != null && (proteinFileName != null || genericInputTable != null)) {
            ErrorThrower.ThrowWarning(101, "Should only specify an event file OR a proteome file OR a generic table.  We will ignore the proteome file/generic table");
            proteinFileName = null;
            genericInputTable = null;
        }
        if (inputFileName != null && !Utils.IsFile(inputFileName)) {
            ErrorThrower.ThrowError(1, inputFileName);
        }
        this.inputEventFileName = inputFileName;
        if (proteinFileName != null && genericInputTable != null) {
            ErrorThrower.ThrowWarning(101, "Should only specify an event file OR a proteome file OR a generic table.  We will ignore the generic table");
            genericInputTable = null;
        }
        if (proteinFileName != null && !Utils.IsFile(proteinFileName)) {
            ErrorThrower.ThrowError(1, proteinFileName);
        }
        this.proteinFileName = proteinFileName;
        if (genericInputTable != null && !Utils.IsFile(genericInputTable)) {
            ErrorThrower.ThrowError(1, genericInputTable);
        }
        this.genericInputTable = genericInputTable;
        if (genericInputTable != null && colNum < 0) {
            ErrorThrower.ThrowError(4, "Column number must be non-negative: " + colNum);
        }
        this.colNum = colNum;
        if (!Utils.IsDir(peptideDir)) {
            ErrorThrower.ThrowError(1, peptideDir);
        }
        this.peptideResultFiles = Utils.ListDir(peptideDir);
        this.labels = new String[labels.length + 1];
        int i = 0;
        while (i < labels.length) {
            this.labels[i] = labels[i].toLowerCase();
            ++i;
        }
        this.labels[labels.length] = "other";
        this.outputFileName = outputFileName;
    }

    public static void main(String[] args) {
        String[] options = new String[]{"-i", "-n", "-t", "-o", "-p", "-r", "-c"};
        boolean[] values = new boolean[]{true, true, true, true, true, true, true};
        Hashtable commandLineArgs = Utils.ParseCommandLineMulti(args, options, values);
        if (!(commandLineArgs.containsKey("-n") && commandLineArgs.containsKey("-t") && commandLineArgs.containsKey("-o"))) {
            System.out.println(usageInfo);
            ErrorThrower.ThrowError(2, "Must specify an output file (-o), an Inspect result directory (-n), and at least one sample label (-t)");
        }
        if (!(commandLineArgs.containsKey("-i") || commandLineArgs.containsKey("-p") || commandLineArgs.containsKey("-r"))) {
            System.out.println(usageInfo);
            ErrorThrower.ThrowError(2, "Must specify an input event file (-i) or a selected protein file (-p) or a generic table file (-r)");
        }
        if (commandLineArgs.containsKey("-i") && commandLineArgs.containsKey("-p") || commandLineArgs.containsKey("-i") && commandLineArgs.containsKey("-r") || commandLineArgs.containsKey("-r") && commandLineArgs.containsKey("-p")) {
            System.out.println(usageInfo);
            ErrorThrower.ThrowError(2, "Must specify exactly one input event file (-i) or a selected protein file (-p) or a generic table file (-r)");
        }
        String inputFileName = null;
        if (commandLineArgs.containsKey("-i")) {
            inputFileName = (String)((ArrayList)commandLineArgs.get("-i")).get(0);
        }
        String proteinFileName = null;
        if (commandLineArgs.containsKey("-p")) {
            proteinFileName = (String)((ArrayList)commandLineArgs.get("-p")).get(0);
        }
        String genericFileName = null;
        if (commandLineArgs.containsKey("-r")) {
            genericFileName = (String)((ArrayList)commandLineArgs.get("-r")).get(0);
        }
        int colNum = 0;
        if (commandLineArgs.containsKey("-c")) {
            colNum = Integer.parseInt((String)((ArrayList)commandLineArgs.get("-c")).get(0));
        }
        String peptideDir = (String)((ArrayList)commandLineArgs.get("-n")).get(0);
        String[] labels = Utils.ConvertArraylistToStringArray((ArrayList)commandLineArgs.get("-t"));
        String outputFileName = (String)((ArrayList)commandLineArgs.get("-o")).get(0);
        AnalyzeNovelEventSource e = new AnalyzeNovelEventSource(inputFileName, proteinFileName, genericFileName, colNum, peptideDir, labels, outputFileName);
        e.runAnalysis();
    }

    private void runAnalysis() {
        if (this.inputEventFileName != null) {
            System.out.println("[1] - Load peptides of interest from events\n");
            this.loadEventPeptides();
            System.out.println(" - Total novel peptides in events: " + this.novelPeptideSourceHash.size());
        } else if (this.proteinFileName != null) {
            System.out.println("[1] - Load peptides of interest from selected proteins\n");
            this.loadProteinPeptides();
            System.out.println(" - Total peptides in selected proteins: " + this.novelPeptideSourceHash.size());
        } else {
            System.out.println("[1] - Load peptides of interest from generic table\n");
            this.loadGenericPeptides();
            System.out.println(" - Total peptides in generic table: " + this.novelPeptideSourceHash.size());
        }
        System.out.println("[2] - Identify sources of peptides\n");
        this.identifyPeptideSources();
        if (this.inputEventFileName != null) {
            System.out.println("[3] - Write out events with sources\n");
            this.writeEventsWithSources();
        } else if (this.proteinFileName != null) {
            System.out.println("[3] - Write out proteins with sources\n");
            this.writeProteinsWithSources();
        } else {
            System.out.println("[3] - Write out generic table with sources\n");
            this.writeGenericWithSources();
        }
    }

    private void writeGenericWithSources() {
        BufferedReader buf = Utils.openBufferedReader(this.genericInputTable);
        FileWriter f = Utils.openFileWriter(this.outputFileName);
        String line = Utils.readNextLine(buf, this.genericInputTable);
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, this.genericInputTable);
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits[this.colNum].split(",");
            HashSet<String> sources = new HashSet<String>();
            int i = 0;
            while (i < pepList.length) {
                String ann = Utils.GetUnModded(pepList[i]);
                int[] vals = (int[])this.novelPeptideSourceHash.get(ann);
                int j = 0;
                while (j < vals.length) {
                    if (vals[j] > 0) {
                        sources.add(this.labels[j]);
                    }
                    ++j;
                }
                ++i;
            }
            if (sources.size() == 0) {
                ErrorThrower.ThrowWarningCustom(101, "No sources found for event: " + line);
            }
            String sEls = Utils.JoinHashSet(sources, ",");
            line = String.valueOf(line) + "\t" + sEls;
            Utils.writeLine(f, this.outputFileName, String.valueOf(line) + "\n");
            line = Utils.readNextLine(buf, this.genericInputTable);
        }
        Utils.closeBufferedReader(buf, this.genericInputTable);
        Utils.closeFileWriter(f, this.outputFileName);
    }

    private void loadGenericPeptides() {
        this.novelPeptideSourceHash = new Hashtable();
        BufferedReader buf = Utils.openBufferedReader(this.genericInputTable);
        String line = Utils.readNextLine(buf, this.genericInputTable);
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, this.genericInputTable);
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits[this.colNum].split(",");
            int i = 0;
            while (i < pepList.length) {
                String ann = Utils.GetUnModded(pepList[i]);
                int[] vals = new int[this.labels.length];
                vals = Utils.initializeIntArray(vals, 0);
                this.novelPeptideSourceHash.put(ann, vals);
                ++i;
            }
            line = Utils.readNextLine(buf, this.genericInputTable);
        }
        Utils.closeBufferedReader(buf, this.genericInputTable);
    }

    private void writeProteinsWithSources() {
        BufferedReader buf = Utils.openBufferedReader(this.proteinFileName);
        FileWriter f = Utils.openFileWriter(this.outputFileName);
        String line = Utils.readNextLine(buf, this.proteinFileName);
        while (line != null) {
            int j;
            int[] vals;
            String ann;
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, this.proteinFileName);
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits[3].split(",");
            HashSet<String> sources = new HashSet<String>();
            int i = 0;
            while (i < pepList.length) {
                ann = Utils.GetUnModded(pepList[i]);
                vals = (int[])this.novelPeptideSourceHash.get(ann);
                j = 0;
                while (j < vals.length) {
                    if (vals[j] > 0) {
                        sources.add(this.labels[j]);
                    }
                    ++j;
                }
                ++i;
            }
            if (eventBits.length > 4) {
                pepList = eventBits[4].split(",");
                i = 0;
                while (i < pepList.length) {
                    ann = Utils.GetUnModded(pepList[i]);
                    vals = (int[])this.novelPeptideSourceHash.get(ann);
                    j = 0;
                    while (j < vals.length) {
                        if (vals[j] > 0) {
                            sources.add(this.labels[j]);
                        }
                        ++j;
                    }
                    ++i;
                }
            }
            if (sources.size() == 0) {
                ErrorThrower.ThrowWarningCustom(101, "No sources found for event: " + line);
            }
            String sEls = Utils.JoinHashSet(sources, ",");
            if (eventBits.length <= 4) {
                line = String.valueOf(line) + "\t";
            }
            line = String.valueOf(line) + "\t" + sEls;
            Utils.writeLine(f, this.outputFileName, String.valueOf(line) + "\n");
            line = Utils.readNextLine(buf, this.proteinFileName);
        }
        Utils.closeBufferedReader(buf, this.proteinFileName);
        Utils.closeFileWriter(f, this.outputFileName);
    }

    private void loadProteinPeptides() {
        this.novelPeptideSourceHash = new Hashtable();
        BufferedReader buf = Utils.openBufferedReader(this.proteinFileName);
        String line = Utils.readNextLine(buf, this.proteinFileName);
        while (line != null) {
            int[] vals;
            String ann;
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                line = Utils.readNextLine(buf, this.proteinFileName);
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits[3].split(",");
            int i = 0;
            while (i < pepList.length) {
                ann = Utils.GetUnModded(pepList[i]);
                vals = new int[this.labels.length];
                vals = Utils.initializeIntArray(vals, 0);
                this.novelPeptideSourceHash.put(ann, vals);
                ++i;
            }
            if (eventBits.length > 4) {
                pepList = eventBits[4].split(",");
                i = 0;
                while (i < pepList.length) {
                    ann = Utils.GetUnModded(pepList[i]);
                    vals = new int[this.labels.length];
                    vals = Utils.initializeIntArray(vals, 0);
                    this.novelPeptideSourceHash.put(ann, vals);
                    ++i;
                }
            }
            line = Utils.readNextLine(buf, this.proteinFileName);
        }
        Utils.closeBufferedReader(buf, this.proteinFileName);
    }

    private void writeEventsWithSources() {
        BufferedReader buf = null;
        FileWriter f = null;
        String line = null;
        try {
            buf = new BufferedReader(new FileReader(this.inputEventFileName));
            line = buf.readLine();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(5, this.inputEventFileName);
        }
        try {
            f = new FileWriter(this.outputFileName);
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(5, this.outputFileName);
        }
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                try {
                    line = buf.readLine();
                }
                catch (IOException E) {
                    ErrorThrower.ThrowError(6, this.inputEventFileName);
                }
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits.length == 5 ? eventBits[3].split(",") : eventBits[2].split(",");
            HashSet<String> sources = new HashSet<String>();
            int i = 0;
            while (i < pepList.length) {
                String[] pepInfo = pepList[i].split(":");
                String ann = Utils.GetUnModded(pepInfo[0]);
                int[] vals = (int[])this.novelPeptideSourceHash.get(ann);
                int j = 0;
                while (j < vals.length) {
                    if (vals[j] > 0) {
                        sources.add(this.labels[j]);
                    }
                    ++j;
                }
                ++i;
            }
            if (sources.size() == 0) {
                ErrorThrower.ThrowWarningCustom(101, "No sources found for event: " + line);
            }
            String sEls = Utils.JoinHashSet(sources, ",");
            line = String.valueOf(line) + "\t" + sEls;
            try {
                f.write(String.valueOf(line) + "\n");
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(7, this.outputFileName);
            }
            try {
                line = buf.readLine();
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(6, this.inputEventFileName);
            }
        }
        try {
            buf.close();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(8, this.inputEventFileName);
        }
        try {
            f.close();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(8, this.outputFileName);
        }
    }

    private void identifyPeptideSources() {
        boolean debug = false;
        int i = 0;
        while (i < this.peptideResultFiles.length) {
            System.out.println(" - Loading peptide sources from " + this.peptideResultFiles[i]);
            Hashtable specHash = InspectAnnotation.LoadInspectAnnotationsIntoSpec2PeptideHash(this.peptideResultFiles[i]);
            Enumeration e = specHash.keys();
            while (e.hasMoreElements()) {
                String hashKey = (String)e.nextElement();
                String peptide = (String)specHash.get(hashKey);
                if (peptide.indexOf(46) >= 0) {
                    peptide = peptide.split("[.]")[1];
                }
                if (!this.novelPeptideSourceHash.containsKey(peptide)) continue;
                if (debug) {
                    System.out.println(String.valueOf(hashKey) + " -> " + peptide + " is in an event!");
                }
                int[] vals = (int[])this.novelPeptideSourceHash.get(peptide);
                boolean found = false;
                int j = 0;
                while (j < this.labels.length - 1 && !found) {
                    if (hashKey.toLowerCase().indexOf(this.labels[j]) >= 0) {
                        if (debug) {
                            System.out.println(" source found: " + this.labels[j]);
                        }
                        int n = j;
                        vals[n] = vals[n] + 1;
                        found = true;
                    }
                    ++j;
                }
                if (!found) {
                    System.err.println("ERROR: No source match for " + hashKey);
                    Utils.WaitForEnter();
                    int n = this.labels.length - 1;
                    vals[n] = vals[n] + 1;
                }
                if (debug) {
                    Utils.WaitForEnter();
                }
                this.novelPeptideSourceHash.put(peptide, vals);
            }
            ++i;
        }
    }

    private void loadEventPeptides() {
        this.novelPeptideSourceHash = new Hashtable();
        BufferedReader buf = null;
        String line = null;
        try {
            buf = new BufferedReader(new FileReader(this.inputEventFileName));
            line = buf.readLine();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(5, this.inputEventFileName);
        }
        while (line != null) {
            if ((line = line.trim()).length() == 0 || line.charAt(0) == '#') {
                try {
                    line = buf.readLine();
                }
                catch (IOException E) {
                    ErrorThrower.ThrowError(6, this.inputEventFileName);
                }
                continue;
            }
            String[] eventBits = line.split("\t");
            String[] pepList = null;
            pepList = eventBits.length == 5 ? eventBits[3].split(",") : eventBits[2].split(",");
            int i = 0;
            while (i < pepList.length) {
                String[] pepInfo = pepList[i].split(":");
                String ann = Utils.GetUnModded(pepInfo[0]);
                int[] vals = new int[this.labels.length];
                vals = Utils.initializeIntArray(vals, 0);
                this.novelPeptideSourceHash.put(ann, vals);
                ++i;
            }
            try {
                line = buf.readLine();
            }
            catch (IOException E) {
                ErrorThrower.ThrowError(6, this.inputEventFileName);
            }
        }
        try {
            buf.close();
        }
        catch (IOException E) {
            ErrorThrower.ThrowError(8, this.inputEventFileName);
        }
    }
}

